package org.jzy3d.plot3d.rendering.view;

import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.List;

import javax.media.opengl.GL2;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.glu.GLU;

import org.jzy3d.chart.Chart;
import org.jzy3d.colors.Color;
import org.jzy3d.events.IViewIsVerticalEventListener;
import org.jzy3d.events.IViewPointChangedListener;
import org.jzy3d.events.ViewIsVerticalEvent;
import org.jzy3d.events.ViewPointChangedEvent;
import org.jzy3d.factories.JzyFactories;
import org.jzy3d.maths.BoundingBox3d;
import org.jzy3d.maths.Coord2d;
import org.jzy3d.maths.Coord3d;
import org.jzy3d.plot3d.primitives.Parallelepiped;
import org.jzy3d.plot3d.primitives.axes.AxeBox;
import org.jzy3d.plot3d.primitives.axes.IAxe;
import org.jzy3d.plot3d.rendering.canvas.CanvasAWT;
import org.jzy3d.plot3d.rendering.canvas.CanvasSwing;
import org.jzy3d.plot3d.rendering.canvas.ICanvas;
import org.jzy3d.plot3d.rendering.canvas.IScreenCanvas;
import org.jzy3d.plot3d.rendering.canvas.Quality;
import org.jzy3d.plot3d.rendering.lights.LightSet;
import org.jzy3d.plot3d.rendering.scene.Scene;
import org.jzy3d.plot3d.rendering.tooltips.ITooltipRenderer;
import org.jzy3d.plot3d.rendering.tooltips.Tooltip;
import org.jzy3d.plot3d.rendering.view.modes.CameraMode;
import org.jzy3d.plot3d.rendering.view.modes.ViewBoundMode;
import org.jzy3d.plot3d.rendering.view.modes.ViewPositionMode;
import org.jzy3d.plot3d.transform.Scale;
import org.jzy3d.plot3d.transform.Transform;

import com.jogamp.opengl.util.awt.Overlay;

/**
 * A {@link View} holds a {@link Scene}, a {@link LightSet}, an {@link ICanvas} to render into.
 * It is the responsability to layout a set of concrete {@link AbstractViewport}s such as the {@Camera}
 * rendering the scene or an {@link ImageViewport} for displaying an image in the same window.
 * 
 * On can control the {@link Camera} with a {@ViewController} and get notifyed by a 
 * {@link IViewPointChangedListener} that the view point has changed. The control is relative
 * to the center of the {@link Scene} and is defined using polar coordinates.
 * 
 * The {@link View} supports post rendering through the addition of {@link Renderer2d}s
 * whose implementation can define Java2d calls to render on top on OpenGL2.
 * 
 * Last, the {@link View} offers the ability to get an {@link AxeBox} for embedding
 * the {@link Scene} and getting values along axes.
 *
 * @author Martin Pernollet
 */
public class View {
    /**
     * Create a view attached to a Scene, with its own Camera and Axe.
     * The initial view point is set at View.DEFAULT_VIEW.
     * <p/>
     * The {@link Quality} allows setting the rendering capabilities that
     * are set one time by the init() method.
     */
    public View(Scene scene, ICanvas canvas, Quality quality) {
        BoundingBox3d sceneBounds = scene.getGraph().getBounds();

        this.viewpoint = DEFAULT_VIEW.clone();
        this.center = sceneBounds.getCenter();
        this.scaling = Coord3d.IDENTITY.clone();
        this.viewmode = ViewPositionMode.FREE;
        this.boundmode = ViewBoundMode.AUTO_FIT;
        this.cameraMode = CameraMode.ORTHOGONAL;

        this.axe = JzyFactories.axe.getInstance(sceneBounds, this);
        this.cam = JzyFactories.camera.getInstance(center);
        this.scene = scene;
        this.canvas = canvas;
        this.quality = quality;

        this.renderers = new ArrayList<Renderer2d>(1);
        this.tooltips = new ArrayList<ITooltipRenderer>();
        this.bgViewport = new ImageViewport();
        this.viewOnTopListeners = new ArrayList<IViewIsVerticalEventListener>();
        this.viewPointChangedListeners = new ArrayList<IViewPointChangedListener>();
        this.wasOnTopAtLastRendering = false;

        if (canvas instanceof CanvasSwing)
            this.overlay = new Overlay((CanvasSwing) canvas);
        else if (canvas instanceof CanvasAWT)
            this.overlay = new Overlay((CanvasAWT) canvas);
        //else if (canvas instanceof OffscreenCanvas)
        //    this.overlay = new Overlay(((OffscreenCanvas) canvas ).getGlpBuffer());
        
        this.glu = new GLU();
        
        current = this;
    }
    
    public void dispose() {
        axe.dispose();
        cam = null;
        renderers.clear();
        viewOnTopListeners.clear();
        scene = null;
        canvas = null;
        quality = null;
    }

    /** Current view selection into the mother Scene, and call to target canvas rendering.*/
    public void shoot() {
        canvas.forceRepaint();
    }
    
    public void project() {
    	//System.out.println("project");
        scene.getGraph().project(getCurrentGL(), glu, cam);
    }
    
    /******************************* GENERAL DISPLAY CONTROLS ***********************************/
	
	public void rotate(final Coord2d move){
		Coord3d eye = getViewPoint();			
		eye.x -= move.x;
		eye.y += move.y;
		setViewPoint(eye, true);		
		//fireControllerEvent(ControllerType.ROTATE, eye);
	}
	
	public void shift(final float factor){
		org.jzy3d.maths.Scale current  = getScale();
		org.jzy3d.maths.Scale newScale = current.add( factor * current.getRange() );
		setScale(newScale, true);
		//fireControllerEvent(ControllerType.SHIFT, newScale);
	}
	
	/** Factor > 1 zoom out.*/
	public void zoom(final float factor){
		org.jzy3d.maths.Scale current  = getScale();
		double range = current.getMax() - current.getMin();
		
		if(range<=0)
			return;

		double center = (current.getMax() + current.getMin())/2;
		double zmin   = center + (current.getMin()-center) * (factor);
		double zmax   = center + (current.getMax()-center) * (factor);

		// set min/max according to bounds
		org.jzy3d.maths.Scale scale = null;
		if(zmin<zmax)
			scale = new org.jzy3d.maths.Scale(zmin, zmax);
		else{
			if(factor<1) // forbid to have zmin = zmax if we zoom in
				scale = new org.jzy3d.maths.Scale(center, center);
		}

		if(scale!=null){
			setScale(scale, true);
			//fireControllerEvent(ControllerType.ZOOM, scale);
		}
	}	
	
	public void setScale(org.jzy3d.maths.Scale scale){
		setScale(scale, true);
	}
	
	public void setScale(org.jzy3d.maths.Scale scale, boolean notify){
		BoundingBox3d bounds = getBounds();
		bounds.setZmin((float)scale.getMin());
		bounds.setZmax((float)scale.getMax());
		setBoundManual(bounds);
		if(notify)
			shoot();
	}
	
	public org.jzy3d.maths.Scale getScale() {
		return new org.jzy3d.maths.Scale(getBounds().getZmin(), getBounds().getZmax());
	}
	
	 /** Set the surrounding AxeBox dimensions and the Camera target,
     * and the colorbar range.*/
    public void lookToBox(BoundingBox3d box) {
        center = box.getCenter();
        axe.setAxe(box);
        targetBox = box;
    }

    /** Get the {@link AxeBox}'s {@link BoundingBox3d}*/
    public BoundingBox3d getBounds() {
        return axe.getBoxBounds();
    }

    /** Set the {@link ViewPositionMode} applied to this view.*/
    public void setViewPositionMode(ViewPositionMode mode) {
        this.viewmode = mode;
    }

    /** Return the {@link ViewPositionMode} applied to this view.*/
    public ViewPositionMode getViewMode() {
        return viewmode;
    }

    /** Set the viewpoint using polar coordinates relative to the target (i.e. the center of the scene).*/
    public void setViewPoint(Coord3d polar, boolean updateView) {
        viewpoint = polar;
        viewpoint.y = viewpoint.y < -PI_div2 ? -PI_div2 : viewpoint.y;
        viewpoint.y = viewpoint.y > PI_div2 ? PI_div2 : viewpoint.y;
        if(updateView)
        	shoot();
        
        fireViewPointChangedEvent(new ViewPointChangedEvent(this, polar));
    }
        
    public void setViewPoint(Coord3d polar) {
    	setViewPoint(polar, true);
    }

    /** Get the viewpoint. */
    public Coord3d getViewPoint() {
        return viewpoint;
    }
    
    public Coord3d getLastViewScaling(){
    	return scaling;
    }
	
	/****************************** CONTROLS ANNOTATIONS & GENERAL RENDERING *******************************/
    
    public void setAxe (AxeBox ax){
        axe = ax;
        updateBounds();
    }
    
    public IAxe getAxe(){
    	return axe;
    }

    public boolean getAxeSquared() {
        return this.squared;
    }
    
    public void setAxeSquared(boolean status) {
        this.squared = status;
    }
    
    public boolean isAxeBoxDisplayed() {
		return axeBoxDisplayed;
	}

	public void setAxeBoxDisplayed(boolean axeBoxDisplayed) {
		this.axeBoxDisplayed = axeBoxDisplayed;
	}
	
	public void setBackgroundColor(Color color) {
        bgColor = color;
    }
    
    public Color getBackgroundColor() {
        return bgColor;
    }
    
    /** Set a buffered image, or null to desactivate background image */
    public void setBackgroundImage(BufferedImage i){
    	bgImg = i;
    	bgViewport.setImage( bgImg, bgImg.getWidth(), bgImg.getHeight() );
    	bgViewport.setStretchToFill(true); 
    	// when stretched, applyViewport() is cheaper to compute, and this does not change
    	// the picture rendering.
    	// bgViewport.setScreenGridDisplayed(true);
    }
    
    public BufferedImage getBackgroundImage(){
    	return bgImg;
    }
	
    public Camera getCamera(){
    	return cam;
    }
    
    /** Set the projection of this view, either Camera.ORTHOGONAL or Camera.PERSPECTIVE. */
    public void setCameraMode(CameraMode mode) {
        this.cameraMode = mode;
    }

    /** Get the projection of this view, either CameraMode.ORTHOGONAL or CameraMode.PERSPECTIVE. */
    public CameraMode getCameraMode() {
        return cameraMode;
    }
    
	public void getMaximized() {
        this.cam.getStretchToFill();
    }
    
    public void setMaximized(boolean status) {
        this.cam.setStretchToFill(true);
    }
    
	public Rectangle getSceneViewportRectangle(){
		return cam.getRectangle();
	}
	
	public void clearTooltips(){
		tooltips.clear();
	}
	
	public void setTooltip(ITooltipRenderer tooltip){
		tooltips.clear();
		tooltips.add(tooltip);
	}
	
	public void addTooltip(ITooltipRenderer tooltip){
		tooltips.add(tooltip);
	}
	
	public void setTooltips(List<ITooltipRenderer> tooltip){
		tooltips.clear();
		tooltips.addAll(tooltip);
	}
	
	public void addTooltips(List<ITooltipRenderer> tooltip){
		tooltips.addAll(tooltip);
	}
	
	public List<ITooltipRenderer> getTooltips(){
		return tooltips;
	}
	
    /*************************************************/

    public void addRenderer2d(Renderer2d renderer) {
        renderers.add(renderer);
    }

    public void removeRenderer2d(Renderer2d renderer) {
        renderers.remove(renderer);
    }

    public boolean addViewOnTopEventListener(IViewIsVerticalEventListener listener) {
        return viewOnTopListeners.add(listener);
    }

    public boolean removeViewOnTopEventListener(IViewIsVerticalEventListener listener) {
        return viewOnTopListeners.remove(listener);
    }

    protected void fireViewOnTopEvent(boolean isOnTop) {
        ViewIsVerticalEvent e = new ViewIsVerticalEvent(this);

        if (isOnTop)
            for (IViewIsVerticalEventListener listener : viewOnTopListeners)
                listener.viewVerticalReached(e);
        else
            for (IViewIsVerticalEventListener listener : viewOnTopListeners)
                listener.viewVerticalLeft(e);
    }
    
    public boolean addViewPointChangedListener(IViewPointChangedListener listener) {
        return viewPointChangedListeners.add(listener);
    }

    public boolean removeViewPointChangedListener(IViewPointChangedListener listener) {
        return viewPointChangedListeners.remove(listener);
    }
    
    protected void fireViewPointChangedEvent(ViewPointChangedEvent e){
    	for(IViewPointChangedListener vp: viewPointChangedListeners)
    		vp.viewPointChanged(e);
    }

    /*******************************************************************/

    /** Select between an automatic bounding (that allows fitting the entire scene graph), or a custom bounding.*/
    public void setBoundMode(ViewBoundMode mode) {
        boundmode = mode;
        updateBounds();
    }

    /** Set the bounds of the view according to the current mode, and orders a shoot(). */
    public void updateBounds() {
        if (boundmode == ViewBoundMode.AUTO_FIT)
            lookToBox(scene.getGraph().getBounds()); // set axe and camera
        else if (boundmode == ViewBoundMode.MANUAL)
            lookToBox(viewbounds); // set axe and camera
        else
            assert false : "unknown bounding mode";

        shoot();
    }

    /** Set a manual bounding box and switch the bounding mode to manual.*/
    public void setBoundManual(BoundingBox3d bounds) {
        viewbounds = bounds;
        boundmode = ViewBoundMode.MANUAL;
        lookToBox(viewbounds);
    }

    /**
     * Return a 3d scaling factor that allows scaling the scene into a square box, according to the
     * current ViewBoundMode.
     * <p/>
     * If the scene bounds are Infinite, NaN or zero, for a given dimension, the scaler will be
     * set to 1 on the given dimension.
     *
     * @return a scaling factor for each dimension.
     */
    protected Coord3d squarify() {
        // Get the view bounds
        BoundingBox3d bounds;
        if (boundmode == ViewBoundMode.AUTO_FIT)
            bounds = scene.getGraph().getBounds();
        else if (boundmode == ViewBoundMode.MANUAL)
            bounds = viewbounds;
        else {
            assert false : "unknown bounding mode";
            bounds = new BoundingBox3d(); // just for compiling
        }

        // Compute factors
        float xLen = bounds.getXmax() - bounds.getXmin();
        float yLen = bounds.getYmax() - bounds.getYmin();
        float zLen = bounds.getZmax() - bounds.getZmin();
        float lmax = Math.max(Math.max(xLen, yLen), zLen);

        if (Float.isInfinite(xLen) || Float.isNaN(xLen) || xLen == 0)
            xLen = 1;//throw new ArithmeticException("x scale is infinite, nan or 0");
        if (Float.isInfinite(yLen) || Float.isNaN(yLen) || yLen == 0)
            yLen = 1;//throw new ArithmeticException("y scale is infinite, nan or 0");
        if (Float.isInfinite(zLen) || Float.isNaN(zLen) || zLen == 0)
            zLen = 1;//throw new ArithmeticException("z scale is infinite, nan or 0");
        if (Float.isInfinite(lmax) || Float.isNaN(lmax) || lmax == 0)
            lmax = 1;//throw new ArithmeticException("z scale is infinite, nan or 0");

        // Return a scaler
        return new Coord3d(lmax / xLen, lmax / yLen, lmax / zLen);
    }

    /********************************* GL2 **********************************/

    public GL2 getCurrentGL(){
    	if (canvas instanceof GLAutoDrawable) {
            GLAutoDrawable c = ((GLAutoDrawable) canvas);
            c.getContext().makeCurrent(); // TODO: should really do that??
            return c.getGL().getGL2();
        }
    	throw new RuntimeException("Failed to retrieve GL2 context");
    }
    
    /**
     * The init function specifies general GL settings that impact
     * the rendering quality and performance (computation speed).
     * <p/>
     * The rendering settings are set by the Quality object given
     * in the constructor parameter.
     */
    protected void init(GL2 gl) {
        // Activate Depth buffer
        if (quality.isDepthActivated()) {        	
            gl.glEnable(GL2.GL_DEPTH_TEST); 
            gl.glDepthFunc(GL2.GL_LEQUAL);
        } else
            gl.glDisable(GL2.GL_DEPTH_TEST);

        // Blending
        gl.glBlendFunc(GL2.GL_SRC_ALPHA, GL2.GL_ONE_MINUS_SRC_ALPHA);
        // on/off is handled by each viewport (camera or image)
        
        // Activate tranparency
        if (quality.isAlphaActivated()) {
            gl.glEnable(GL2.GL_ALPHA_TEST);
        	if(quality.isDisableDepthBufferWhenAlpha())
        		gl.glDisable(GL2.GL_DEPTH_TEST); // gl.glDepthFunc(GL2.GL_ALWAYS); // seams better for transparent polygons since they're sorted
        	//gl.glAlphaFunc(GL2.GL_EQUAL,1.0f);
        } else {
        	gl.glDisable(GL2.GL_ALPHA_TEST);
        }

        // Make smooth colors for polygons (interpolate color between points)
        if (quality.isSmoothColor())
            gl.glShadeModel(GL2.GL_SMOOTH);
        else
            gl.glShadeModel(GL2.GL_FLAT);

        // Make smoothing setting
        if (quality.isSmoothPolygon()) {
            gl.glEnable(GL2.GL_POLYGON_SMOOTH);
            gl.glHint(GL2.GL_POLYGON_SMOOTH_HINT, GL2.GL_FASTEST);
        } else
            gl.glDisable(GL2.GL_POLYGON_SMOOTH);

        if (quality.isSmoothLine()) {
            gl.glEnable(GL2.GL_LINE_SMOOTH);
            gl.glHint(GL2.GL_LINE_SMOOTH_HINT, GL2.GL_FASTEST);
        } else
            gl.glDisable(GL2.GL_LINE_SMOOTH);
        
        if (quality.isSmoothPoint()) {
            gl.glEnable(GL2.GL_POINT_SMOOTH);
            gl.glHint(GL2.GL_POINT_SMOOTH_HINT, GL2.GL_FASTEST);
            //gl.glDisable(GL2.GL_BLEND);
            //gl.glHint(GL2.GL_POINT_SMOOTH_HINT, GL2.GL_NICEST);
        } else
            gl.glDisable(GL2.GL_POINT_SMOOTH);
        
        
        // Init light
        scene.getLightSet().init(gl);
        scene.getLightSet().enableLightIfThereAreLights(gl);
    }

    /** Clear the color and depth buffer. */
    protected void clear(GL2 gl) {
        gl.glClearColor(bgColor.r, bgColor.g, bgColor.b, bgColor.a); // clear with background
        gl.glClearDepth(1);
        gl.glClear(GL2.GL_COLOR_BUFFER_BIT | GL2.GL_DEPTH_BUFFER_BIT); 
    }
    
    /**********/

    protected synchronized void render(GL2 gl, GLU glu) {
    	renderBackground(gl, glu, 0f, 1f);
        renderScene(gl, glu);
        renderOverlay(gl);
        if( dimensionDirty )
        	dimensionDirty = false;
    }
    
    protected void renderBackground(GL2 gl, GLU glu, float left, float right) {
    	if( bgImg != null ){
    		bgViewport.setViewPort(canvas.getRendererWidth(), canvas.getRendererHeight(), left, right);
    		bgViewport.render(gl, glu);
    	}
    }

    protected void renderBackground(GL2 gl, GLU glu, ViewPort viewport) {
    	if( bgImg != null ){
    		bgViewport.setViewPort(viewport);
    		bgViewport.render(gl, glu);
    	}
    }
    
    protected void renderScene(GL2 gl, GLU glu) {
    	renderScene(gl, glu, new ViewPort(canvas.getRendererWidth(), canvas.getRendererHeight()));
    }
    
    protected void renderScene(GL2 gl, GLU glu, float left, float right) {
    	renderScene(gl, glu, ViewPort.slice(canvas.getRendererWidth(), canvas.getRendererHeight(), left, right));
    }
    
    protected void renderScene(GL2 gl, GLU glu, ViewPort viewport) {
    	if(quality.isAlphaActivated())
            gl.glEnable(GL2.GL_BLEND);
    	else
            gl.glDisable(GL2.GL_BLEND);

    	// -- Scale the scene's view -------------------
        if (squared) // force square scale
            scaling = squarify();
        else
            scaling = Coord3d.IDENTITY.clone();  
        
        // -- Compute the bounds for computing cam distance, clipping planes, etc
        if(targetBox==null)
        	targetBox = new BoundingBox3d(0,1,0,1,0,1);
        BoundingBox3d boundsScaled = new BoundingBox3d();
        boundsScaled.add(targetBox.scale(scaling));
        if (MAINTAIN_ALL_OBJECTS_IN_VIEW)
            boundsScaled.add(scene.getGraph().getBounds().scale(scaling));
        float sceneRadiusScaled = (float) boundsScaled.getRadius();

        // -- Camera settings --------------------------
        Coord3d target = center.mul(scaling);

        Coord3d eye;
        viewpoint.z = sceneRadiusScaled * 2; // maintain a reasonnable distance to the scene for viewing it.

        if (viewmode == ViewPositionMode.FREE) {
            eye = viewpoint.cartesian().add(target);
        } else if (viewmode == ViewPositionMode.TOP) {
            eye = viewpoint;
            eye.x = -(float) Math.PI / 2; // on x
            eye.y = (float) Math.PI / 2; // on top
            eye = eye.cartesian().add(target);
        } else if (viewmode == ViewPositionMode.PROFILE) {
            eye = viewpoint;
            eye.y = 0;
            eye = eye.cartesian().add(target);
        } else
            throw new RuntimeException("Unsupported ViewMode: " + viewmode);

        Coord3d up;
        if (Math.abs(viewpoint.y) == (float) Math.PI / 2) {
        	// handle up vector
            Coord2d direction = new Coord2d(viewpoint.x, viewpoint.z).cartesian(); 
            if (viewpoint.y > 0) // on top
                up = new Coord3d(-direction.x, -direction.y, 0);
            else
                up = new Coord3d(direction.x, direction.y, 0);

            // handle "on-top" events
            if (!wasOnTopAtLastRendering) {
                wasOnTopAtLastRendering = true;
                fireViewOnTopEvent(true);
            }
        } else {
            // handle up vector
            up = new Coord3d(0, 0, 1);

            // handle "on-top" events
            if (wasOnTopAtLastRendering) {
                wasOnTopAtLastRendering = false;
                fireViewOnTopEvent(false);
            }
        }

        // -- Apply camera settings ------------------------
        cam.setTarget(target);
        cam.setUp(up);
        cam.setEye(eye);

        // Set rendering volume
        if (viewmode == ViewPositionMode.TOP) {
            cam.setRenderingSphereRadius(Math.max(boundsScaled.getXmax() - boundsScaled.getXmin(), boundsScaled.getYmax() - boundsScaled.getYmin()) / 2);
            correctCameraPositionForIncludingTextLabels(gl, glu, viewport); // quite experimental!
        } else{
            cam.setRenderingSphereRadius(sceneRadiusScaled);
        }

        // Setup camera (i.e. projection matrix)
        //cam.setViewPort(canvas.getRendererWidth(), canvas.getRendererHeight(), left, right);
        cam.setViewPort(viewport);
        cam.shoot(gl, glu, cameraMode);
        
        
        // -- Render elements -----------------
        if (axeBoxDisplayed) {
            gl.glMatrixMode(GL2.GL_MODELVIEW);
            scene.getLightSet().disable(gl);
            
        	axe.setScale(scaling);
            axe.draw(gl, glu, cam);
            if (DISPLAY_AXE_WHOLE_BOUNDS) { // for debug
            	AxeBox abox = (AxeBox)axe;
                BoundingBox3d box = abox.getWholeBounds();
                Parallelepiped p = new Parallelepiped(box);
                p.setFaceDisplayed(false);
                p.setWireframeColor(Color.MAGENTA);
                p.setWireframeDisplayed(true);
                p.draw(gl, glu, cam);
            }
            
            scene.getLightSet().enableLightIfThereAreLights(gl);
        }

        Transform transform = new Transform(new Scale(scaling));
        scene.getLightSet().apply(gl, scaling);
        //gl.glEnable(GL2.GL_LIGHTING);
        //gl.glEnable(GL2.GL_LIGHT0);
        //gl.glDisable(GL2.GL_LIGHTING);
        scene.getGraph().setTransform(transform);
        scene.getGraph().draw(gl, glu, cam);
    }
    
    protected void renderOverlay(GL2 gl) {
    	renderOverlay(gl, new ViewPort(0, 0, canvas.getRendererWidth(), canvas.getRendererHeight()));
    }

    /** Renders all provided {@link Tooltip}s and {@link Renderer2d}s on top of the scene. 
     * 
     * Due to the behaviour of the {@link Overlay} implementation, Java2d geometries must be drawn relative
     * to the {@link Chart}'s {@link IScreenCanvas}, BUT will then be stretched to fit in the {@link Camera}'s
     * viewport. This bug is very important to consider, since the Camera's viewport may not occupy the full
	 * {@link IScreenCanvas}. Indeed, when View is not maximized (like the default behaviour), the viewport remains
	 * square and centered in the canvas, meaning the Overlay won't cover the full canvas area.
     * 
     * In other words, the following piece of code draws a border around the {@link View}, and not around the
	 * complete chart canvas, although queried to occupy chart canvas dimensions:
	 * 
	 * g2d.drawRect(1, 1, chart.getCanvas().getRendererWidth()-2, chart.getCanvas().getRendererHeight()-2);
	 * 
     * {@link renderOverlay()} must be called while the OpenGL2 context for the drawable is current, 
     * and after the OpenGL2 scene has been rendered.*/
    protected void renderOverlay(GL2 gl, ViewPort viewport) {
		gl.glPolygonMode(GL2.GL_FRONT_AND_BACK, GL2.GL_FILL); // TODO: don't know why needed to allow working with Overlay!!!????
		gl.glViewport(viewport.x, viewport.y, viewport.width, viewport.height);
		
		if(overlay!=null && viewport.width>0 && viewport.height>0){
	    	Graphics2D g2d = overlay.createGraphics();    	
			g2d.setBackground(bgOverlay);
	        g2d.clearRect(0, 0, canvas.getRendererWidth(), canvas.getRendererHeight());
	        
	        // Tooltips
	        for(ITooltipRenderer t: tooltips)
	        	t.render(g2d);
	        	
	        // Renderers
	        for (Renderer2d renderer : renderers)
	            renderer.paint(g2d);
	        
	        overlay.markDirty(0, 0, canvas.getRendererWidth(), canvas.getRendererHeight());
	        overlay.drawAll();
	        g2d.dispose();
		}
    }
    
    protected void correctCameraPositionForIncludingTextLabels(GL2 gl, GLU glu, ViewPort viewport) {
        cam.setViewPort(viewport);
    	cam.shoot(gl, glu, cameraMode);
        axe.draw(gl, glu, cam);
        clear(gl);

        AxeBox abox = (AxeBox)axe;
        BoundingBox3d newBounds = abox.getWholeBounds().scale(scaling);

        if (viewmode == ViewPositionMode.TOP){
        	float radius = Math.max(newBounds.getXmax() - newBounds.getXmin(), newBounds.getYmax() - newBounds.getYmin()) / 2;
            radius += (radius * STRETCH_RATIO);
        	cam.setRenderingSphereRadius(radius);
        }else
            cam.setRenderingSphereRadius((float) newBounds.getRadius());

        Coord3d target = newBounds.getCenter();
        Coord3d eye = viewpoint.cartesian().add(target);
        cam.setTarget(target);
        cam.setEye(eye);
    }
    
    /******************************************************************/

    public static View current(){
    	return current;
    }

    /******************************************************************/

    protected GLU glu;
    
    public static float STRETCH_RATIO = 0.25f;
    
    protected boolean MAINTAIN_ALL_OBJECTS_IN_VIEW = false; // force to have all object maintained in screen, meaning axebox won't always keep the same size.
    protected boolean DISPLAY_AXE_WHOLE_BOUNDS = false; // display a magenta parallelepiped (debug)

    protected boolean axeBoxDisplayed = true;
    protected boolean squared = true;

    protected Camera cam;
    protected IAxe axe;
    protected Quality quality;
    protected Overlay overlay;
    protected Scene scene;
    protected ICanvas canvas;
    
    protected Coord3d viewpoint;
    protected Coord3d center;
    protected Coord3d scaling;
	protected BoundingBox3d viewbounds;
	
    protected CameraMode cameraMode;
    protected ViewPositionMode viewmode;
    protected ViewBoundMode boundmode;    

    protected ImageViewport bgViewport;
    protected BufferedImage bgImg = null;
	//protected TooltipRenderer tooltipRenderer;
	
	protected BoundingBox3d targetBox;

    protected Color bgColor = Color.BLACK;
	protected java.awt.Color bgOverlay = new java.awt.Color(0,0,0,0);
	
	protected List<ITooltipRenderer> tooltips;

    protected List<Renderer2d> renderers;
    protected List<IViewPointChangedListener> viewPointChangedListeners;
    protected List<IViewIsVerticalEventListener> viewOnTopListeners;
    protected boolean wasOnTopAtLastRendering;

    protected static final float PI_div2 = (float) Math.PI / 2;

    public static final Coord3d DEFAULT_VIEW = new Coord3d(Math.PI / 3, Math.PI / 3, 2000);

    protected boolean dimensionDirty = false; /** can be set to true by the Renderer3d so that the View knows it is rendering due to a canvas size change*/
    protected boolean viewDirty = false;
    
    protected static View current;
}

