/*
 * Decompiled with CFR 0.152.
 */
package org.irods.jargon.core.connection;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.net.Socket;
import java.net.UnknownHostException;
import java.nio.channels.ClosedChannelException;
import java.util.Arrays;
import org.irods.jargon.core.connection.ConnectionProgressStatus;
import org.irods.jargon.core.connection.ConnectionProgressStatusListener;
import org.irods.jargon.core.connection.IRODSAccount;
import org.irods.jargon.core.connection.IRODSManagedConnection;
import org.irods.jargon.core.connection.IRODSProtocolManager;
import org.irods.jargon.core.connection.IRODSSession;
import org.irods.jargon.core.connection.PipelineConfiguration;
import org.irods.jargon.core.exception.JargonException;
import org.irods.jargon.core.utils.Host;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class IRODSConnection
implements IRODSManagedConnection {
    private Logger log = LoggerFactory.getLogger(IRODSConnection.class);
    private IRODSProtocolManager irodsProtocolManager;
    private String connectionInternalIdentifier;
    private boolean connected = false;
    private Socket connection;
    private InputStream irodsInputStream;
    private OutputStream irodsOutputStream;
    private IRODSSession irodsSession = null;
    private final IRODSAccount irodsAccount;
    private final PipelineConfiguration pipelineConfiguration;
    public static final int HEADER_INT_LENGTH = 4;
    private byte[] outputBuffer = null;
    private int outputOffset = 0;

    static IRODSConnection instance(IRODSAccount irodsAccount, IRODSProtocolManager irodsConnectionManager, PipelineConfiguration pipelineConfiguration) throws JargonException {
        IRODSConnection irodsSimpleConnection = new IRODSConnection(irodsAccount, irodsConnectionManager, pipelineConfiguration);
        irodsSimpleConnection.initializeConnection(irodsAccount);
        return irodsSimpleConnection;
    }

    private void initializeConnection(IRODSAccount irodsAccount) throws JargonException {
        this.log.debug("initializing connection with account:{}", irodsAccount);
        if (irodsAccount == null) {
            this.log.error("no irods account");
            throw new JargonException("no irods account specified, cannot connect");
        }
        if (this.irodsProtocolManager == null) {
            this.log.error("null irods connection manager");
            throw new JargonException("null irods connection manager");
        }
        this.log.info("opening irods socket");
        this.connect(irodsAccount);
        StringBuilder connectionInternalIdentifierBuilder = new StringBuilder();
        connectionInternalIdentifierBuilder.append(this.getConnectionUri());
        connectionInternalIdentifierBuilder.append('/');
        connectionInternalIdentifierBuilder.append(Thread.currentThread().getName());
        connectionInternalIdentifierBuilder.append('/');
        connectionInternalIdentifierBuilder.append(System.currentTimeMillis());
        this.connectionInternalIdentifier = connectionInternalIdentifierBuilder.toString();
    }

    private IRODSConnection(IRODSAccount irodsAccount, IRODSProtocolManager irodsConnectionManager, PipelineConfiguration pipelineConfiguration) throws JargonException {
        if (irodsConnectionManager == null) {
            throw new IllegalArgumentException("null irodsConnectionManager");
        }
        if (irodsAccount == null) {
            throw new IllegalArgumentException("null irodsAccount");
        }
        if (pipelineConfiguration == null) {
            throw new IllegalArgumentException("null pipelineConfiguration");
        }
        this.irodsProtocolManager = irodsConnectionManager;
        this.irodsAccount = irodsAccount;
        this.pipelineConfiguration = pipelineConfiguration;
        this.log.info("pipeline configuration:{}", pipelineConfiguration);
        if (pipelineConfiguration.getInternalCacheBufferSize() > 0) {
            this.log.info("using internal cache buffer of size:{}", pipelineConfiguration.getInternalCacheBufferSize());
            this.outputBuffer = new byte[pipelineConfiguration.getInternalCacheBufferSize()];
        }
    }

    private void connect(IRODSAccount irodsAccount) throws JargonException {
        this.log.info("connecting socket...");
        if (this.connected) {
            this.log.warn("doing connect when already connected!, will bypass connect and proceed");
            return;
        }
        try {
            this.log.info("connecting socket to agent");
            this.connection = new Socket(irodsAccount.getHost(), irodsAccount.getPort());
            int socketTimeout = this.pipelineConfiguration.getIrodsSocketTimeout();
            if (socketTimeout > 0) {
                this.log.info("setting a connection timeout of:{} seconds", socketTimeout);
                this.connection.setSoTimeout(socketTimeout * 1000);
            }
            if (this.pipelineConfiguration.getInternalInputStreamBufferSize() <= -1) {
                this.log.info("no buffer on input stream");
                this.irodsInputStream = this.connection.getInputStream();
            } else if (this.pipelineConfiguration.getInternalInputStreamBufferSize() == 0) {
                this.log.info("default buffer on input stream");
                this.irodsInputStream = new BufferedInputStream(this.connection.getInputStream());
            } else {
                this.log.info("buffer of size:{} on input stream", this.pipelineConfiguration.getInternalInputStreamBufferSize());
                this.irodsInputStream = new BufferedInputStream(this.connection.getInputStream(), this.pipelineConfiguration.getInternalInputStreamBufferSize());
            }
            if (this.pipelineConfiguration.getInternalOutputStreamBufferSize() <= -1) {
                this.log.info("no buffer on output stream");
                this.irodsOutputStream = this.connection.getOutputStream();
            } else if (this.pipelineConfiguration.getInternalOutputStreamBufferSize() == 0) {
                this.log.info("default buffer on input stream");
                this.irodsOutputStream = new BufferedOutputStream(this.connection.getOutputStream());
            } else {
                this.log.info("buffer of size:{} on output stream", this.pipelineConfiguration.getInternalOutputStreamBufferSize());
                this.irodsOutputStream = new BufferedOutputStream(this.connection.getOutputStream(), this.pipelineConfiguration.getInternalOutputStreamBufferSize());
            }
        }
        catch (UnknownHostException e) {
            this.log.error("exception opening socket to:" + irodsAccount.getHost() + " port:" + irodsAccount.getPort(), e);
            throw new JargonException(e);
        }
        catch (IOException ioe) {
            this.log.error("io exception opening socket to:" + irodsAccount.getHost() + " port:" + irodsAccount.getPort(), ioe);
            throw new JargonException(ioe);
        }
        this.connected = true;
        this.log.info("socket opened successfully");
    }

    @Override
    public void disconnect() throws JargonException {
        if (!this.connection.isConnected()) {
            this.log.debug("not connected, just bypass");
        }
        this.log.info("disconnecting...");
        this.irodsProtocolManager.returnIRODSConnection(this);
        this.log.info("disconnected");
    }

    @Override
    public void disconnectWithIOException() {
        this.log.info("disconnecting...");
        this.irodsProtocolManager.returnConnectionWithIoException(this);
    }

    @Override
    public void shutdown() throws JargonException {
        this.log.info("shutting down connection: {}", this.connected);
        this.closeDownSocketAndEatAnyExceptions();
    }

    @Override
    public void obliterateConnectionAndDiscardErrors() {
        this.closeDownSocketAndEatAnyExceptions();
    }

    private void closeDownSocketAndEatAnyExceptions() {
        if (this.isConnected()) {
            this.log.info("closing underlying iRODS socket connections, errors will be discarded");
            try {
                this.connection.shutdownInput();
            }
            catch (Exception e) {
                // empty catch block
            }
            try {
                this.connection.shutdownOutput();
            }
            catch (Exception e) {
                // empty catch block
            }
            try {
                this.connection.close();
            }
            catch (Exception exception) {
                // empty catch block
            }
            this.connected = false;
        }
    }

    @Override
    public String getConnectionUri() throws JargonException {
        return "irodsSimpleConnection";
    }

    @Override
    public boolean isConnected() {
        return this.connected;
    }

    public IRODSProtocolManager getIRODSProtocolManager() {
        return this.irodsProtocolManager;
    }

    public String toString() {
        return this.connectionInternalIdentifier;
    }

    void send(byte[] value) throws IOException {
        try {
            if (value == null) {
                return;
            }
            if (value.length == 0) {
                return;
            }
            if (value.length + this.outputOffset >= this.pipelineConfiguration.getInternalCacheBufferSize()) {
                this.irodsOutputStream.write(this.outputBuffer, 0, this.outputOffset);
                this.irodsOutputStream.write(value);
                this.outputOffset = 0;
            } else {
                System.arraycopy(value, 0, this.outputBuffer, this.outputOffset, value.length);
                this.outputOffset += value.length;
            }
        }
        catch (IOException ioe) {
            this.disconnectWithIOException();
            throw ioe;
        }
    }

    void send(byte[] value, int offset, int length) throws IOException {
        if (value == null) {
            this.log.error("value cannot be null");
            throw new IllegalArgumentException("value cannot be null");
        }
        if (value.length == 0) {
            this.log.warn("nothing to send, ignoring...");
            return;
        }
        if (offset > value.length) {
            String err = "trying to send a byte buffer from an offset that is out of range";
            this.log.error(err);
            this.disconnectWithIOException();
            throw new IllegalArgumentException(err);
        }
        if (length <= 0) {
            String err = "send length is zero";
            this.log.error(err);
            this.disconnectWithIOException();
            throw new IllegalArgumentException(err);
        }
        byte[] temp = new byte[length];
        System.arraycopy(value, offset, temp, 0, length);
        try {
            this.send(temp);
        }
        catch (IOException ioe) {
            this.disconnectWithIOException();
            throw ioe;
        }
    }

    void send(String value) throws IOException {
        if (value == null) {
            this.log.debug("null input packing instruction, do not send");
            return;
        }
        try {
            this.send(value.getBytes(this.pipelineConfiguration.getDefaultEncoding()));
        }
        catch (IOException ioe) {
            this.disconnectWithIOException();
            throw ioe;
        }
    }

    void sendInNetworkOrder(int value) throws IOException {
        byte[] bytes = new byte[4];
        try {
            Host.copyInt(value, bytes);
            this.send(bytes);
            this.flush();
        }
        catch (IOException ioe) {
            this.disconnectWithIOException();
            throw ioe;
        }
    }

    long send(InputStream source, long length, ConnectionProgressStatusListener connectionProgressStatusListener) throws IOException {
        if (source == null) {
            String err = "value is null";
            this.log.error(err);
            throw new IllegalArgumentException(err);
        }
        try {
            int lenThisRead = 0;
            long lenOfTemp = Math.min((long)this.pipelineConfiguration.getInputToOutputCopyBufferByteSize(), length);
            long dataSent = 0L;
            byte[] temp = new byte[(int)lenOfTemp];
            while (length > 0L) {
                if ((long)temp.length > length) {
                    temp = new byte[(int)length];
                }
                if ((lenThisRead = source.read(temp)) == -1) {
                    this.log.info("done with stream");
                    break;
                }
                length -= (long)lenThisRead;
                dataSent += (long)lenThisRead;
                this.send(temp, 0, lenThisRead);
                if (connectionProgressStatusListener == null) continue;
                connectionProgressStatusListener.connectionProgressStatusCallback(ConnectionProgressStatus.instanceForSend(lenThisRead));
            }
            this.log.debug("final flush of data sent");
            this.flush();
            this.log.info("total sent:{}", dataSent);
            return dataSent;
        }
        catch (IOException ioe) {
            this.disconnectWithIOException();
            throw ioe;
        }
    }

    void flush() throws IOException {
        if (this.connection.isClosed()) {
            throw new ClosedChannelException();
        }
        try {
            this.irodsOutputStream.write(this.outputBuffer, 0, this.outputOffset);
            this.irodsOutputStream.flush();
        }
        catch (IOException ioe) {
            this.disconnectWithIOException();
            throw ioe;
        }
        byte zerByte = 0;
        Arrays.fill(this.outputBuffer, zerByte);
        this.outputOffset = 0;
    }

    byte read() throws JargonException {
        try {
            return (byte)this.irodsInputStream.read();
        }
        catch (IOException ioe) {
            this.log.error("io exception reading", ioe);
            this.disconnectWithIOException();
            throw new JargonException(ioe);
        }
    }

    int read(byte[] value) throws JargonException {
        try {
            return this.read(value, 0, value.length);
        }
        catch (IOException ioe) {
            this.log.error("io exception reading", ioe);
            this.disconnectWithIOException();
            throw new JargonException(ioe);
        }
    }

    void read(OutputStream destination, long length) throws IOException {
        this.read(destination, length, null);
    }

    public void read(OutputStream destination, long length, ConnectionProgressStatusListener intraFileStatusListener) throws IOException {
        if (destination == null) {
            String err = "destination is null";
            this.log.error(err);
            throw new IllegalArgumentException(err);
        }
        if (length == 0L) {
            String err = "read length is set to zero";
            this.log.error(err);
            throw new IllegalArgumentException(err);
        }
        BufferedOutputStream bos = new BufferedOutputStream(destination);
        try {
            byte[] temp = new byte[Math.min(this.pipelineConfiguration.getInternalCacheBufferSize(), (int)length)];
            int n = 0;
            while (length > 0L) {
                n = this.read(temp, 0, Math.min(this.pipelineConfiguration.getInternalCacheBufferSize(), (int)length));
                if (n > 0) {
                    length -= (long)n;
                    bos.write(temp, 0, n);
                    if (intraFileStatusListener == null) continue;
                    intraFileStatusListener.connectionProgressStatusCallback(ConnectionProgressStatus.instanceForSend(n));
                    continue;
                }
                length = n;
            }
            bos.flush();
        }
        catch (IOException ioe) {
            this.log.error("io exception reading", ioe);
            this.disconnectWithIOException();
            throw ioe;
        }
        finally {
            try {
                bos.close();
            }
            catch (Exception e) {}
        }
    }

    int read(byte[] value, int offset, int length) throws ClosedChannelException, InterruptedIOException, IOException {
        if (value == null) {
            String err = "no data sent";
            this.log.error(err);
            this.disconnectWithIOException();
            throw new IllegalArgumentException(err);
        }
        if (this.log.isDebugEnabled()) {
            this.log.debug("IRODSConnection.read, byte array size =  {}", value.length);
            this.log.debug("offset = {}", offset);
            this.log.debug("length = {}", length);
        }
        if (length == 0) {
            String err = "read length is set to zero";
            this.log.error(err);
            this.disconnectWithIOException();
            throw new IOException(err);
        }
        int result = 0;
        if (length + offset > value.length) {
            this.log.error("index out of bounds exception, length + offset larger then byte array");
            this.disconnectWithIOException();
            throw new IllegalArgumentException("length + offset larger than byte array");
        }
        try {
            int bytesRead;
            int read;
            for (bytesRead = 0; bytesRead < length && (read = this.irodsInputStream.read(value, offset + bytesRead, length - bytesRead)) != -1; bytesRead += read) {
            }
            result = bytesRead;
            return result;
        }
        catch (ClosedChannelException e) {
            this.log.error("exception reading from socket", e);
            this.disconnectWithIOException();
            throw e;
        }
        catch (InterruptedIOException e) {
            this.log.error("exception reading from socket", e);
            this.disconnectWithIOException();
            throw e;
        }
        catch (IOException e) {
            this.log.error("exception reading from socket", e);
            this.disconnectWithIOException();
            throw e;
        }
    }

    @Override
    public IRODSSession getIrodsSession() {
        return this.irodsSession;
    }

    @Override
    public void setIrodsSession(IRODSSession irodsSession) {
        this.irodsSession = irodsSession;
    }

    @Override
    public IRODSAccount getIrodsAccount() {
        return this.irodsAccount;
    }

    protected void finalize() throws Throwable {
        if (this.connected) {
            this.log.error("**************************************************************************************");
            this.log.error("********  WARNING: POTENTIAL CONNECTION LEAK  ******************");
            this.log.error("********  finalizer has run and found a connection left opened, please check your code to ensure that all connections are closed");
            this.log.error("********  connection is:{}, will attempt to disconnect", (Object)this.getConnectionUri());
            this.log.error("**************************************************************************************");
            this.disconnect();
        }
        super.finalize();
    }

    public IRODSProtocolManager getIrodsProtocolManager() {
        return this.irodsProtocolManager;
    }

    public void setIrodsProtocolManager(IRODSProtocolManager irodsProtocolManager) {
        this.irodsProtocolManager = irodsProtocolManager;
    }
}

