#!/bin/bash

# BioEmergences Workflow - A standalone workflow for embryo processing on personal computers 
# Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
# 
# To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
# You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.


#takes 2 arguments, the input vtk file (either regular .vtk or .vtk.gz) and the outputDir
#  returns:
#	vtk - the location of the vtk file (in case it was gunzipped)
#	centers - the centers file associated with the vtk (if it exists, otherwise this contains "")
#	baseName - the name of the file by itself
#	strippedName - the name of the file sans extension (ex: abc.vtk.gz -> abc)
#	n1,n2,n3 - the x,y,z dimensions of the vtk
#	sx,sy,sz - the spacing of the vtk along the x,y, and z axes
#	header - path to a file containing the header of the vtk
#	binaryData - the name of the binary data. The path of this data is always $tempDir/$binaryData
#	tempDir - where work is done (and later deleted)
#	output - the path to send the result of your algorithm to
function processVTK {
  local inputFile=$1
  local outputDir=$2

  if [[ $inputFile = *.vtk.gz ]]; then
    gunzip $inputFile
    inputFile=${inputFile%.*}
  fi
 
  vtk=$inputFile
  baseName=$(basename ${inputFile})
  strippedName=${baseName%.*}
  
  tempDir=$outputDir/tmp
 
  output=$outputDir/$strippedName/
  header=$tempDir/head.txt
  binaryData=bin-part_$strippedName
  local binaryPath=$tempDir/$binaryData
  
  mkdir -p $tempDir $output
 
  echo "inputFile=$inputFile"
  #read dimensions from vtk headers
  local dimData=$(grep -a DIMENSIONS $inputFile)
  local spacing=$(grep -a SPACING $inputFile)
  n1=$(echo $dimData|awk '{print $2}')
  n2=$(echo $dimData|awk '{print $3}')
  n3=$(echo $dimData|awk '{print $4}')
  sx=$(echo $spacing|awk '{print $2}')
  sy=$(echo $spacing|awk '{print $3}')
  sz=$(echo $spacing|awk '{print $4}')

  echo "DIMENSIONS $n1 $n2 $n3"
  echo "SPACING $sx $sy $sz"

  let size=n1*n2*n3 
  tail -c $size $inputFile > $binaryPath

  filesize=$(ls -nl "${inputFile}" | awk '{print $5}')
  head -c $((${filesize} - ${size})) ${inputFile} > $header
}

function getFilePattern
{
  local input="$1"
  local channel="$2"
  local extension="$3"

  if [ "${extension}" == "" ];
  then
    extension="*"
  fi

  if [ "${channel}" == "" ];
  then
    channel="ch00"
  fi

  local first="$(basename "$(ls "${input}"/*${channel}*.${extension} | head -n 1)")"
  local time="$(echo "${first}" | grep -oE '_t[0-9]+')"
  local left="${first%%${time}*}"
  local right="${first##*${time}}"
  local num=$((${#time} - 2))

  echo "${input}/${left}_t%0${num}g${right}"
}

#Check Min and max Time
function getStartTime {
  local FilePattern=$1
  local StartTime=0
  local  common=0
  while [ "$common" -eq "0" ];
  do
    local filename=$(printf ${FilePattern} ${StartTime})
    if [ -e "$filename" ]; then
      common=1
    else
      let StartTime=StartTime+1
    fi

    if [ "$StartTime" -gt "10000" ]; then
      echo "ERROR finding StartTime expression "
      exit
     fi
  done
  echo $StartTime
}

function getEndTime {
  local FilePattern=$1
  local EndTime=$2
  local common=0
  #Not a Valid File Pattern
  isFilePattern=`echo $FilePattern | grep "%"`
  if [[ "$FilePattern" != "$isFilePattern" ]]; then
    echo $EndTime
    exit
  fi

  while [ "$common" -eq "0" ];
  do
    local filename=$(printf ${FilePattern} ${EndTime})
    if [ -e "$filename" ]; then
      let EndTime=EndTime+1
    else
      let EndTime=EndTime-1
      common=1
    fi

    if [ "$EndTime" -gt "10000" ]; then
      echo "ERROR finding EndTime expression "
      exit
    fi
  done
  echo $EndTime
}


function getTrunkEMB {
  local inputDirT=$1
  local filtype=$2
  if [ "$filtype" == "" ]; then
      filtype="*"
  fi
  patt=`ls $inputDirT/*.${filtype}`
  nbline=`echo $patt | tr ' ' '\n' | wc -l`
  if [ "$nbline" -eq "0" ]; then
    echo "ERROR getTrunkEMB -> No files *.${filtype} found in $inputDirT"
    exit
  fi
  local common=0
  local pos=0
  while [ "$common" -eq "0" ];
  do
    let pos=pos+1
    trunk=""
    for f in $patt;
    do
      f=$(basename $f)
      if [ "$trunk" == "" ]; then
        trunk=${f:0:$pos}
      else
        ff=${f:0:$pos}
        if [ "$ff" != "$trunk" ]; then
          common=1
        fi
      fi
    done
  done
  let pos=pos-1
  trunk=${f:0:$pos}
  let size=${#trunk}
  #Cut until the t
  local common=0
  pos=0
  while [ "$common" -eq "0" ];
  do
   
    local cc=${trunk:(-$pos):1}
    if [ "${cc}" == "t" ]; then 
      common=1
    else
       let pos=pos+1
       if [ "${pos}" -gt "${size}" ]; then
          echo "ERROR getTrunkEMB -> No 't' character found "
          exit
        fi
    fi
  done
  let pos=size-pos
  trunk=${f:0:$pos}
  echo $trunk
}


function getEndTrunkEMB {
  local inputDirT=$1
  local filtype=$2
  local trunk=$3
  if [ "$filtype" == "" ]; then
      filtype="*"
  fi
  patt=`ls $inputDirT/*.${filtype}`
  nbline=`echo $patt | tr ' ' '\n' | wc -l`
  if [ "$nbline" -eq "0" ]; then
    echo "ERROR getEndTrunkEMB -> No files *.${filtype} found in $inputDirT"
    exit
  fi
  local common=0
  local pos=0
  while [ "$common" -eq "0" ];
  do
    let pos=pos+1
    endtrunk=""
    for f in $patt;
    do
      f=$(basename $f)
      if [ "$endtrunk" == "" ]; then
        endtrunk=${f:(-$pos)}
      else
        ff=${f:(-$pos)}
        if [ "$ff" != "$endtrunk" ]; then
          common=1
        fi
      fi
    done

    if [ "$pos" -gt "10000" ]; then
      echo "ERROR finding Pattern file expression "
      exit
    fi
  done
  let endpos=pos-1
  endtrunk=${f:(-$endpos)}
  echo $endtrunk
}


function convertCenterTXTtoVTK {
  local filename=$1

  local baseName=$(basename ${filename})
  local strippedName=${baseName%.*}
  local inputdir=$(dirname ${filename})
  local nbpoints=`cat $filename | wc -l`

  local outputfile="${inputdir}/${strippedName}.vtk"
  echo "# vtk DataFile Version 3.0"> $outputfile
  echo "vtk output">> $outputfile
  echo "ASCII">> $outputfile
  echo "DATASET POLYDATA">> $outputfile
  echo "POINTS ${nbpoints} double">> $outputfile


  IFS=$'\n'
  n=0
  for next in `cat $filename`
  do
      if [ "${n}" -eq "3" ]; then
        echo "$next " >> $outputfile
        n=0
      else
        
        echo "$next " | tr '\n' ' '>> $outputfile
        let n=n+1
      fi
  done
  rm -f $filename


}

export -f getTrunkEMB
export -f getEndTrunkEMB
export -f convertCenterTXTtoVTK
export -f getEndTime
export -f getStartTime
export -f getFilePattern
export -f processVTK
