#!/bin/bash

# SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
# Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
# Paul Bourgine paul.bourgine@polytechnique.edu
# Alessandro Sarti alessandro.sarti@ehess.fr
# Camilo Melani camilomelani@gmail.com
# Rene Doursat rene.doursat@inaf.cnrs-gif.fr
# 
# To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
# You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.




PATHDESTINATION="$(O="$(cd "$(dirname "$0")" && pwd)/$(basename "$0")"; while [ ! -e "$O/.bioemergences" -a "$O" != "/" ]; do O="$(dirname "$O")"; done; echo "$O")"
BIWF_ALGOSDIR="${PATHDESTINATION}/Algorithmes"
MPI_HOME="${PATHDESTINATION}/lib/MPICH/mpich"
MPI="${MPI_HOME}/bin/mpirun"
algoDir=${BIWF_ALGOSDIR}/Tracking/SimulatedAnnealing
YOU_CAN_CLOSE="*** You can now close the window ***";

#fill in these three variables for your dataset

#number of threads to use per axis, total number of threads will be : NumberBlockX * NumberBlockY * NumberBlockZ
#N.B.: the volume will be cut by planes orthogonal to the 3 axes and one thread will process each subvolume
#NumberBlockX -1 is the number of splitting planes orthogonal to the X axis, same for Y, Z.
if [[ -z $NumberBlockX ]]; then
	NumberBlockX=2
	NumberBlockY=2
	NumberBlockZ=2
fi

	
##### DON'T MODIFY BELOW THIS LINE ######

#the naming pattern of the center vtk files. Use %0ng to indicate the timestep id part of the file name, 
# where n is the number of digits used for the timestep identifier

echo 'progression=-1' > "${algoDir}/.run"

if [ -d "$1" ]; then
  inputDir="$1"
else
  echo "$1 is not a directory"
  echo "${YOU_CAN_CLOSE}"
  exit 1
fi

echo "Input VTK Directory : $inputDir"

if [ -d "$2" ]; then
  CenterPath="$2"
else
  echo "$2 is not a directory"
  echo "${YOU_CAN_CLOSE}"
  exit 1
fi
echo "CenterPath : $CenterPath"
if [ -n "$2" ]; then
  outputDir="$3"
  if [ "$outputDir" == "" ]; then 
    echo "You must specify an output path"
    echo "${YOU_CAN_CLOSE}"
    exit 1
  fi
else
  echo "You must specify an output path"
  echo "${YOU_CAN_CLOSE}"
  exit 1
fi
echo "outputDir : $outputDir"
mkdir -p "$outputDir"

. "${BIWF_ALGOSDIR}/HelperFunctions.sh"
. "${algoDir}/Parameters.sh"

FilePattern=$(getFilePattern "$CenterPath" ${channel} "vtk")
echo "FilePattern=$FilePattern"

StartTimeStep=$(getStartTime $FilePattern)
echo "StartTimeStep=$StartTimeStep"

EndTimeStep=$(getEndTime $FilePattern $StartTimeStep)
echo "EndTimeStep=$EndTimeStep"

Trunk=$(basename $FilePattern)
Trunk=${Trunk%%_*}


#the output file path (with file name
LineageOutput=$outputDir/${Trunk}.lng
echo "Lineage $LineageOutput"

#Jsut to get the Dimension and Spacing
FirstFile=`ls "${inputDir}"/*${channel}*.vtk | head -n 1`

if [ ! -e "${FirstFile}" ];
then
  FirstFile=`ls "${inputDir}"/*${channel}*.vtk.gz | head -n 1`

  if [ ! -e "${FirstFile}" ];
  then
    echo "No files found in ${inputDir}"
    echo "${YOU_CAN_CLOSE}"
    exit 1
  else
    dimData=$(gzip -dc "${FirstFile}" | grep -a DIMENSIONS)
    spacing=$(gzip -dc "${FirstFile}" | grep -a SPACING)
  fi
else
  dimData=$(grep -a DIMENSIONS "$FirstFile")
  spacing=$(grep -a SPACING "$FirstFile")
fi

n1=$(echo $dimData|awk '{print $2}')
n2=$(echo $dimData|awk '{print $3}')
n3=$(echo $dimData|awk '{print $4}')
sx=$(echo $spacing|awk '{print $2}')
sy=$(echo $spacing|awk '{print $3}')
sz=$(echo $spacing|awk '{print $4}')


#We have to go in this directory to connect the boost librairires
cd "$algoDir/EXE/"
exeCommand="$algoDir/EXE/mainParallelTracking"

#Generate a configuration file
configFile=config.txt
cat "$algoDir"/Parameters.sh | grep -vE '^channel=' > "$configFile"
echo "energy.Name0=ElasticEnergy" >> "$configFile"
echo "energy.Name1=SymetricDaugthersEnergy" >> "$configFile"
echo "energy.Name2=InercialEnergy" >> "$configFile"
echo "energy.Name3=SucesorEnergy" >> "$configFile"
echo "energy.Name4=PredeserorEnergy" >> "$configFile"
echo "energy.Name5=AgeEnergy" >> "$configFile"
echo "energy.Name6=AccelerationEnergy" >> "$configFile"
echo "energy.Name7=MaxSpeed" >> "$configFile"
echo "energy.Name8=DistanceEnergy" >> "$configFile"
echo "energy.Name9=NewBourneEnergy" >> "$configFile"
echo "parallel.BufferSize=10"  >> "$configFile"
echo "tracking.timeWindow=1"  >> "$configFile"

if [ ! -e "$LineageOutput" ]; then

	fileOutput=$FilePattern

	for p in 0 1 2 3 4 5
	do
	    echo "progression=$((${p} * 100 / 6))" > "${algoDir}/.run"

		if [ $p -eq 0 ];then
			PhaseName="NULL"
		fi
		if [ $p -eq 1 ];then
			PhaseName="Initialization"
		fi
		if [ $p -eq 2 ];then
			PhaseName="Minimization"
		fi
		if [ $p -eq 3 ];then
			PhaseName="AddForward"
		fi
		if [ $p -eq 4 ];then
			PhaseName="AddBackward"
		fi
		if [ $p -eq 5 ];then
			PhaseName="FixShortTimeCells"
		fi
		
		echo "********* PHASE  =$p -> $PhaseName"
		

		
		fileInput="${fileOutput}"
		fileOutput=/"${outputDir}"${Trunk}_${PhaseName}.LineageTree
		
		BIWF_EXE_INPUT="--configFile=${configFile}  --input.ProcessType=${PhaseName}   --input.time_start_number=$StartTimeStep --input.time_end_number=$EndTimeStep   --experiment.deltaT=1  --experiment.minutesBegin=400  --input.file.ImageSpacingX=${sx} --input.file.ImageSpacingY=${sy} --input.file.ImageSpacingZ=${sz}   --input.file.input_centersPatern=$FilePattern  --input.file.ImageSizeX=${n1} --input.file.ImageSizeY=${n2} --input.file.ImageSizeZ=${n3}  --parallel.NumberBlockX=${NumberBlockX} --parallel.NumberBlockY=${NumberBlockY} --parallel.NumberBlockZ=${NumberBlockZ}  --parallel.BufferSize=10  --output.file.fileNameLineageTree=${fileOutput}"
		
		if [ ${PhaseName} == NULL ] ; then
		    BIWF_EXE_INPUT="${BIWF_EXE_INPUT} --input.type=centers_file  "
		else
		   BIWF_EXE_INPUT="${BIWF_EXE_INPUT}   --input.type=lineageTree --input.file.fileNameLineageTree=${fileInput}"
		fi
		
		echo $BIWF_EXE_INPUT | tr " " "\n"
		"${MPI}" -n $(expr $NumberBlockX \* $NumberBlockY \* $NumberBlockZ) "${exeCommand}" ${BIWF_EXE_INPUT}
		
		if [ ! -e "${fileOutput}" ]; then
			echo "Algo Exit with Errors : Miss Output File ${fileOutput}"
      echo "${YOU_CAN_CLOSE}"
			exit 1
		fi
	done

  if [ -e "${algoDir}/.run" ]
  then
    rm -f "${algoDir}/.run"
  fi

	cp "$fileOutput" "${LineageOutput}"
fi

	#Export Lineage for Mov-IT
	emblineage="${outputDir}/${Trunk}.emb"

if [ ! -e "$emblineage" ]; then
 	#Cel ID, Cell Mother ID, X, Y, Z, Time, Vx, Vy, vZ, Virtual, Name, Center Number,Deformation, Distance
	echo "Export $LineageOutput to $emblineage"   
	 "${BIWF_ALGOSDIR}/Converter/CreateMOV-IT/EXE/ConvertLNGtoEMB" \
	 --inputfile "${LineageOutput}" \
	 --outputfile "$emblineage" \
	 --SpacingX ${sx} \
	 --SpacingY ${sy} \
	 --SpacingZ ${sz} \
	 --ImageSizeX ${n1} \
	 --ImageSizeY ${n2} \
	 --ImageSizeZ ${n3}  
fi

#If the centers were never exported we dot it !
if [ ! -e "${outputDir}"/Centers ]; then
  mkdir -p "${outputDir}"/Centers
  mkdir -p "${outputDir}"/IDS
  "${BIWF_ALGOSDIR}/Converter/ExportMOV-ITCenters/EXE/ExportMOVITCenters" --inputfile "${emblineage}" --outputcenters  "${outputDir}"/Centers/Centers_t%03d.txt --outputID "${outputDir}"/IDS/IDS_t%03d.txt --ImageSizeX ${n1} --ImageSizeY ${n2} --ImageSizeZ ${n3}
fi

echo "${YOU_CAN_CLOSE}"
