/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */

#ifndef i_Convert_h__
#define __Convert_h__


 #include <iostream>
 #include <sstream>
 #include <string>
 #include <stdexcept>
 
 /*!
  * @file Convert.h
  * @author Matteo Campana
  * @brief A file that contains the BadConversion class as well as a number of overloads of stringify.
  */
 
 
 /*!
  * @class BadConversion "Convert.h"
  * @brief A runtime exception that reports when a bad conversion was attempted
  */
 class BadConversion : public std::runtime_error {
 public:
   BadConversion(const std::string& s)
     : std::runtime_error(s)
     { }
 };
 
 /*!
  * @brief a function that converts a double to a string
  * @param x the double to convert
  * @return the string representation of x
  * @throws BadConversion
  */
 inline std::string stringify(double x)
 {
   std::ostringstream o;
   if (!(o << x))
     throw BadConversion("stringify(double)");
   return o.str();
 } 
 
 /*!
  * @brief a function that converts a float to a string
  * @param x the float to convert
  * @return the string representation of x
  * @throws BadConversion
  */
 inline std::string stringify(float x)
 {
   std::ostringstream o;
   if (!(o << x))
     throw BadConversion("stringify(float)");
   return o.str();
 } 

 /*!
  * @brief a function that converts an int to a string
  * @param x the int to convert
  * @return the string represenation of x
  * @throws BadConversion
  */
 inline std::string stringify(int x)
 {
   std::ostringstream o;
   if (!(o << x))
     throw BadConversion("stringify(int)");
   return o.str();
 } 
 
 /*!
  * @brief a function that converts a long to a string
  * @param x the long to convert
  * @return the string represenation of x
  * @throws BadConversion
  */
 inline std::string stringify(long x)
 {
   std::ostringstream o;
   if (!(o << x))
     throw BadConversion("stringify(int)");
   return o.str();
 } 

 /*!
  * @brief a function that converts an unsigned int to a string
  * @param x the unsigned int to convert
  * @return the string represenation of x
  * @throws BadConversion
  */
 inline std::string stringify(unsigned int x)
 {
   std::ostringstream o;
   if (!(o << x))
     throw BadConversion("stringify(unsigned int)");
   return o.str();
 }

 /*!
  * @brief a function that converts a string to a double
  * @param s the string to convert
  * @return the double representation of s
  * @throws BadConversion
  */
 inline double convertToDouble(const std::string& s)
 {
   std::istringstream i(s);
   double x;
   if (!(i >> x))
     throw BadConversion("convertToDouble(\"" + s + "\")");
   return x;
 } 

 /*!
  * @brief a function that converts a string to a float
  * @param s the string to convert
  * @return the float representation of s
  * @throws BadConversion
  */
 inline float convertToFloat(const std::string& s)
 {
   std::istringstream i(s);
   float x;
   if (!(i >> x))
     throw BadConversion("convertToFloat(\"" + s + "\")");
   return x;
 } 

 /*!
  * @brief a function that converts a string to an int
  * @param s the string to convert
  * @return the int representation of s
  * @throws BadConversion
  */
 inline int convertToInt(const std::string& s)
 {
   std::istringstream i(s);
   int x;
   if (!(i >> x))
     throw BadConversion("convertToInt(\"" + s + "\")");
   return x;
 } 

 /*!
  * @brief a function that converts a string to an unsigned int
  * @param s the string to convert
  * @return the unsigned int representation of s
  * @throws BadConversion
  */
 inline unsigned int convertToUnsignedInt(const std::string& s)
 {
   std::istringstream i(s);
   unsigned int x;
   if (!(i >> x))
     throw BadConversion("convertTounsignedInt(\"" + s + "\")");
   return x;
 } 
#endif

