/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef graphcelltype_h
#define graphcelltype_h

#include <boost/graph/use_mpi.hpp>
#include <boost/config.hpp>
#include <boost/graph/adjacency_iterator.hpp>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/adj_list_serialize.hpp>
#include <boost/dynamic_bitset.hpp>
#include <boost/pending/indirect_cmp.hpp>
//#include <boost/range/irange.hpp>
#include <boost/serialization/map.hpp>
#include <boost/serialization/set.hpp>
#include <boost/pending/property_serialize.hpp>
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include <boost/pool/pool_alloc.hpp>
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include "boost/lambda/lambda.hpp"
#include "boost/lambda/bind.hpp"
#include "boost/lambda/construct.hpp"
#include <boost/foreach.hpp>


#include <strstream>
#include <sstream>
#include <fstream>

#include "MultiTimeSetPoint.h"
#include "SetPointsWithLocator.h"
#include "EmbryoType.h"
#include "EnergyType.h"
#include "EdgePropertyType.h"




/*!
 * @author Camilo Melani
 * @brief Implement a set of Cells with relations between cells of differents times steps.
 * It could be seen as branching process, the cells have relations with cells in next times steps, they can have 1 relation or two.
 * @tparam EPtr A cell type such as CelluleType
 */
template < typename EPtr >
class GraphCellType : public EmbryoType < EPtr >
{
  public:
    /*! todo: delete */
    void test();
    typedef GraphCellType < EPtr >  Self;
    typedef EmbryoType < EPtr > Superclass;

  public:
    /*! A node in the cell graph */
    struct Node
    {
      /*! The cell ID */
      long IdDb;
      
      /* A cell */
      EPtr cell;

      Node() {};

      friend class boost::serialization::access;
      /*! serialization helper function for boost::serialization */
      template<class Archive> void serialize ( Archive & ar, const unsigned int version )
      {
        ar & cell;
      }
    };


//    typedef typename boost::adjacency_list< boost::listS, boost::listS, boost::undirectedS, Node, EdgePropertyType> Graph;
    typedef typename boost::adjacency_list< boost::vecS, boost::vecS, boost::undirectedS, Node, EdgePropertyType> Graph;

    typedef typename boost::graph_traits<Graph>::edge_descriptor Edge;
    typedef typename boost::graph_traits<Graph>::edge_iterator edge_iterator;
    typedef typename boost::graph_traits<Graph>::vertex_descriptor Vertex;
    typedef typename boost::graph_traits<Graph>::vertex_iterator vertex_iterator;
    typedef typename boost::graph_traits<Graph>::adjacency_iterator adjacency_iterator;
    typedef typename boost::graph_traits<Graph>::out_edge_iterator out_edge_iterator;
    typedef typename std::map<  EPtr, Vertex > MapCellNodeType;


//    typedef CallBack< Self, double, EPtr, EPtr > EnergyInterTimeNodeType;
    typedef CallBack< Self, double, EPtr > EnergyNodeType;



  private:
    friend class boost::serialization::access;
    BOOST_SERIALIZATION_SPLIT_MEMBER()

    template<class Archive>
    void save ( Archive & ar, const unsigned int version ) const
    {
      ar << boost::serialization::base_object<Superclass> ( *this );
      ar << graph;
      ar << multiSet;
      ar << cellsToEnumerate;
    }

    template<class Archive>
    void load ( Archive & ar, const unsigned int version )
    {
//       invoke serialization of the base class
      ar >> boost::serialization::base_object<Superclass> ( *this );
      ar >> graph;
      ar >> multiSet;
      ar >> cellsToEnumerate;

      vertex_iterator viG1, vi_endG1;
      boost::tie ( viG1, vi_endG1 ) = vertices ( this->graph );


      for ( ; viG1 != vi_endG1 ; ++viG1 )
      {
        Vertex v1 = *viG1;
        mapCellNode.insert ( std::make_pair ( this->graph[v1].cell, v1 ) );
      }
    }




  private:
    /*! A graph of cells */
    Graph graph;
    MultiTimeSetPoint< EPtr > multiSet;
    MapCellNodeType mapCellNode;
    std::set<EPtr>  cellsToEnumerate;


    std::vector< EnergyNodeType > energyNodeSet;


  private:
    void AdjacentVertexSameTime ( Vertex s, std::list< Vertex > &list );
    void AdjacentVertexSuccessor ( Vertex s, std::list< Vertex > &list ) ;
    void AdjacentVertexPredescessor ( Vertex s, std::list< Vertex > &list );

  public:
    virtual void GetPredecessor ( EPtr p_Cell, std::list< EPtr > &p_mother );
    virtual void GetSuccessor ( EPtr p_Cell, std::list< EPtr > &p_daughters ) ;
    virtual const std::set< long > &GetTimes();
  public:
    void AddCell ( EPtr p_Cell );
    void RemoveCell ( EPtr p_Cell );
    void RemoveCells ( std::set<EPtr> &cellsToDelete );
    bool AddRelationCell ( EPtr from_Cell, EPtr to_Cell );
    bool AddRelationIdDb ( long s, long v );
    bool RemoveRelationCell ( EPtr from_Cell, EPtr to_Cell );
    bool RemoveRelationIdDb ( long s, long v );

    bool ExistCell ( EPtr a_cell );
    bool ExistEdge ( EPtr from_Cell, EPtr to_Cell );
    std::set<EPtr>  GetCellsToEnumerate();
    void AddCellToEnumerate ( EPtr p_Cell );
    void GetMorphoFields ( std::set<EPtr> & cells, GraphCellType< EPtr > &newEmbryo );
    void MergeLinegeTree ( GraphCellType< EPtr > &p_lineage, BoxType3d &boxSelection, long regionNumber );
    void UpdateProperties();


    int DegreeIn ( EPtr p_Cell );
    int DegreeOut ( EPtr p_Cell );

    EPtr GetCellIdDb ( long idDb );

    bool operator== ( const GraphCellType< EPtr >& other1 ) const;
    bool TestIntegrity();


    void AsignEdgeProperty ( );
    EdgePropertyType GetEdgeProperty ( EPtr from_Cell, EPtr to_Cell );
    void SetEdgeProperty ( EPtr from_Cell, EPtr to_Cell, EdgePropertyType edgeProperty );


    EmbryoStateType < EPtr >* GetEmbryoState ( long time );

    void RegisterNodeEnergy ( std::string energyName, double energyValue );
    void RegisterNodeEnergy ( std::vector <std::string> &energy_name, std::vector <double>  &energy_value );

  public:


    void GetEnergy ( EPtr  cell, std::ostream &os );

    double GetEnergy ( const std::string &name, Vertex s );
    double GetEnergy ( Vertex s );

    double GetEnergy ( const std::set< Vertex > &vertexSet );
    double GetEnergy ( const std::set< EPtr > &cells );
    double GetEnergy ( EPtr );

    double GetEmbryoEnergy ( long time );
    double GetEmbryoEnergy ( long timeBegin, long timeEnd );


    double ElasticEnergy ( EPtr p_Cell );
    double SymetricDaugthersEnergy ( EPtr s_Cell );
    double NewBourneEnergy ( EPtr s_Cell );
    double DistanceEnergy ( EPtr s_Cell );
    double SucesorEnergy ( EPtr p_Cell );
    double PredeserorEnergy ( EPtr p_Cell );
    double AgeEnergy ( EPtr p_Cell );
    double MaxSpeed ( EPtr p_Cell );
    double AccelerationEnergy ( EPtr p_Cell );
    double BorderEnergy ( EPtr p_Cell );
    double PredictionEnergy ( EPtr p_Cell );





    void WriteNearestMesh ( std::ostream& ofs, long time );
    void SaveSQLFileAllEmbryo ( std::ostream& ofs, long id_process_tracking );
    void SaveSQLFileEmbryoState ( std::ostream& ofs, long id_process_tracking, long time );

  public:
    GraphCellType();
    ~GraphCellType();
};

template <typename EPtr>
GraphCellType< EPtr >::GraphCellType() : Superclass(), cellsToEnumerate ( std::set<EPtr>() )
{
}


template <typename EPtr>
GraphCellType< EPtr >::~GraphCellType()
{}

template <typename EPtr>
void GraphCellType< EPtr >::AddCell ( EPtr p_Cell )
{
  Vertex s;
  this->multiSet.AddCell ( p_Cell );

  s = add_vertex ( this->graph );
  this->graph[s].cell = p_Cell;

  mapCellNode.insert ( std::make_pair ( p_Cell, s ) );

//  p_Cell->SetimageProvider ( this->GetImageProvider() );
}



template <typename EPtr>
    void GraphCellType< EPtr >::RemoveCell ( EPtr p_Cell )
{
  typename MapCellNodeType::iterator toIt = this->mapCellNode.find ( p_Cell );
  if ( toIt != this->mapCellNode.end() )
  {
    Vertex s;
    s = toIt->second;

    clear_vertex ( s,  this->graph );
    remove_vertex ( s, this->graph );

//   reindex the vertex with vertex greater than s

    for ( typename MapCellNodeType::iterator it = mapCellNode.begin(); it != mapCellNode.end();++it )
    {
      if ( it->second > s )
      {
        //if the vertex list is vecS
        -- ( it->second );
      }
    }

    this->multiSet.RemoveCell ( p_Cell );
    mapCellNode.erase ( p_Cell );
    cellsToEnumerate.erase(p_Cell);
  }
}
template < typename EPtr >
    void GraphCellType< EPtr >::RemoveCells ( std::set<EPtr> &cellsToDelete )
{
  for_each(cellsToDelete.begin(),
           cellsToDelete.end(),
           boost::bind( &GraphCellType< EPtr >::RemoveCell,this,_1));
}

/*template <typename EPtr>
void GraphCellType< EPtr >::RemoveCell ( EPtr p_Cell )
{
  std::set<EPtr> cellsToDelete;
  cellsToDelete.insert ( p_Cell );
  RemoveCells ( cellsToDelete );
}



template < typename EPtr >
void GraphCellType< EPtr >::RemoveCells ( std::set<EPtr> &cellsToDelete )
{
  std::set<long> nodesToReindex;
  for ( typename std::set<EPtr>::iterator it = cellsToDelete.begin();it != cellsToDelete.end();++it )
  {
    EPtr p_Cell = *it;
    typename MapCellNodeType::iterator toIt = this->mapCellNode.find ( p_Cell );
    if ( toIt != this->mapCellNode.end() )
    {
      this->multiSet.RemoveCell ( p_Cell );
      mapCellNode.erase ( p_Cell );
      cellsToEnumerate.erase ( p_Cell );

      Vertex s;
      s = toIt->second;;
      nodesToReindex.insert ( toIt->second );

      clear_vertex ( s,  this->graph );
      remove_vertex ( s, this->graph );
    }
  }
//   reindex the vertex with vertex greater than s
  for ( typename MapCellNodeType::iterator it = mapCellNode.begin(); it != mapCellNode.end();++it )
  {
    Vertex node = it->second;
    long index = std::distance ( nodesToReindex.begin(), nodesToReindex.lower_bound ( node ) ) ;
    it->second -= index;
  }
}
*/



/**
 * @param from_Cell
 * @param to_Cell
 * @return true if the edge is added oo, false if the edge is not added.
 */
template <typename EPtr>
bool GraphCellType< EPtr >::AddRelationCell ( EPtr from_Cell, EPtr to_Cell )
{
  long MaxPredesesor = 1;
  long MaxSuccesor = 2;

  Vertex s, v;
  bool result;
  typename MapCellNodeType::iterator fromIt = this->mapCellNode.find ( from_Cell );
  typename MapCellNodeType::iterator toIt = this->mapCellNode.find ( to_Cell );

  bool NotExist_From = ( fromIt == this->mapCellNode.end() );
  bool NotExist_To = ( toIt == this->mapCellNode.end() );

  if ( NotExist_From ||
       NotExist_To )
  {
    result = false;
  }
  else
  {
    s = fromIt->second;
    v = toIt->second;

    long numPredesesor = this->GetNumberPredecessor ( to_Cell );
    long numSuccesors = this->GetNumberSuccessor ( from_Cell );


    if ( ! ( numSuccesors < MaxSuccesor ) )
    {
      result = false;
    }
    else if ( ! ( numPredesesor < MaxPredesesor ) )
    {
      result = false;
    }
    else if ( s == v )
    {
      result = false;
    }
    else if ( this->ExistEdge ( from_Cell, to_Cell ) )
    {
      result = false;
    }
    else
    {
      result = true;
    }
  }

  if ( result )
  {
    add_edge ( s, v, this->graph );
  }

  return result;
}


template <typename EPtr>
bool GraphCellType< EPtr >::AddRelationIdDb ( long s_id, long v_id )
{
  bool result = false;
  
  try
  {
    EPtr fromCell = this->multiSet.GetCellIdDb ( s_id );
    EPtr toCell = this->multiSet.GetCellIdDb ( v_id );

    if ( fromCell == NULL || toCell == NULL )
    {
      result = false;
    }
    else
    {
      result = this->AddRelationCell ( fromCell, toCell );
    }
  }
  catch(...)
  {
     std::cout << "Error Triyng to add a link" << std::endl;
  }
  return result;
}


template <typename EPtr>
bool GraphCellType< EPtr >::RemoveRelationCell ( EPtr from_Cell, EPtr to_Cell )
{
  Vertex s, v;
  bool result;
  typename MapCellNodeType::iterator fromIt = this->mapCellNode.find ( from_Cell );
  typename MapCellNodeType::iterator toIt = this->mapCellNode.find ( to_Cell );

  if ( fromIt == this->mapCellNode.end() ||
       toIt == this->mapCellNode.end() )
  {
    result = false;
  }
  else
  {
    s = fromIt->second;
    v = toIt->second;

    remove_edge ( s, v, this->graph );

    result = true;
  }
  return result;
}


template <typename EPtr>
bool GraphCellType< EPtr >::RemoveRelationIdDb ( long s, long v )
{
  bool result;
  EPtr fromCell = this->multiSet.GetCellIdDb ( s );
  EPtr toCell = this->multiSet.GetCellIdDb ( v );

  if ( fromCell == NULL || toCell == NULL )
  {
    result = false;
  }
  else
  {
    result = this->RemoveRelationCell ( fromCell, toCell );
  }
  return result;
}

template <typename EPtr>
double GraphCellType< EPtr >::GetEnergy ( const std::set < EPtr>  &cells )
{
  std::set< Vertex > vertexSet;

  for ( typename std::set< EPtr >::iterator it = cells.begin();it != cells.end(); ++it )
  {
    vertexSet.insert ( this->mapCellNode[*it] );
  }
  return ( this->GetEnergy ( vertexSet ) );
}


/**
 * Return the sumatory of energy of the incident edges
 * @param s Node
 * @return  Sumatory of the Energy
 */
template <typename EPtr>
double GraphCellType< EPtr >::GetEnergy ( const std::set<Vertex > &vertexSet )
{
  double localEnergy = 0;
  std::set< Edge > edgesSet;
  // iterate over all the cells.
  for ( typename std::set< Vertex >::iterator it = vertexSet.begin();it != vertexSet.end(); ++it )
  {
    Vertex s = *it;
    localEnergy += this->GetEnergy ( s );

  }
  return localEnergy;
}






template <typename EPtr>
double GraphCellType< EPtr >::GetEnergy ( Vertex s )
{
  double localEnergy = 0;

  typename std::vector< EnergyNodeType >::iterator it;
  for ( it = energyNodeSet.begin();it != energyNodeSet.end();++it )
  {
    double energy = 0;
    energy = it->execute ( this->graph[s].cell ) ;
    localEnergy += energy;
      if ( std::isnan ( energy ) )
    {
      std::cerr << it->GetName() << std::endl;
      it->execute ( this->graph[s].cell ) ;
    }
  }
  return ( localEnergy );
}


template <typename EPtr>
void GraphCellType< EPtr >::GetEnergy ( EPtr cell, std::ostream &os )
{
  typename std::vector< EnergyNodeType >::iterator it;
  for ( it = energyNodeSet.begin();it != energyNodeSet.end();++it )
  {
    os << it->GetName() << ": " << it->execute ( cell ) << '\n';
  }
  os << std::endl << "------------" << std::endl;
}



template <typename EPtr>
double GraphCellType< EPtr >::GetEnergy ( EPtr aCell )
{
  return ( this->GetEnergy ( this->mapCellNode[aCell] ) );
}


template <typename EPtr>
double GraphCellType< EPtr >::GetEnergy ( const std::string &name, Vertex s )
{
  double localEnergy = 0;

  typename std::vector< EnergyNodeType >::iterator it;
  for ( it = energyNodeSet.begin();it != energyNodeSet.end();++it )
  {
    if ( name == it->GetName() )
      localEnergy += it->execute ( this->graph[s].cell ) ;
  }

  return ( localEnergy );
}


template < typename EPtr >
void GraphCellType< EPtr >::AdjacentVertexSameTime ( Vertex s, std::list< Vertex > &list )
{

  EPtr s_Cell = this->graph[s].cell;

  //Add edges between neighbours nodes;
  EmbryoStateType< EPtr > *embryoState;
  embryoState = this->GetEmbryoState ( s_Cell->GetT() );

  std::list< EPtr > listCell;
  embryoState->GetNearestNCells ( s_Cell, 0,4, listCell );

  BOOST_FOREACH ( EPtr cell, listCell )
  {
    list.push_back ( this->mapCellNode[cell] );
  }
}








template < typename EPtr >
void GraphCellType< EPtr >::AdjacentVertexPredescessor ( Vertex s, std::list<Vertex> &list )
{
  EPtr bCell = this->graph[s].cell;
  EPtr aCell;


  adjacency_iterator vi, vi_end;
  for ( tie ( vi, vi_end ) = adjacent_vertices ( s, this->graph ); vi != vi_end; ++vi )
  {
    Vertex v = *vi;
    aCell = this->graph[v].cell;

    if ( aCell->GetT() < bCell->GetT() )
    {
//      Vertex b = *vi;
      list.push_back ( *vi );
    }

  }
}


template < typename EPtr >
void GraphCellType< EPtr >::AdjacentVertexSuccessor ( Vertex s, std::list<Vertex> &list )
{
  adjacency_iterator vi, vi_end;


  for ( boost::tie ( vi, vi_end ) = adjacent_vertices ( s, this->graph ); vi != vi_end; ++vi )
  {

    if ( this->graph[*vi].cell->GetT() > this->graph[s].cell->GetT() )
    {
      Vertex v = *vi;
      list.push_back ( v );
    }
  }
}

template < typename EPtr >
bool GraphCellType< EPtr >::ExistEdge ( EPtr from_Cell, EPtr to_Cell )
{
  bool result = false;


  typename MapCellNodeType::iterator fromIt = this->mapCellNode.find ( from_Cell );
  typename MapCellNodeType::iterator toIt = this->mapCellNode.find ( to_Cell );

  if ( fromIt == this->mapCellNode.end() ||
       toIt == this->mapCellNode.end() )
  {
    result = false;
  }
  else
  {
    Vertex s = fromIt->second;

    adjacency_iterator vi, vi_end;

    for ( boost::tie ( vi, vi_end ) = adjacent_vertices ( s, this->graph ); vi != vi_end; ++vi )
    {
      Vertex v = *vi;
      if ( v == s )
      {
        result = true;
        return result;
      }
    }
  }

  return result;
}

template < typename EPtr >
bool GraphCellType< EPtr >::ExistCell ( EPtr a_cell )
{
  typename MapCellNodeType::iterator toIt = this->mapCellNode.find ( a_cell );

  return ( toIt != this->mapCellNode.end() ) ;
}







template < typename EPtr >
EmbryoStateType < EPtr >* GraphCellType< EPtr >::GetEmbryoState ( long time )
{
  EmbryoStateType < EPtr >* embryoState = this->multiSet.GetEmbryoState ( time );

  return ( embryoState );

}

template < typename EPtr >
void GraphCellType< EPtr >::GetPredecessor ( EPtr p_Cell, std::list< EPtr > &p_mother )
{
  std::list< Vertex > predList;

  Vertex s = this->mapCellNode[p_Cell];

  this->AdjacentVertexPredescessor ( s, predList );

  for ( typename std::list< Vertex >::iterator itPred = predList.begin(); itPred != predList.end();++itPred )
  {
//    Vertex b = *itPred;

    p_mother.push_back ( this->graph[*itPred].cell );
  }
}

template < typename EPtr >
void GraphCellType< EPtr >::GetSuccessor ( EPtr p_Cell, std::list< EPtr > &p_daughters )
{
  std::list< Vertex > sucesors;

  Vertex s = this->mapCellNode[p_Cell];

  std::list< Vertex > listSucessorS;
  this->AdjacentVertexSuccessor ( s, listSucessorS );

  for ( typename std::list< Vertex >::iterator itSucS = listSucessorS.begin(); itSucS != listSucessorS.end();++itSucS )
  {
    EPtr a_Cell = this->graph[*itSucS].cell;
    p_daughters.push_back ( a_Cell );
  }
}


template<typename EPtr>
inline
const std::set< long > &GraphCellType< EPtr >::GetTimes()
{
  return this->multiSet.GetTimes();
}


template < typename EPtr >
void GraphCellType< EPtr >::SetEdgeProperty ( EPtr from_Cell, EPtr to_Cell, EdgePropertyType edgeProperty )
{
  Vertex s = this->mapCellNode[from_Cell];
  Vertex v = this->mapCellNode[to_Cell];

  std::pair<Edge, bool> resu;
  resu = edge ( s, v, this->graph );

  this->graph[resu.first] = edgeProperty;
}



template < typename EPtr >
void  GraphCellType< EPtr >::AsignEdgeProperty ( )
{
  std::set<EPtr> allCells = this->GetAllCells();
//  double maxEnergy = -100000;
//  for ( typename std::set<EPtr>::iterator it = allCells.begin();it != allCells.end();++it )
//  {
//    double actualEnergy = this->GetEnergy ( *it );
//    maxEnergy = ( actualEnergy > maxEnergy ) ? actualEnergy : maxEnergy;
//  }

  for ( typename std::set<EPtr>::iterator it = allCells.begin();it != allCells.end();++it )
  {
    EPtr cell = *it;
    std::list< EPtr > predList;
    this->GetPredecessor ( cell, predList );
    if ( predList.size() != 0 )
    {
      EPtr pred = * ( predList.begin() );

      EdgePropertyType property;
      property.SetProbability ( ( double ) this->GetEnergy ( pred ) /*/ maxEnergy*/ );
      this->SetEdgeProperty ( pred, cell, property );
    }
  }
}

template < typename EPtr >
EdgePropertyType GraphCellType< EPtr >::GetEdgeProperty ( EPtr from_Cell, EPtr to_Cell )
{
  Vertex s = this->mapCellNode[from_Cell];
  Vertex v = this->mapCellNode[to_Cell];

  std::pair<Edge, bool> resu;
  resu = edge ( s, v, this->graph );

  return this->graph[resu.first] ;
}


template<typename EPtr>
double GraphCellType< EPtr >::GetEmbryoEnergy ( long time )
{
  EmbryoStateType< EPtr > *embryoState;
  embryoState = this->GetEmbryoState ( time );


  std::set< EPtr > cellsSet;
  typename EmbryoStateType< EPtr >::iterator it;


  double energy = 0;
  for ( it = embryoState->begin();it != embryoState->end();++it )
  {
    cellsSet.insert ( *it );
  }

  energy = this->GetEnergy ( cellsSet );

  return energy;
}


template<typename EPtr>
double GraphCellType< EPtr >::GetEmbryoEnergy ( long timeBegin, long timeEnd )
{

  double sum = 0;
  for ( long time = timeBegin; time < timeEnd; time++ )
  {
    sum += this->GetEmbryoEnergy ( time );
  }

  return sum;


}


template<typename EPtr>
EPtr GraphCellType< EPtr >::GetCellIdDb ( long idDb )
{
  return this->multiSet.GetCellIdDb ( idDb );
}

template<typename EPtr>
void GraphCellType< EPtr >::WriteNearestMesh ( std::ostream& ofs, long time )
{
  typedef std::map<std::string, std::ostringstream*> outputType;
  outputType output;

  output["header1"] = new std::ostringstream();
  output["header2"] = new std::ostringstream();
  output["Points"] = new std::ostringstream();
  output["EnergyTotal"] = new std::ostringstream();

  for ( typename std::vector< EnergyNodeType >::iterator it2 = energyNodeSet.begin();it2 != energyNodeSet.end();++it2 )
  {
    output[it2->GetName() ] = new std::ostringstream();
  }

  EmbryoStateType< EPtr > *embryoState;
  embryoState = this->GetEmbryoState ( time );

  long count = 0;
  for ( typename EmbryoStateType< EPtr >::iterator it = embryoState->begin();it != embryoState->end();++it )
  {
    Vertex s = this->mapCellNode[*it];

    //Get energy of the edge
    *output["EnergyTotal"] << this->GetEnergy ( s )  << " ";


    for ( typename std::vector< EnergyNodeType>::iterator it3 = energyNodeSet.begin();it3 != energyNodeSet.end();++it3 )
    {

      *output[it3->GetName() ] << this->GetEnergy ( it3->GetName(), s )  << " ";
    }

    *output["Points"] << this->graph[s].cell->GetX() << " " << this->graph[s].cell->GetY() << " " << this->graph[s].cell->GetZ() << " ";

    count++;
  }
  ofs << "# vtk DataFile Version 3.0\n" ;
  ofs << "vtk output\n" ;
  ofs << "ASCII\n";
  ofs << "DATASET POLYDATA\n";
  ofs << "POINTS " << count << " float\n";
  ofs << output["Points"]->str() << "\n";

  ofs << "FIELD EnergyTotal 1 \n";
  ofs << "EnergyTotal 1 " << count << " double\n";
  ofs << output["EnergyTotal"]->str() ;

  for ( typename std::vector< EnergyNodeType >::iterator it2 = energyNodeSet.begin();it2 != energyNodeSet.end();++it2 )
  {
    ofs << "FIELD " << it2->GetName() << " 1 \n";
    ofs << "EnergyElastic 1 " << count << " double\n";
    ofs << output[it2->GetName() ]->str() ;
  }

  ofs << std::endl;
  ofs.flush();

//        for_each(output.begin(),
//                 output.end(),
//                 boost::lambda::bind( boost::lambda::delete_ptr(), boost::lambda::_1));
  for ( outputType::iterator it = output.begin(); it != output.end(); it++ )
  {
    delete it->second;
  }
}


// template<typename EPtr>
//     void GraphCellType< EPtr >::WriteNearestMesh ( std::ostream& ofs, long time )
// {
//   EmbryoStateType< EPtr > *embryoState;
//   embryoState = this->GetEmbryoState ( time );
//
//   std::set< Edge > edgesSet;
//
//   for ( typename EmbryoStateType< EPtr >::iterator it = embryoState->begin();it != embryoState->end();++it )
//   {
//     Vertex s = this->mapCellNode[*it];
//
//     std::list< Vertex > listAdjacent;
//     this->AdjacentVertexSameTime ( s, listAdjacent );
//     for ( typename std::list< Vertex >::iterator it = listAdjacent.begin(); it != listAdjacent.end();++it )
//     {
//       std::pair< Edge , bool > edgeD = edge ( s, *it, this->graph );
//       edgesSet.insert ( edgeD.first );
//     }
//   }
//
//   // iterate over all the edges.
//   typedef std::map<std::string, std::ostringstream*> outputType;
//   outputType output;
//
//   output["header1"] = new std::ostringstream();
//   output["header2"] = new std::ostringstream();
//   output["Points"] = new std::ostringstream();
//   output["Vectors"] = new std::ostringstream();
//   output["EnergyTotal"] = new std::ostringstream();
//
//
//
//   for ( typename std::vector< EnergyNodeType >::iterator it2 = energyNodeSet.begin();it2 != energyNodeSet.end();++it2 )
//   {
//     output[it2->GetName() ] = new std::ostringstream();
//   }
//
//
//   long count = 0;
//   for ( typename std::set< Edge >::iterator edgesIt = edgesSet.begin();edgesIt != edgesSet.end(); ++edgesIt )
//   {
//     Vertex s = source ( *edgesIt, this->graph );
//     Vertex v = target ( *edgesIt, this->graph );
//
//     //Get energy of the edge
//     *output["EnergyTotal"] << this->GetEnergy ( s ) +  this->GetEnergy ( v ) << " ";
//
//
//     for ( typename std::vector< EnergyNodeType>::iterator it3 = energyNodeSet.begin();it3 != energyNodeSet.end();++it3 )
//     {
//
//       *output[it3->GetName() ] << this->GetEnergy ( it3->GetName(), s ) + this->GetEnergy ( it3->GetName(), v ) << " ";
//     }
//
//
//     //Get the distance between s and v
//     *output["Points"] << this->graph[s].cell->GetX() << " " << this->graph[s].cell->GetY() << " " << this->graph[s].cell->GetZ() << " ";
//
//     double A[3];
//     A[0] = this->graph[v].cell->GetX() - this->graph[s].cell->GetX() ;
//     A[1] = this->graph[v].cell->GetY() - this->graph[s].cell->GetY() ;
//     A[2] = this->graph[v].cell->GetZ() - this->graph[s].cell->GetZ() ;
//
//     *output["Vectors"] << A[0] << " " << A[1] << " " << A[2] << " ";
//
//     count++;
//   }
//   ofs << "# vtk DataFile Version 3.0\n" ;
//   ofs << "vtk output\n" ;
//   ofs << "ASCII\n";
//   ofs << "DATASET POLYDATA\n";
//   ofs << "POINTS " << count << " float\n";
//   ofs << output["Points"]->str() << "\n";
//   ofs << "POINT_DATA " << count << "\n";
//   ofs << "VECTORS Estimated%20Direction double\n";
//   ofs << output["Vectors"]->str() << "\n";
//
//   ofs << "FIELD EnergyTotal 1 \n";
//   ofs << "EnergyTotal 1 " << count << " double\n";
//   ofs << output["EnergyTotal"]->str() ;
//
//
//
//   for ( typename std::vector< EnergyNodeType >::iterator it2 = energyNodeSet.begin();it2 != energyNodeSet.end();++it2 )
//   {
//     ofs << "FIELD " << it2->GetName() << " 1 \n";
//     ofs << "EnergyElastic 1 " << count << " double\n";
//     ofs << output[it2->GetName() ]->str() ;
//
//   }
//
//   ofs << std::endl;
//   ofs.flush();
//
//
// //        for_each(output.begin(),
// //                 output.end(),
// //                 boost::lambda::bind( boost::lambda::delete_ptr(), boost::lambda::_1));
//
//
//   for ( outputType::iterator it = output.begin(); it != output.end(); it++ )
//   {
//     delete it->second;
//   }
// }



template<typename EPtr>
void GraphCellType< EPtr >::RegisterNodeEnergy ( std::vector <std::string> &energy_name, std::vector <double>  &energy_value )
{
  for (unsigned int i = 0;i < energy_name.size();++i )
  {
    this->RegisterNodeEnergy ( energy_name[i], energy_value[i] );
  }
}




template<typename EPtr>
void GraphCellType< EPtr >::RegisterNodeEnergy ( std::string energyName, double energyValue )
{
  double ( Self::*method ) ( EPtr ) = NULL;
  if ( energyName == "ElasticEnergy" )
    method = &Self::ElasticEnergy;
  else if ( energyName == "SymetricDaugthersEnergy" )
    method = &Self::SymetricDaugthersEnergy;
  else if ( energyName == "SucesorEnergy" )
      method = &Self::SucesorEnergy;
  else if ( energyName == "PredeserorEnergy" )
    method = &Self::PredeserorEnergy;
  else if ( energyName == "AgeEnergy" )
    method = &Self::AgeEnergy;
  else if ( energyName == "AccelerationEnergy" )
    method = &Self::AccelerationEnergy;
  else if ( energyName == "BorderEnergy" )
    method = &Self::BorderEnergy;
  else if ( energyName == "DistanceEnergy" )
    method = &Self::DistanceEnergy;
  else if ( energyName == "NewBourneEnergy" )
    method = &Self::NewBourneEnergy;
  else if ( energyName == "MaxSpeed" )
    method = &Self::MaxSpeed;
  else
    return;

  EnergyNodeType Energy ( std::string ( energyName ), energyValue, this, method );
  energyNodeSet.push_back ( Energy );
}







/**
 * Energy that represent the difference between the inertial movement and the link.
 * @param e
 * @return the energy
 */
template <typename EPtr>
double GraphCellType< EPtr >::MaxSpeed ( EPtr s_Cell )
{
  double localEnergy = 0;

  //Get the sucesors of s
  std::list< EPtr > listSucessorS;
  this->GetSuccessor ( s_Cell, listSucessorS );
  if ( listSucessorS.size() == 1 )
  {
    typename std::list< EPtr >::iterator itSucS = listSucessorS.begin();
    EPtr m_Cell = *itSucS;

    double speed = this->GetSpeed ( m_Cell ) / this->GetTimeStep();
    localEnergy = ( speed > 5 );
  }
  else if ( listSucessorS.size() == 2 )
  {
    typename std::list< EPtr >::iterator itSucS = listSucessorS.begin();
    EPtr m_Cell = *itSucS;
    ++itSucS;
    EPtr n_Cell = *itSucS;

    double a_speed = this->GetSpeed ( m_Cell ) / this->GetTimeStep();
    double b_speed = this->GetSpeed ( n_Cell ) / this->GetTimeStep();

    localEnergy = ( (a_speed > 10) + (b_speed > 10) );

  }
  return localEnergy;
}



template <typename EPtr>
double GraphCellType< EPtr >::SucesorEnergy ( EPtr s_Cell )
{
  double localEnergy = 0;

  //Get the sucesors of s
  std::list< EPtr > listSucessorS;
  this->GetSuccessor ( s_Cell, listSucessorS );

  if ( listSucessorS.size() == 0 )
  {
    localEnergy = 1;
  }
  else if ( listSucessorS.size() == 1 )
  {
    localEnergy = 0;
  }
  else if ( listSucessorS.size() == 2 )
  {
    localEnergy = 0;
  }
  else
  {
    localEnergy = 1000;
  }
  return localEnergy;
}



template <typename EPtr>
    double GraphCellType< EPtr >::PredeserorEnergy ( EPtr s_Cell )
{
  double localEnergy = 0;

  //Get the sucesors of s
  std::list< EPtr > listPredS;
  this->GetPredecessor ( s_Cell, listPredS);

  if ( listPredS.size() == 0 )
  {
    localEnergy = 1;
  }
  else if ( listPredS.size() == 1 )
  {
    localEnergy = 0;
  }
  else
  {
    localEnergy = 1000;
  }
  return localEnergy;
}


template <typename EPtr>
double GraphCellType< EPtr >::AgeEnergy ( EPtr s_Cell )
{
  double localEnergy = 0;

  //Get the sucesors of s
  std::list< EPtr > cellLife;
  cellLife = this->GetCellLife ( s_Cell );
 
  //has Mother Cell?
  long hasMother = this->GetNumberPredecessor ( *cellLife.begin() );
  double ageInMinutes =  abs (  (s_Cell->GetT() - ( *cellLife.begin() )->GetT()) ) * this->GetTimeStep();
  long numberSucc = this->GetNumberSuccessor ( s_Cell );

  if ( hasMother == 1 )
  {
    //NewBorn
    if ( ageInMinutes < 50 )
    {
      switch ( numberSucc )
      {
        case 0:
          localEnergy = 0;
          break;
        case 1:
          localEnergy = 0;
          break;
        case 2:
          localEnergy = 1;
          break;
        default:
          localEnergy = 1;
      }
    }//not NewBorn
    else
    {
      switch ( numberSucc )
      {
        case 0:
          localEnergy = 0;
          break;
        case 1:
          localEnergy = 0;
          break;
        case 2:
          localEnergy = 0;
          break;
        default:
          localEnergy = 0;
      }
    }
  }
  else
  {
    //NewBorn
    if ( numberSucc < 50 )
    {
      switch ( numberSucc )
      {
        case 0:
          localEnergy = 0;
          break;
        case 1:
          localEnergy = 0;
          break;
        case 2:
          localEnergy = 0;
          break;
        default:
          localEnergy = 1;
      }
    }//not NewBorn
    else
    {
      switch ( numberSucc )
      {
        case 0:
          localEnergy = 0;
          break;
        case 1:
          localEnergy = 0;
          break;
        case 2:
          localEnergy = 0;
          break;
        default:
          localEnergy = 1;
      }
    }
  }
  return localEnergy;
}






template <typename EPtr>
double GraphCellType< EPtr >::SymetricDaugthersEnergy ( EPtr s_Cell )
{
  double localEnergy = 0;

  //Get the sucesors of s
  std::list< EPtr > listSucessorS;
  this->GetSuccessor ( s_Cell, listSucessorS );

  if ( listSucessorS.size() == 2 )
  {
    typename std::list< EPtr >::iterator itSucS = listSucessorS.begin();
    EPtr m_Cell = *itSucS;
    ++itSucS;
    EPtr n_Cell = *itSucS;
    
    std::vector<double> s_direction = this->GetDirection ( s_Cell );
    
    std::vector<double> s_positionNoDrift(3,0);
    std::vector<double> m_positionNoDrift(3,0);
    std::vector<double> n_positionNoDrift(3,0);

    m_positionNoDrift[0] = m_Cell->GetX() - s_direction[0];
    m_positionNoDrift[1] = m_Cell->GetX() - s_direction[1];
    m_positionNoDrift[2] = m_Cell->GetX() - s_direction[2];
    
    n_positionNoDrift[0] = n_Cell->GetX() - s_direction[0];
    n_positionNoDrift[1] = n_Cell->GetX() - s_direction[1];
    n_positionNoDrift[2] = n_Cell->GetX() - s_direction[2];
    
    s_positionNoDrift[0] = s_Cell->GetX() ;
    s_positionNoDrift[1] = s_Cell->GetX() ;
    s_positionNoDrift[2] = s_Cell->GetX() ;

    
    double a_speed = sqrt ( pow ( m_positionNoDrift[0] - s_positionNoDrift[0], 2 ) + pow ( m_positionNoDrift[1] - s_positionNoDrift[1], 2 ) + pow ( m_positionNoDrift[2] - s_positionNoDrift[2], 2 ) ) / this->GetTimeStep();
    double b_speed = sqrt ( pow ( n_positionNoDrift[0] - s_positionNoDrift[0], 2 ) + pow ( n_positionNoDrift[1] - s_positionNoDrift[1], 2 ) + pow ( n_positionNoDrift[2] - s_positionNoDrift[2], 2 ) ) / this->GetTimeStep();

    double cellDistance = sqrt ( pow ( m_positionNoDrift[0] - n_positionNoDrift[0], 2 ) + pow ( m_positionNoDrift[1] - n_positionNoDrift[1], 2 ) + pow ( m_positionNoDrift[2] - n_positionNoDrift[2], 2 ) ) ;

    int lenghDiff = abs ( int ( a_speed - b_speed ) ) > 2;
    int mean1 = ( a_speed + b_speed ) / 2.0 > 8;
    int mean2 = ( a_speed + b_speed ) / 2.0 < 2;
    int distance1 = cellDistance < 5;
    int distance2 = cellDistance > 11;

    localEnergy = lenghDiff + mean1 + mean2 + distance1 + distance2;

  }
  return localEnergy;
}


template <typename EPtr>
double GraphCellType< EPtr >::NewBourneEnergy ( EPtr s_Cell )
{
  double localEnergy = 0;

  //Get the sucesors of s
  std::list< EPtr > listSucessorS;
  this->GetSuccessor ( s_Cell, listSucessorS );

  if ( listSucessorS.size() == 2 )
  {
    typename std::list< EPtr >::iterator itSucS = listSucessorS.begin();
    EPtr m_Cell = *itSucS;
    ++itSucS;
    EPtr n_Cell = *itSucS;

    localEnergy = ( m_Cell->GetIsNewBourne() != n_Cell->GetIsNewBourne() );
  }
  return localEnergy;
}



template <typename EPtr>
double GraphCellType< EPtr >::DistanceEnergy ( EPtr s_Cell )
{
  double localEnergy = 0;


  EmbryoStateType< EPtr > *embryoStateS;
  embryoStateS = this->GetEmbryoState ( s_Cell->GetT() );

  //Get the sucesors of s
  std::list< EPtr > listSucessorS;
  this->GetSuccessor ( s_Cell, listSucessorS );


  for ( typename std::list< EPtr >::iterator itSucS = listSucessorS.begin(); itSucS != listSucessorS.end();++itSucS )
  {
    EPtr m_Cell = *itSucS;

    EmbryoStateType< EPtr > *embryoStateM;
    embryoStateM = this->GetEmbryoState ( m_Cell->GetT() );

    std::list< EPtr > listFuture ;
    embryoStateM->GetNearestNCells ( s_Cell, 1, 1, listFuture );

    std::list< EPtr > listPresent ;
    embryoStateS->GetNearestNCells ( m_Cell, -1, 1, listPresent );

    if ( s_Cell == * ( listPresent.begin() ) )
    {
      if ( m_Cell == * ( listFuture.begin() ) )
      {
        localEnergy += 0;
      }
      else
      {
        localEnergy += 1;
      }
    }
    else
    {
      if ( m_Cell == * ( listFuture.begin() ) )
      {
        localEnergy += 1;
      }
      else
      {
        localEnergy += 2;
      }
    }
  }

  return localEnergy;
}




template <typename EPtr>
double GraphCellType< EPtr >::AccelerationEnergy ( EPtr s_Cell )
{
  double localEnergy = 0;

  //Get the pred of s
  std::list< EPtr > listPredS;
  this->GetPredecessor ( s_Cell, listPredS );

  if ( listPredS.size() > 0 )
  {
    EPtr p_Cell = * ( listPredS.begin() );

    double s_speed = sqrt ( pow ( p_Cell->GetX() - s_Cell->GetX(), 2 ) + pow ( p_Cell->GetY() - s_Cell->GetY(), 2 ) + pow ( p_Cell->GetZ() - s_Cell->GetZ(), 2 ) ) ;

    //Get the sucesors of s
    std::list< EPtr > listSucessorS;
    this->GetSuccessor ( s_Cell, listSucessorS );
    double a_speed = 0;

    if ( listSucessorS.size() == 1 )
    {
      localEnergy = std::fabs ( ( float ) ( s_speed - a_speed ) ) ;
    }
    else
    {
      for ( typename std::list< EPtr >::iterator itSucS = listSucessorS.begin(); itSucS != listSucessorS.end();++itSucS )
      {
        EPtr m_Cell = *itSucS;

        a_speed = sqrt ( pow ( m_Cell->GetX() - s_Cell->GetX(), 2 ) + pow ( m_Cell->GetY() - s_Cell->GetY(), 2 ) + pow ( m_Cell->GetZ() - s_Cell->GetZ(), 2 ) ) ;

        localEnergy += std::fabs ( ( float ) ( s_speed - a_speed ) ) ;

      }
    }
  }
  return localEnergy;
}



template <typename EPtr>
double GraphCellType< EPtr >::BorderEnergy ( EPtr s_Cell )
{
  double localEnergy = 0;

  //Get the sucesors of s
  std::list< EPtr > listSucessorS;
  this->GetSuccessor ( s_Cell, listSucessorS );

  long numSuccesors = listSucessorS.size();
  if ( this->CellAtBorder ( s_Cell ) )
  {
    switch ( numSuccesors )
    {
      case 0:
        localEnergy = 1;
        break;
      case 1:
        localEnergy = 0;
        break;
      case 2:
        localEnergy = 2;
        break;
      default:
        localEnergy = 1000;

    }
  }
  else
  {
    switch ( numSuccesors )
    {
      case 0:
        localEnergy = 2;
        break;
      case 1:
        localEnergy = 0;
        break;
      case 2:
        localEnergy = 1;
        break;
      default:
        localEnergy = 1000;
    }
  }
  return localEnergy;
}






template <typename EPtr>
double GraphCellType< EPtr >::PredictionEnergy ( EPtr s_Cell )
{
  double localEnergy = 0;

  //Get the sucesors of s
  std::list< EPtr > listSucessorS;
  this->GetSuccessor ( s_Cell, listSucessorS );

  for ( typename std::list< EPtr >::iterator itSucS = listSucessorS.begin(); itSucS != listSucessorS.end();++itSucS )
  {

    EPtr m_Cell = *itSucS;

    double tempEnergy;

    long outEdge_number = listSucessorS.size();

    long predictedOutEdge_number = 0;
    double predictedOutEdge_Reliability = 0;

    double predictedAge = 0;
    double age_Reliability = 0;

//    ERROR   s_Cell->PredictedNumberOutEdgesOfACell ( predictedOutEdge_number,predictedOutEdge_Reliability );
//       m_Cell->PredictedAge ( predictedAge,age_Reliability );
//
//       tempEnergy = ( double ( outEdge_number != predictedOutEdge_number ) * ( 1-predictedOutEdge_Reliability ) + double ( abs ( this->GetRealTimeLiving ( m_Cell ) - predictedAge ) ) * ( 1.0-age_Reliability ) );

    localEnergy += tempEnergy;
  }

  return localEnergy ;
}

/**
 * Calculate the energy of the edge ||A-B||
 * @param e
 */
template <typename EPtr>
double GraphCellType< EPtr >::ElasticEnergy ( EPtr p_Cell )
{
  double localEnergy = 0;
  Vertex s, v;
  s = this->mapCellNode[p_Cell];
  long countCells = 0;

  //Get the neighbour of s
  std::list< Vertex > listAdjacent;
  this->AdjacentVertexSameTime ( s, listAdjacent );

  for ( typename std::list< Vertex >::iterator it = listAdjacent.begin(); it != listAdjacent.end();++it )
  {
    v = *it;
    EPtr q_Cell;

    q_Cell = this->graph[v].cell;


    //Get the distance between s and v
    double A[3];
    A[0] = p_Cell->GetX() - q_Cell->GetX() ;
    A[1] = p_Cell->GetY() - q_Cell->GetY() ;
    A[2] = p_Cell->GetZ() - q_Cell->GetZ() ;

    //Get the sucesors of p
    std::list< EPtr > listSucessorS;
    this->GetSuccessor ( p_Cell, listSucessorS );

    //Get the sucesors of v
    std::list< EPtr > listSucessorV;
    this->GetSuccessor ( q_Cell, listSucessorV );
    std::vector< double > sucMedianPositionSucV ( 3, 0 );

    std::vector< double > sucMedianPositionSucS ( 3, 0 );

    for ( typename std::list< EPtr >::iterator itSucS = listSucessorS.begin(); itSucS != listSucessorS.end();++itSucS )
    {
      sucMedianPositionSucS[0] += ( *itSucS )->GetX();
      sucMedianPositionSucS[1] += ( *itSucS )->GetY();
      sucMedianPositionSucS[2] += ( *itSucS )->GetZ();
    }

    for ( typename std::list< EPtr >::iterator itSucV = listSucessorV.begin(); itSucV != listSucessorV.end();++itSucV )
    {
      sucMedianPositionSucV[0] += ( *itSucV )->GetX();
      sucMedianPositionSucV[1] += ( *itSucV )->GetY();
      sucMedianPositionSucV[2] += ( *itSucV )->GetZ();

    }


    if ( listSucessorS.size() > 0 && listSucessorV.size() > 0 )
    {
      sucMedianPositionSucS[0] /= ( double ) listSucessorS.size();
      sucMedianPositionSucS[1] /= ( double ) listSucessorS.size();
      sucMedianPositionSucS[2] /= ( double ) listSucessorS.size();

      sucMedianPositionSucV[0] /= ( double ) listSucessorV.size();
      sucMedianPositionSucV[1] /= ( double ) listSucessorV.size();
      sucMedianPositionSucV[2] /= ( double ) listSucessorV.size();

      double B[3];
      B[0] = sucMedianPositionSucS[0] - sucMedianPositionSucV[0] ;
      B[1] = sucMedianPositionSucS[1] - sucMedianPositionSucV[1] ;
      B[2] = sucMedianPositionSucS[2] - sucMedianPositionSucV[2] ;

      double C[3];
      C[0] = A[0] - B[0] ;
      C[1] = A[1] - B[1] ;
      C[2] = A[2] - B[2] ;

      double energy = ( C[0] * C[0] + C[1] * C[1] + C[2] * C[2] );

      countCells++;
      localEnergy +=  energy;
    }
  }

  double resu  = 0;
  if ( countCells > 0 )
  {
    resu = localEnergy / ( double ) countCells;
  }
  
  p_Cell->SetDeformation( resu );
  
  return resu;
}


template <typename EPtr>
void  GraphCellType< EPtr >::SaveSQLFileEmbryoState ( std::ostream& ofs, long id_process_tracking, long time )
{
  std::stringstream SqlReQuest;

  EmbryoStateType< EPtr > *embryoState;
  embryoState = this->GetEmbryoState ( time );

  typename EmbryoStateType< EPtr >::iterator it;

  for ( it = embryoState->begin();it != embryoState->end();++it )
  {

    SqlReQuest << "INSERT INTO z_tree ( id_tree, id_cell, id_cellchild1, id_cellchild2, id_cellmother, cell_name, probability, id_process_tracking ) VALUES (" ;

    SqlReQuest << "'" << 0 << "'" << ", ";
    SqlReQuest << "'" << ( *it )->GetIdDb() << "'" << ", ";


    std::list< EPtr > p_daughters;
    typename std::list< EPtr >::iterator p_daughtersIt;
    this->GetSuccessor ( *it, p_daughters );
    if ( p_daughters.size() == 0 )
    {
      SqlReQuest << "NULL,NULL,";
    }
    else if ( p_daughters.size() == 1 )
    {
      p_daughtersIt = p_daughters.begin();
      EPtr cell = * ( p_daughtersIt );
      SqlReQuest << "'" << cell->GetIdDb() << "'" << ", ";
      SqlReQuest << "NULL,";
    }
    else if ( p_daughters.size() == 2 )
    {
      p_daughtersIt = p_daughters.begin();
      EPtr cell1 = * ( p_daughtersIt );
      SqlReQuest << "'" << cell1->GetIdDb() << "'" << ", ";
      p_daughtersIt++;
      EPtr cell2 = *p_daughtersIt;
      SqlReQuest << "'" << cell2->GetIdDb() << "'" << ", ";
    }


    std::list< EPtr > p_mother;
    this->GetPredecessor ( *it, p_mother );
    if ( p_mother.size() == 0 )
    {
      SqlReQuest << "NULL,";
    }
    else
    {
      EPtr cell = * ( p_mother.begin() );
      SqlReQuest << "'" << cell->GetIdDb() << "'" << ", ";
    }


    std:: stringstream s;
    s << id_process_tracking;

    SqlReQuest << ( *it )->GetName() <<  ", 0 ," << s.str() << ");";

    SqlReQuest << std::endl;
  }

  ofs << SqlReQuest.str();

}




template <typename EPtr>
void  GraphCellType< EPtr >::SaveSQLFileAllEmbryo ( std::ostream& ofs, long  id_process_tracking )
{
  std::set< long > times = this->GetTimes();
  typename std::set< long >::iterator timesIt;

  for ( timesIt = times.begin();timesIt != times.end();++timesIt )
  {
    this->SaveSQLFileEmbryoState ( ofs, id_process_tracking, *timesIt );
  }
}


template <typename EPtr>
bool GraphCellType< EPtr >::operator== ( const GraphCellType< EPtr >& other ) const
{
  bool iqual = true;

  iqual &= Superclass::operator== ( other );//  &&

  //Iquality of the Graph;
  {
    {
//All the vertex of this sould be in other.graph
      vertex_iterator viG1, vi_endG1;
      vertex_iterator viG2, vi_endG2;
      tie ( viG1, vi_endG1 ) = vertices ( this->graph );
      tie ( viG2, vi_endG2 ) = vertices ( other.graph );


      for ( ; viG1 != vi_endG1 && viG2 != vi_endG2; ++viG1, ++viG2 )
      {
        Vertex v1 = *viG1;
        Vertex v2 = *viG2;

        iqual &= v1 == v2;
      }
      iqual &= viG1 == vi_endG1 && viG2 == vi_endG2;
    }
    {
//All the edges of this sould be in other.graph
      edge_iterator eiG1, ei_endG1;
      edge_iterator eiG2, ei_endG2;
      tie ( eiG1, ei_endG1 ) = edges ( this->graph );
      tie ( eiG2, ei_endG2 ) = edges ( other.graph );


      for ( ; eiG1 != ei_endG1 && eiG2 != ei_endG2; ++eiG1, ++eiG2 )
      {
        Edge e1 = *eiG1;
        Edge e2 = *eiG2;

        Vertex s1 = source ( e1, this->graph );
        Vertex v1 = target ( e1, this->graph );

        Vertex s2 = source ( e2, this->graph );
        Vertex v2 = target ( e2, this->graph );

        iqual &= s1 == s2 & v1 == v2;
      }
      iqual &= eiG1 == ei_endG1 && eiG2 == ei_endG2;
    }
  }

  {
    //Iquality of the mapCellNode;

    iqual &=  this->mapCellNode.size() == other.mapCellNode.size();

    typename MapCellNodeType::const_iterator it1 = this->mapCellNode.begin();



    for ( ;it1 != this->mapCellNode.end() && iqual ;++it1 )
    {
      Vertex s = it1->second;
      EPtr  sCell = it1->first;

      std::cout << sCell->GetIdDb() << std::endl;

      for ( typename MapCellNodeType::const_iterator it2 = other.mapCellNode.begin();it2 != other.mapCellNode.end(); ++it2 )
      {
        Vertex v = it2->second;
        EPtr vCell = it2->first;

        if ( v == s )
        {
          iqual &=  vCell->GetIdDb() == sCell->GetIdDb();
          break;
        }
      }
    }
  }

  {
    //Iquality of the multiSet;
//FALTA    iqual &=  multiSet == other.multiSet;
  }


  return iqual;
}


template <typename EPtr>
bool GraphCellType< EPtr >::TestIntegrity()
{
  bool iqual = true;

  /*
  graph.size == mapCellNode.size == multiSet.size;
  V node in map < graph.size
  */


  typename MapCellNodeType::const_iterator it1 = this->mapCellNode.begin();
  for ( ;it1 != this->mapCellNode.end() ;++it1 )
  {
    Vertex s = it1->second;
//    iqual = s < num_vertices(this->graph);
  }
  iqual &= num_vertices ( this->graph ) == mapCellNode.size();
  iqual &= num_vertices ( this->graph ) == multiSet.size();

  /*
    std::set< long > times = GetTimes();
    typename std::set< long >::iterator timesIt;

      for ( timesIt = times.begin();timesIt != times.end();++timesIt )
      {
        long time = *timesIt;


        EmbryoStateType< EPtr > *embryoState;
        embryoState = GetEmbryoState ( time );

        for ( typename EmbryoStateType< EPtr >::iterator stateIt = embryoState->begin();stateIt != embryoState->end();++stateIt )
        {
          EPtr aCell = *stateIt;
          std::list< std::pair < EPtr, double >  > distance;
          std::list< EPtr > aCell_mother;
          GetPredecessor ( aCell, aCell_mother ) ;
        }
      }*/

  typedef std::pair<EPtr, Vertex> PairType;
  BOOST_FOREACH ( PairType pair, mapCellNode )
  {
    std::cout << pair.first << "\t" << pair.second << "\t" ;
  }



  if ( iqual )
    return true;
  else
    return false;

}



template <typename EPtr>
int  GraphCellType< EPtr >::DegreeOut ( EPtr p_Cell )
{
  typename MapCellNodeType::iterator toIt = this->mapCellNode.find ( p_Cell );
  if ( toIt != this->mapCellNode.end() )
  {
    Vertex s;
    s = toIt->second;
    return out_degree ( s, this->graph );
  }
  return -1;
}


template <typename EPtr>
int  GraphCellType< EPtr >::DegreeIn ( EPtr p_Cell )
{
  typename MapCellNodeType::iterator toIt = this->mapCellNode.find ( p_Cell );
  if ( toIt != this->mapCellNode.end() )
  {
    Vertex s;
    s = toIt->second;
    return in_degree ( s, this->graph );
  }
  return -1;
}



template <typename EPtr>
std::set<EPtr> GraphCellType< EPtr >::GetCellsToEnumerate()
{
  return cellsToEnumerate;
}


template <typename EPtr>
void GraphCellType< EPtr >::AddCellToEnumerate ( EPtr p_Cell )
{
  static long iddb = 0;
  p_Cell->SetIdDb ( iddb++ );
  this->cellsToEnumerate.insert ( p_Cell );
}


template < typename EPtr >
void GraphCellType< EPtr >::GetMorphoFields ( std::set<EPtr> & cells, GraphCellType< EPtr > &newEmbryo )
{
  bool first;
  for ( typename std::set<EPtr>::iterator it = cells.begin(); it != cells.end(); ++it )
  {
    try
    {
    //Add the cell Life
      std::list< EPtr > cellLife = this->GetCellLife ( *it );

      {
        EPtr previusCell;
        first = true;
        for ( typename std::list< EPtr >::iterator cell = cellLife.begin();cell != cellLife.end();++cell )
        {
          newEmbryo.AddCell ( *cell );
          if ( !first )
          {
            first = false;
            if ( ExistEdge ( *cell, previusCell ) )
            {
              newEmbryo.AddRelationCell ( *cell, previusCell );
            }
          }
          previusCell = *cell;
        }
      }
    
/*    
    //Cell Life of the Sister
      EPtr sister = GetSister(*(cellLife.begin()));
    
      {
        std::list< EPtr > cellLife = this->GetCellLife ( sister );
        EPtr previusCell;
        first = true;
        for ( typename std::list< EPtr >::iterator cell = cellLife.begin();cell != cellLife.end();++cell )
        {
          newEmbryo.AddCell ( *cell );
          if ( !first )
          {
            first = false;
            if ( ExistEdge ( *cell, previusCell ) )
            {
              newEmbryo.AddRelationCell ( *cell, previusCell );
            }
          }
          previusCell = *cell;
        }
      }
    
    //Mother
      std::list< EPtr > p_mother;
      GetPredecessor ( *(cellLife.begin()),p_mother );
      EPtr  mother = *(p_mother.begin());
    
      {
        std::list< EPtr > cellLife = this->GetCellLife ( mother );
        EPtr previusCell;
        first = true;
        for ( typename std::list< EPtr >::iterator cell = cellLife.begin();cell != cellLife.end();++cell )
        {
          newEmbryo.AddCell ( *cell );
          if ( !first )
          {
            first = false;
            if ( ExistEdge ( *cell, previusCell ) )
            {
              newEmbryo.AddRelationCell ( *cell, previusCell );
            }
          }
          previusCell = *cell;
        }
      }
      
      newEmbryo.AddRelationCell(mother,*it);
      newEmbryo.AddRelationCell(mother,sister);*/
    }
    catch(...)
    {
    
    }
  }
}


template < typename EPtr >
void GraphCellType< EPtr >::MergeLinegeTree ( GraphCellType< EPtr > &p_lineage, BoxType3d &boxSelection, long regionNumber )
{
  //First enumerate and add the cells added
  std::set<EPtr> cellsToEnu = p_lineage.GetCellsToEnumerate();
  for ( typename std::set<EPtr>::iterator cellIt = cellsToEnu.begin();cellIt != cellsToEnu.end();++cellIt )
  {
    EPtr cell = *cellIt;
    if ( boxSelection.IsInside ( cell->GetX(), cell->GetY(), cell->GetZ(), regionNumber ) )
    {
      AddCellToEnumerate ( cell );
      AddCell ( cell );
    }
  }

  std::set<EPtr> allCell = p_lineage.GetAllCells();

  std::cout << "Merging region: " << regionNumber << " Cells: " << allCell.size() << std::endl;
  for ( typename std::set<EPtr>::iterator cellIt = allCell.begin();cellIt != allCell.end();++cellIt )
  {
    EPtr m_Cell = *cellIt;
    if ( boxSelection.GetTimeInit() <= m_Cell->GetT() && m_Cell->GetT() < boxSelection.GetTimeEnd() )
    {

      //THE m_cell is inside, I have to presses it
//      if (regionNumber == 2){std::cout << m_Cell->GetX()<< " " << m_Cell->GetY() << " " <<  m_Cell->GetZ() << std::endl;}
      if ( boxSelection.IsInside ( m_Cell->GetX(), m_Cell->GetY(), m_Cell->GetZ(), regionNumber ) )
      {

        //Get the same cell in myEmbryo
        EPtr a_Cell;
        bool sameCell = this->GetCell ( m_Cell->GetX(), m_Cell->GetY(), m_Cell->GetZ(), m_Cell->GetT(), a_Cell );
        if ( sameCell )
        {
          //If the new configuration has less energy, I modify the embryo.
          double aEnergy = this->GetEnergy ( a_Cell );

          double mEnergy = p_lineage.GetEnergy ( m_Cell );
          if ( aEnergy > mEnergy )
          {
            typename std::list< EPtr > succ_m_Cells;
            p_lineage.GetSuccessor ( m_Cell, succ_m_Cells );


            typename std::list< EPtr > succ_a_Cells;
            this->GetSuccessor ( a_Cell, succ_a_Cells );
            this->RemoveEdgeSuccesors ( a_Cell );


            for ( typename std::list< EPtr >::iterator succ_a_CellsIt = succ_a_Cells.begin(); succ_a_CellsIt != succ_a_Cells.end();++succ_a_CellsIt )
            {
//            if (boxSelection.IsInside((*succ_a_CellsIt)->GetX(),(*succ_a_CellsIt)->GetY(),(*succ_a_CellsIt)->GetZ(),regionNumber))
              {
                this->RemoveEdgePredecessors ( *succ_a_CellsIt );
              }
            }

            for ( typename std::list< EPtr >::iterator succ_m_CellsIt = succ_m_Cells.begin(); succ_m_CellsIt != succ_m_Cells.end();++succ_m_CellsIt )
            {
              EPtr m_CellS = *succ_m_CellsIt;
              EPtr a_CellS ;

              bool sameCell2 = this->GetCell ( m_CellS->GetX(), m_CellS->GetY(), m_CellS->GetZ(), m_CellS->GetT(), a_CellS );


              if ( sameCell2 )
              {
                this->AddRelationCell ( a_Cell, a_CellS );
              }
            }
          }
        }
      }
    }
  }

}



template < typename EPtr >
    void GraphCellType< EPtr >::test()
{
  typename boost::property_map<GraphCellType, long Node::*>::type timeMap = boost::get(&Node::IdDb, graph);
  
  std::ofstream file(std::string(std::string("1.dot")).c_str());
//  write_graphviz(file, graph, boost::make_label_writer(get(timeMap, graph)));
} 


template < typename EPtr >
    void GraphCellType< EPtr >::UpdateProperties()
{
  vertex_iterator vi, vi_end;
  for (boost::tie(vi, vi_end) = boost::vertices(graph); vi != vi_end; ++vi)
  {
    EPtr cell = this->graph[*vi].cell;
    ElasticEnergy(cell);
  }
}
    

#endif
