/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef __MultiTimeSetPoint_h
#define __MultiTimeSetPoint_h

#include "definitions.h"
#include <iostream>
#include <map>
#include <set>
#include "SetPointsWithLocator.h"


#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include <boost/serialization/map.hpp>
#include <boost/serialization/set.hpp>



/*! \brief Implement a set where the points are agrupated with the Time coordinate
 * The class implemente a set of point where the points can be access by the locator in a fast way. The points should provide the members GetX(),GetY(),GetZ(),GetT().
 */
template <typename EPtr>
class MultiTimeSetPoint
{
	public:
		typedef MultiTimeSetPoint Self;
	public:
		typedef std::map< long, SetPointsWithLocator < EPtr > >  MapTimeCellsType;
                typedef std::set< long > SetTimeType;
        private:
		typedef std::map<  long, EPtr > MapIdDBCellType;

	private:

	public:
		MultiTimeSetPoint();
		~MultiTimeSetPoint();
                long size();
                
                const std::set< long >& GetTimes();
                

		void CalculateNeighborAllTimeSteps();
		EPtr GetCellIdDb ( long IdDb );
                
		/*!   Add a cell to the set
		    \param p_Cell cell to add to the set
		 */
		virtual void AddCell ( EPtr p_Cell );
		/*!   Retrieve the SetPointsWithLocator of the time t
		\param time time
		*/
                SetPointsWithLocator < EPtr >* GetEmbryoState ( long time );
                
                virtual void RemoveCell ( EPtr p_Cell );

                
                
	private:
		MapTimeCellsType m_MapTimeCells;
		SetTimeType m_times;
		MapIdDBCellType  m_MapIdDBCell;

		friend class boost::serialization::access;
		template<class Archive> void serialize ( Archive & ar, const unsigned int version )
		{
			ar & m_MapTimeCells;
			ar & m_times;
			ar & m_MapIdDBCell;
		}
};

template<typename EPtr>
    MultiTimeSetPoint< EPtr >::MultiTimeSetPoint()
{

}
template<typename EPtr>
MultiTimeSetPoint< EPtr >::~MultiTimeSetPoint()
{
}


/**
 * For all the cells the atribute of the cell Neigbourb is calculated
 */
template<typename EPtr>
    void MultiTimeSetPoint< EPtr >::CalculateNeighborAllTimeSteps()
{
	for ( SetTimeType::iterator it = m_times.begin(); it != m_times.end(); ++it )
	{
		long time = *it;
		SetPointsWithLocator < EPtr >* cells;
                cells = GetEmbryoState ( time );
		if ( cells!=  NULL )
		{
			cells->CalculateNeighbor();
		}

	}
}


/*! Return the cell the have the IdDb or NULL
 * \param IdDb the id of the cell to retrieve
 * \return return the a Cell pointer or NULL
 */
template<typename EPtr>
    EPtr MultiTimeSetPoint< EPtr >::GetCellIdDb ( long IdDb )
{
	typename MapIdDBCellType::iterator it;

	it = m_MapIdDBCell.find ( IdDb );
	if ( it == m_MapIdDBCell.end() )
		throw 1;
	else
		return it->second;
};


/**
 * Add a cell to the set
 * \param p_Cell cell to add to the set
 */
template<typename EPtr>
    void MultiTimeSetPoint< EPtr >::AddCell ( EPtr p_Cell )
{
	typename MapTimeCellsType::iterator it;

	it = this->m_MapTimeCells.find ( p_Cell->GetT() );
	if ( it == this->m_MapTimeCells.end() )
	{
		this->m_MapTimeCells.insert(std::make_pair(p_Cell->GetT(),SetPointsWithLocator < EPtr >()));
		it = this->m_MapTimeCells.find ( p_Cell->GetT() );
	}
	it->second.AddCell ( p_Cell );
	this->m_MapIdDBCell.insert ( std::make_pair ( p_Cell->GetIdDb(), p_Cell ) ); //

	this->m_times.insert ( p_Cell->GetT() );
}




template<typename EPtr>
    void MultiTimeSetPoint< EPtr >::RemoveCell ( EPtr p_Cell )
{
  this->m_MapIdDBCell.erase ( p_Cell->GetIdDb() );
  
  typename MapTimeCellsType::iterator it;
  it = this->m_MapTimeCells.find ( p_Cell->GetT() );
  if ( it != this->m_MapTimeCells.end() )
  {
    it->second.RemoveCell( p_Cell );
    if (it->second.size()==0)
    {
      this->m_MapTimeCells.erase ( p_Cell->GetT() );
      this->m_times.erase ( p_Cell->GetT() );
    }
  }
}


/*!   Retrieve the SetPointsWithLocator of the time t
\param time time
 */
template<typename EPtr>
    SetPointsWithLocator < EPtr >* MultiTimeSetPoint< EPtr >::GetEmbryoState ( long time )
{
	typename MapTimeCellsType::iterator it;

	it = this->m_MapTimeCells.find ( time );

	if ( it != this->m_MapTimeCells.end() )
	{
		return &it->second;
	}
        throw 1;
}


template<typename EPtr>
 const std::set< long >& MultiTimeSetPoint< EPtr >::GetTimes()
{
  return (m_times);
}

template<typename EPtr>
long  MultiTimeSetPoint< EPtr >::size()
{
  long resu = 0;
  typename MapTimeCellsType::iterator it;
  for (it = m_MapTimeCells.begin();it != m_MapTimeCells.end();++it)
  {
    resu += it->second.size();
  }

return resu;

}

#endif


