#pragma once

/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */


#ifndef __sortpoints3_h

#define __sortpoints3_h



#include <list>

#include <math.h>

#include <ANN/ANN.h>

#include <algorithm>

#include <boost/archive/text_oarchive.hpp>

#include <boost/archive/text_iarchive.hpp>

#include <boost/serialization/set.hpp>





/*! \brief A locator class.

 * The class implemente a set of points where the points located fast.

 */

template < class T, class EPtr>

class SortPoints3

{

    long dim;

  private:

    /*! Constructor

     */

    SortPoints3 ();

  public:

    /*! Constructor

    \param p_collection colection with the elements

     */



    SortPoints3 ( T* p_collection ) : dim ( 3 ), ANNtree ( boost::shared_ptr< ANNkd_tree >() ), pa ( NULL ), collection ( p_collection )

    {

    }



    /*!

    \param point point where the returned cells are near

    \param nn number of returned cells

    \param list the elements list are the nn neares cells to the input paremter point

     */

    void GetNearestNCells ( const double point[3], long nn, std::list< EPtr > &list )

    {

      list.clear();

      ANNpoint      queryPt; // query point

      ANNidxArray   nnIdx;   // near neighbor indices

      ANNdistArray  dists;   // near neighbor distances



      queryPt = annAllocPt ( dim );        // allocate query point

      queryPt[0] = point[0];

      queryPt[1] = point[1];

      queryPt[2] = point[2];



      nnIdx = new ANNidx[nn];              // allocate near neigh indices

      dists = new ANNdist[nn];             // allocate near neighbor dists



      long n = std::distance ( this->collection->begin(), this->collection->end() );

      if ( this->ANNtree.get() == NULL )

      {

        typename T::iterator it2;

        pa = annAllocPts ( n, dim );



        long i;

        it2 = collection->begin();

        for ( i = 0; i < n; ++i )

        {

          pa[i][0] = ( *it2 )->GetX();

          pa[i][1] = ( *it2 )->GetY();

          pa[i][2] = ( *it2 )->GetZ();



          ++it2;

        }



        boost::shared_ptr< ANNkd_tree > ANNtree2(new ANNkd_tree (	// build it

                                                            pa,			// the data points

                                                            n,			// number of points

                                                            dim ));

        this->ANNtree = ANNtree2;

      }



      if ( n < nn )

      {

        nn = n - 1;

      }

      this->ANNtree->annkSearch ( queryPt, nn, nnIdx, dists );



      for ( int v = 0;v < nn;v++ )

      {

        int index = nnIdx[v];

        typename T::iterator it2 = this->collection->begin();



        std::advance ( it2, index );

        list.push_back ( ( *it2 ) );

      }

      annDeallocPt ( queryPt );



      delete [] nnIdx;

      delete [] dists;

    }

    /*!

          \param aCell Cell where the returned cells are near

          \param nn number of returned cells

          \param list the elements list are the nn neares cells to the input paremter aCell

     */

    void GetSortedList ( const EPtr aCell, long nn, std::list< EPtr > &list )

    {

      double point[3];

      point[0] = aCell->GetX();

      point[1] = aCell->GetY();

      point[2] = aCell->GetZ();



      this->GetNearestNCells ( point, nn, list );

    }

    /*!

    Destructor

    */



    ~SortPoints3 ( void )

    {

      if ( pa != NULL )

      {

        annDeallocPts ( pa );

        pa = NULL;

      }

    }

  private:

    friend class boost::serialization::access;

    template<class Archive> void serialize ( Archive & ar, const unsigned int version )

    {

      ar & collection;

    }

  private:

    T* collection;

    ANNpointArray pa; // data points

  public:

    boost::shared_ptr< ANNkd_tree > ANNtree;// search structure

};



template < class T, class EPtr>

    SortPoints3< T, EPtr >::SortPoints3() : dim ( 3 ), ANNtree ( boost::shared_ptr< ANNkd_tree >() ), pa ( NULL )

{

}





#endif



