/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef lineageconstructor_h
#define lineageconstructor_h

#include "EmbryoType.h"
#include "GraphCellType.h"

#include "EmbryoStateType.h"


#include <functional>
#include <numeric>
#include <boost/foreach.hpp>



/**
	@author
 */
template < typename EPtr >
class LineageConstructor
{

  private:
    typedef LineageConstructor Self;
  public:
    typedef typename std::pair<EPtr, EPtr> LinkType;



    LineageConstructor();

    ~LineageConstructor();

    GraphCellType < EPtr > *GetEmbryo()
    {
      return this->m_embryo;
    };
    void SetEmbryo ( GraphCellType< EPtr > *p_embryo )
    {
      this->m_embryo = p_embryo;
    };


  public:
    void SetDoubleLinks ( long time_start_number, long time_end_number );
    void SetForwardLinks ( long time_start_number, long time_end_number );
    void SetBackwardLinks ( long time_start_number, long time_end_number );
    void SetMitoticLinks ( long time_start_number, long time_end_number );
    void FilterShortLifeCells ( long time );
    void DeleteShortLifeCells ( long p_time );
    void DeleteShortLifeCells ( long p_time, long InitUpload, long EndUpload );

    std::set <EPtr> GetCellsDeleted ( );

  private:
    void GetNearestPairs ( EmbryoStateType< EPtr > *embryoState1, EmbryoStateType< EPtr > *embryoState2, long direction, std::set<LinkType> &links );

    std::set <EPtr> cellsToDelete;


//     friend class boost::serialization::access;
//     template<class Archive> void serialize ( Archive & ar, const unsigned int version )
//     {
//     }

  private:

    GraphCellType < EPtr > *m_embryo;
};

template < typename EPtr >
    bool goodCellDivition ( EPtr mCell, EPtr aCell, EPtr bCell )
{
  double a_speed = sqrt ( pow ( aCell->GetX() - mCell->GetX(), 2 ) + pow ( aCell->GetY() - mCell->GetY(), 2 ) + pow ( aCell->GetZ() - mCell->GetZ(), 2 ) ) ;
  double b_speed = sqrt ( pow ( bCell->GetX() - mCell->GetX(), 2 ) + pow ( bCell->GetY() - mCell->GetY(), 2 ) + pow ( bCell->GetZ() - mCell->GetZ(), 2 ) ) ;

  double cellDistance = sqrt ( pow ( aCell->GetX() - bCell->GetX(), 2 ) + pow ( aCell->GetY() - bCell->GetY(), 2 ) + pow ( aCell->GetZ() - bCell->GetZ(), 2 ) ) ;

  bool lenghDiff = abs ( int ( a_speed - b_speed ) ) < 6;
  bool mean1 = ( a_speed + b_speed ) / 2.0 < 8;
  bool mean2 = ( a_speed + b_speed ) / 2.0 > 2;
  bool distance = cellDistance > 4;

  return ( lenghDiff && mean1 && mean2 && distance );
}


template < typename EPtr >
LineageConstructor< EPtr >::LineageConstructor()
{


}


template < typename EPtr >
    LineageConstructor< EPtr >::~LineageConstructor()
{


}

template < typename EPtr >
    void LineageConstructor< EPtr >::GetNearestPairs ( EmbryoStateType< EPtr > *embryoState1, EmbryoStateType< EPtr > *embryoState2, long direction, std::set<LinkType> &links )
{
  typename EmbryoStateType< EPtr >::iterator it1;
  for ( it1 = embryoState2->begin(); it1 != embryoState2->end(); ++it1 )
  {
    EPtr y_Cell = *it1;

    std::list< EPtr > list ;
    embryoState1->GetNearestNCells ( y_Cell, (-1)*direction,1, list );

    if ( list.size() == 1 )
    {
      LinkType link;

      EPtr x_Cell = * ( list.begin() );
      if ( direction== 1)
      {
        link = std::make_pair ( x_Cell , y_Cell );
      }
      else
      {
        link = std::make_pair ( y_Cell, x_Cell );
      }

      links.insert ( link );
    }
  }
}


template < typename EPtr >
    void LineageConstructor< EPtr >::SetDoubleLinks ( long time_start_number, long time_end_number )
{
  std::set<LinkType> doubleLinks;
  std::cout << "Begin " << std::endl;
  std::set< long > times = this->GetEmbryo()->GetTimes();
  typename std::set< long >::iterator timesIt;

  std::cout <<"SetDoubleLinks "<<time_start_number <<" " <<time_end_number <<" " << std::endl;
	
  if ( times.size() != 0 )
  {
   std::cout <<"GetTimeMax Min "<<this->GetEmbryo()->GetTimeMin() <<" " << this->GetEmbryo()->GetTimeMax() <<" " << std::endl;
  
   for ( timesIt = times.begin();timesIt != times.end();++timesIt )
   {
     if ( time_start_number <= *timesIt && *timesIt  < time_end_number )
     {
        typename std::set< long >::iterator timesIt2 = timesIt;
        timesIt2++;


        if ( timesIt2 != times.end() )
        {
          EmbryoStateType< EPtr > *embryoState1;
          embryoState1 = this->GetEmbryo()->GetEmbryoState ( *timesIt );

          EmbryoStateType< EPtr > *embryoState2;
          embryoState2 = this->GetEmbryo()->GetEmbryoState ( *timesIt2 );

          typename std::set<LinkType> forwardLinks;
          this->GetNearestPairs ( embryoState1, embryoState2, 1, forwardLinks );
          typename std::set<LinkType> backwarLinks;
          this->GetNearestPairs ( embryoState2, embryoState1, -1, backwarLinks );

          std::cout << "Total Cells: \t" << *timesIt << "\t" << forwardLinks.size() << std::endl;

          set_intersection ( forwardLinks.begin(), forwardLinks.end(), backwarLinks.begin(), backwarLinks.end(), std::insert_iterator< std::set< LinkType > > ( doubleLinks, doubleLinks.begin() ) );
        }
      }
    }

    std::vector<long> vectorCant ( * ( times.rbegin() ) + 1, 0 );

    for ( typename std::set<LinkType>::iterator it = doubleLinks.begin();
          it != doubleLinks.end();++it )
    {
      EPtr sCell = it->first;
      EPtr nCell = it->second;
      
      if ( this->GetEmbryo()->GetNumberSuccessor ( it->first ) == 0 &&
           this->GetEmbryo()->GetNumberPredecessor ( it->second ) == 0 )
      {
        this->GetEmbryo()->AddRelationCell ( it->first, it->second );

        vectorCant[it->first->GetT() ]++;
      }
    }




    for ( timesIt = times.begin();timesIt != times.end();++timesIt )
    {
      long time = *timesIt;
      if (vectorCant[time]!=0)
      {
        std::cout << "Time: \t" << *timesIt << "\t Double: \t " << vectorCant[*timesIt] << std::endl;
      }
    }
  }
}

template < typename EPtr >
    void LineageConstructor< EPtr >::SetForwardLinks ( long time_start_number, long time_end_number )
{
  std::set< long > times = this->GetEmbryo()->GetTimes();
  typename std::set< long >::iterator timesIt;

  std::set<LinkType> doubleLinks;


  std::cout <<"SetForwardLinks "<<time_start_number <<" " <<time_end_number <<" " << std::endl;

  for ( timesIt = times.begin();timesIt != times.end();++timesIt )
  {
    long time = *timesIt;

    if ( time_start_number <= time && time < time_end_number )
    {
      std::set< EPtr > setCells = this->GetEmbryo()->GetDyingCells ( time );


      typename std::set< EPtr >::iterator it;
      for ( it = setCells.begin(); it != setCells.end();++it )
      {
        EPtr aCell = *it;
        EmbryoStateType< EPtr > *embryoState;
        embryoState = this->GetEmbryo()->GetEmbryoState ( time );

        std::list< std::pair < EPtr, double >  > distance;

        if ( !this->GetEmbryo()->CellAtBorder ( aCell ) )
        {

          typename std::set< long >::iterator timesIt2, timesItEnd;
          timesIt2 = times.find ( time );
          std::advance ( timesIt2, 1 );
          timesItEnd = timesIt2;
          std::advance ( timesItEnd, 1 );

          std::set<EPtr> possibleCells;

          for ( ; timesIt2 != timesItEnd && timesIt2 != times.end();++timesIt2 )
          {
            EmbryoStateType< EPtr > *embryoStateFuture;
            embryoStateFuture = this->GetEmbryo()->GetEmbryoState ( *timesIt2 );

            std::list< EPtr > list ;
            embryoStateFuture->GetNearestNCells ( aCell, 1, 4, list );

            for ( typename std::list< EPtr >::iterator posibleIt = list.begin() ; posibleIt != list.end() ;++posibleIt )
            {

              std::list< EPtr > p_mother;
              this->GetEmbryo()->GetPredecessor ( *posibleIt, p_mother ) ;

              if ( p_mother.size() == 0 )
              {
                std::vector<double> position ( 3 );
                position[0] = aCell->GetX();
                position[1] = aCell->GetY();
                position[2] = aCell->GetZ();


                std::vector<double> direction = this->GetEmbryo()->GetDirection ( aCell );

                std::vector<double> positionFuture ( 3 );

                long deltaT = *timesIt2 - aCell->GetT();

                positionFuture[0] = direction[0] * deltaT + aCell->GetX();
                positionFuture[1] = direction[1] * deltaT + aCell->GetY();
                positionFuture[2] = direction[2] * deltaT + aCell->GetZ();

                double dist = sqrt ( pow ( positionFuture[0] - direction[0], 2 ) +
                                     pow ( positionFuture[1] - direction[1], 2 ) +
                                     pow ( positionFuture[2] - direction[2], 2 ) );

                distance.push_back ( std::make_pair ( *posibleIt, dist ) );


              }
            }
          }
        }
        typename std::list< std::pair < EPtr, double >  >::iterator distanceIt;

#define MAX 10000000
        double MIN_DISTANCE = MAX;
        EPtr nearestCell;



        for ( distanceIt = distance.begin();distanceIt != distance.end();++distanceIt )
        {

          if ( MIN_DISTANCE > distanceIt->second )
          {
            MIN_DISTANCE = distanceIt->second;
            nearestCell = distanceIt->first;
          }
        }

        if ( MIN_DISTANCE != MAX )
        {
          doubleLinks.insert ( std::make_pair ( aCell, nearestCell ) );
        }
      }
    }
  }


//  std::vector<long> vectorCant(*timesIt,0);
//  std::vector<long> vectorCant ( * ( times.rbegin() ) + 1, 0 );

  for ( typename std::set<LinkType>::iterator it = doubleLinks.begin();
        it != doubleLinks.end();++it )
  {
    this->GetEmbryo()->AddRelationCell ( it->first, it->second );
//    vectorCant[it->first->GetT() ]++;
  }
}





//set a link to a previus times step
// possible a mithosys is added.
template < typename EPtr >
void LineageConstructor< EPtr >::SetBackwardLinks ( long time_start_number, long time_end_number )
{
  std::set< long > times = this->GetEmbryo()->GetTimes();
  typename std::set< long >::iterator timesIt;

  std::set<LinkType> doubleLinks;

  std::cout <<"SetBackwardLinks "<<time_start_number <<" " <<time_end_number <<" " << std::endl;

  for ( timesIt = times.begin();timesIt != times.end();++timesIt )
  {
    long time = *timesIt;

    if ( time_start_number < *timesIt && *timesIt  <= time_end_number )
    {
      EmbryoStateType< EPtr > *embryoState;
      embryoState = this->GetEmbryo()->GetEmbryoState ( time );

      for ( typename EmbryoStateType< EPtr >::iterator stateIt = embryoState->begin();stateIt != embryoState->end();++stateIt )
      {
        EPtr aCell = *stateIt;
        std::list< std::pair < EPtr, double >  > distance;
        std::list< EPtr > aCell_mother;
        this->GetEmbryo()->GetPredecessor ( aCell, aCell_mother ) ;

        if ( aCell_mother.size() == 0 )
        {
          if ( !this->GetEmbryo()->CellAtBorder ( aCell ) )
          {
            typename std::set< long >::iterator timesIt2, timesItEnd;
            timesIt2 = times.find ( time );
            if ( timesIt2 != times.begin() )
            {
              timesItEnd = timesIt2;
              std::advance ( timesItEnd, -1 );

              EmbryoStateType< EPtr > *embryoStatePast;
              embryoStatePast = this->GetEmbryo()->GetEmbryoState ( *timesItEnd );

              std::list< EPtr > list ;
              embryoStatePast->GetNearestNCells ( aCell,-1 , 4, list );

              BOOST_FOREACH ( EPtr mCell, list )
              {
                std::list< EPtr > mDaugthers;
                this->GetEmbryo()->GetSuccessor ( mCell, mDaugthers ) ;

                //IF It is possible to add a new link.
                if ( mDaugthers.size() < 2 )
                {
                  //IF It is possible to add a new link.
                  if ( mDaugthers.size() == 0 )
                  {
                    doubleLinks.insert ( std::make_pair ( mCell, aCell ) );
                    break;
                  }
                  if ( mDaugthers.size() == 1 )
                  {


                    EPtr bCell = * ( mDaugthers.begin() );

                    if ( goodCellDivition ( mCell, aCell, bCell ) )
                    {

                      if ( aCell != bCell )
                      {
                        doubleLinks.insert ( std::make_pair ( mCell, aCell ) );
                      }

                      break;
                    }
                  }
                }
              }
              //FOREACH_mCell
            }
          }
        }
      }
      //FOREACH_aCell
    }
    //For time
  }
//  std::vector<long> vectorCant(*timesIt,0);
  if ( times.size() != 0 )
  {
    std::vector<long> vectorCant ( * ( times.rbegin() ) + 1, 0 );


    for ( typename std::set<LinkType>::iterator it = doubleLinks.begin();
          it != doubleLinks.end();++it )
    {
      this->GetEmbryo()->AddRelationCell ( it->first, it->second );
      vectorCant[it->first->GetT() ]++;
    }


    long time = 0;
//    for ( std::vector<long>::iterator it = vectorCant.begin();it != vectorCant.end();it++ )
    for ( timesIt = times.begin();timesIt != times.end();++timesIt )
    {
      time = *timesIt;
      if (vectorCant[time]!=0)
      {
        std::cout << "Time: \t" << time << "\t Backward: \t " << vectorCant[time] << std::endl;
      }
    }
  }
}



/*
template < typename EPtr >
void LineageConstructor< EPtr >::SetMitoticLinks()
{
  std::set< long > times = this->GetEmbryo()->GetTimes();
  typename std::set< long >::iterator timesIt;

  std::set<LinkType> doubleLinks;

  for ( timesIt = times.begin();timesIt != times.end();++timesIt )
  {
    long time = *timesIt;

    EmbryoStateType< EPtr > *embryoState;
    embryoState = this->GetEmbryo()->GetEmbryoState ( time );

    for ( typename EmbryoStateType< EPtr >::iterator stateIt = embryoState->begin();stateIt != embryoState->end();++stateIt )
    {
      EPtr aCell = *stateIt;
      std::list< std::pair < EPtr, double >  > distance;
      std::list< EPtr > aCell_mother;
      this->GetEmbryo()->GetPredecessor ( aCell, aCell_mother ) ;

      if ( aCell_mother.size() == 0 )
      {
        if ( !this->GetEmbryo()->CellAtBorder ( aCell ) )
        {

          typename std::set< long >::iterator timesIt2, timesItEnd;
          timesIt2 = times.find ( time );
          if ( timesIt2 != times.begin() )
          {
            timesItEnd = timesIt2;
            std::advance ( timesItEnd, -1 );

            EmbryoStateType< EPtr > *embryoStatePast;
            embryoStatePast = this->GetEmbryo()->GetEmbryoState ( *timesItEnd );

            std::list< EPtr > list ;
            embryoStatePast->GetNearestNCells ( aCell, 4, list );

            BOOST_FOREACH ( EPtr mCell, list )
            {
              std::list< EPtr > mDaugthers;
              this->GetEmbryo()->GetSuccessor ( mCell, mDaugthers ) ;

              //IF It is possible to add a new link.
              if ( mDaugthers.size() == 0 )
              {

                std::list< EPtr > aSiblings;
                embryoState->GetNearestNCells ( aCell, 4, aSiblings );

                BOOST_FOREACH ( EPtr bCell, aSiblings )
                {

                  std::list< EPtr > bCell_mother;
                  this->GetEmbryo()->GetPredecessor ( bCell, bCell_mother ) ;

                  if ( bCell_mother.size() == 0 )
                  {
                    if (goodCellDivition(mCell,aCell,bCell))
                    {

                      if (bCell != aCell)
                      {
                      doubleLinks.insert ( std::make_pair ( mCell, bCell ) );
                      doubleLinks.insert ( std::make_pair ( mCell, aCell ) );
                      break;
                      }

                    }
                  }
                }
              }
            }
            //FOREACH_mCell
          }
        }
      }
    }
    //FOREACH_aCell
  }
  //For time
  std::vector<long> vectorCant(*timesIt,0);

  for ( typename std::set<LinkType>::iterator it = doubleLinks.begin();
        it != doubleLinks.end();++it )
  {
    this->GetEmbryo()->AddRelationCell ( it->first, it->second );
    vectorCant[it->first->GetT()]++;
  }


  long time = 0;
  for (std::vector<long>::iterator it = vectorCant.begin();it != vectorCant.end();it++)
  {
    std::cout << "Time: \t" << time << "\t Mithotic: \t " << *it << std::endl;
    time++;
  }
}


*/

template < typename EPtr >
void LineageConstructor< EPtr >::SetMitoticLinks ( long time_start_number, long time_end_number )
{
  std::set< long > times = this->GetEmbryo()->GetTimes();
  std::set<LinkType> doubleLinks;

  long a = 0;
  std::cout <<"SetMitoticLinks "<<time_start_number <<" " <<time_end_number <<" " << std::endl;

  for ( typename EmbryoType< EPtr >::iterator cellIt = this->GetEmbryo()->begin(); cellIt != this->GetEmbryo()->end();++cellIt )
  {

    EPtr aCell;
    aCell = *cellIt;

    if ( time_start_number < aCell->GetT() && aCell->GetT() <= time_end_number )
    {
      EmbryoStateType< EPtr > *embryoState;
      embryoState = this->GetEmbryo()->GetEmbryoState ( aCell->GetT() );


      std::list< std::pair < EPtr, double >  > distance;
      std::list< EPtr > aCell_mother;
      this->GetEmbryo()->GetPredecessor ( aCell, aCell_mother ) ;

      if ( aCell_mother.size() == 0 )
      {
        if ( !this->GetEmbryo()->CellAtBorder ( aCell ) )
        {

          typename std::set< long >::iterator timesIt2, timesItEnd;
          timesIt2 = times.find ( aCell->GetT() );
          if ( timesIt2 != times.begin() )
          {
            timesItEnd = timesIt2;
            std::advance ( timesItEnd, -1 );

            EmbryoStateType< EPtr > *embryoStatePast;
            embryoStatePast = this->GetEmbryo()->GetEmbryoState ( *timesItEnd );

            std::list< EPtr > list ;
            embryoStatePast->GetNearestNCells ( aCell, -1, 4, list );

            BOOST_FOREACH ( EPtr mCell, list )
            {
              std::list< EPtr > mDaugthers;
              this->GetEmbryo()->GetSuccessor ( mCell, mDaugthers ) ;

              //IF It is possible to add a new link.
              if ( mDaugthers.size() == 0 )
              {

                std::list< EPtr > aSiblings;
                embryoState->GetNearestNCells ( aCell, 0, 4, aSiblings );

                BOOST_FOREACH ( EPtr bCell, aSiblings )
                {

                  std::list< EPtr > bCell_mother;
                  this->GetEmbryo()->GetPredecessor ( bCell, bCell_mother ) ;

                  if ( bCell_mother.size() == 0 )
                  {
                    if ( goodCellDivition ( mCell, aCell, bCell ) )
                    {

                      if ( bCell != aCell )
                      {
                        doubleLinks.insert ( std::make_pair ( mCell, bCell ) );
                        doubleLinks.insert ( std::make_pair ( mCell, aCell ) );
                        break;
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  }
  //For time
  if ( times.size() != 0 )
  {

    std::vector<long> vectorCant ( * ( times.rbegin() ) + 1, 0 );

    for ( typename std::set<LinkType>::iterator it = doubleLinks.begin();
          it != doubleLinks.end();++it )
    {
      this->GetEmbryo()->AddRelationCell ( it->first, it->second );
      vectorCant[it->first->GetT() ]++;
    }


    for ( typename std::set< long >::iterator timesIt = times.begin();timesIt != times.end();++timesIt )
    {
      long time = *timesIt;
      if (vectorCant[time]!=0)
      {
        std::cout << "Time: \t" << *timesIt << "\t Mitotic: \t " << vectorCant[*timesIt] << std::endl;
      }
    }
  }
}


template < typename EPtr >
/**
 * DElete the link with the mother if the cell live for less than p_time time steps.
 * @param p_time
 */
void LineageConstructor< EPtr >::FilterShortLifeCells ( long p_time )
{
  std::set< long > times = this->GetEmbryo()->GetTimes();
  typename std::set< long >::iterator timesIt;

  std::set<LinkType> doubleLinks;

  if ( times.size() != 0 )
  {

    std::vector<long> vectorCant ( *times.rbegin(), 0 );

    for ( timesIt = times.begin();timesIt != times.end();++timesIt )
    {
      long time = *timesIt;


      EmbryoStateType< EPtr > *embryoState;
      embryoState = this->GetEmbryo()->GetEmbryoState ( time );

      for ( typename EmbryoStateType< EPtr >::iterator stateIt = embryoState->begin();stateIt != embryoState->end();++stateIt )
      {
        EPtr aCell = *stateIt;

        if ( this->GetEmbryo()->GetTimeStepsLiving ( aCell ) == 0 )
        {
          std::list< EPtr > aCellLife = this->GetEmbryo()->GetCellLife ( aCell );
          if ( aCellLife.size() <= p_time )
          {
            std::list< EPtr > aCell_mother;
            this->GetEmbryo()->GetPredecessor ( aCell, aCell_mother ) ;

            if ( aCell_mother.size() > 0 )
            {
              EPtr mCell;
              mCell = * ( aCell_mother.begin() );
              this->GetEmbryo()->RemoveRelationCell ( mCell , aCell );
              vectorCant[mCell ->GetT() ]++;
            }
          }
        }
      }
    }


    for ( timesIt = times.begin();timesIt != times.end();++timesIt )
    {
      long time = *timesIt;
      if (vectorCant[time]!=0)
      {
        std::cout << "Time: \t" << time << "\t Deleted: \t " << vectorCant[time] << std::endl;
      }
    }

    
    
  }
}


template < typename EPtr >
void LineageConstructor< EPtr >::DeleteShortLifeCells ( long p_time )
{
  std::set< long > times = this->GetEmbryo()->GetTimes();
  if ( times.size() != 0 )
  {
    DeleteShortLifeCells ( p_time, *times.begin(), *times.rbegin() );
  }
}

template < typename EPtr >
void LineageConstructor< EPtr >::DeleteShortLifeCells ( long p_time, long InitUpload, long EndUpload )
{
  for ( typename EmbryoType< EPtr >::iterator cellIt = this->GetEmbryo()->begin(); cellIt != this->GetEmbryo()->end();++cellIt )
  {

    EPtr aCell;
    aCell =  *cellIt;

    if ( InitUpload <= aCell->GetT() && aCell->GetT() <= EndUpload )
    {


      if ( aCell->GetT() < * ( this->GetEmbryo()->GetTimes().rbegin() ) - p_time )
      {
        std::list< EPtr > aCellLife = this->GetEmbryo()->GetCellLife ( aCell );
        if ( aCellLife.size() <= p_time )
        {
          if ( this->GetEmbryo()->GetNumberSuccessor ( * ( aCellLife.rbegin() ) ) == 0 )
          {
            BOOST_FOREACH ( EPtr aCell, aCellLife )
            {
              cellsToDelete.insert ( aCell );
            }
          }
        }
      }
    }
  }
  std::cout << "Cells to delete: " << cellsToDelete.size() << std::endl;
  this->GetEmbryo()->RemoveCells ( cellsToDelete );
}



template < typename EPtr >
std::set <EPtr> LineageConstructor< EPtr >::GetCellsDeleted ()
{
  return cellsToDelete;
}



#endif
