/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef simulatedannealing_h
#define simulatedannealing_h


#include <boost/foreach.hpp>
#include <boost/filesystem.hpp>

#include <functional>
#include <algorithm>
#include <numeric>

#include "GraphCellType.h"
#include "EmbryoType.h"
#include "MultiTimeSetPoint.h"
#include "CommandType.h"
#include "IOfunctions.h"
#include "Solution.h"



typedef enum AllowedMovesType { AMT_MOVE, AMT_ADD, AMT_DELETE, AMT_REPLACE, AMT_NOTHING, AMT_ADD_BACK, AMT_DELETE_BACK} ;


template <class EPtr>
class FixedStates
{
  public:
    typedef enum {InEdges, OutEdges} StatesType;
  private:
    typedef std::set< EPtr > setCellsType;
    typedef std::map < StatesType, setCellsType > mapStatedCellSetType;
  public:

    FixedStates() {};
    ~FixedStates() {};

    void SetFix ( EPtr p_Cell, StatesType state, bool fix )
    {
      typename mapStatedCellSetType::iterator mapIt;

      mapIt = statesSetCells.find ( state );

      if ( mapIt == statesSetCells.end() )
      {
        statesSetCells[state] = std::set < EPtr > ();
        mapIt = statesSetCells.find ( state );
      }

      setCellsType setCells =  mapIt.second;

      if ( fix )
      {
        setCells.insert ( p_Cell );
      }
      else
      {
        setCells.erase ( p_Cell );
      }
    }

    bool isFixed ( EPtr p_Cell, const StatesType state )
    {
      typename mapStatedCellSetType::iterator mapIt;

      mapIt = this->statesSetCells.find ( state );

      if ( mapIt == statesSetCells.end() )
      {
        statesSetCells[state] = std::set< EPtr >();
        mapIt = statesSetCells.find ( state );
      }

      setCellsType setCells =  mapIt->second;

      return setCells.count ( p_Cell ) != 0;
    }

  private:
    friend class boost::serialization::access;
    template<class Archive> void serialize ( Archive & ar, const unsigned int version )
    {
      ar & statesSetCells;
    }
  private:
    mapStatedCellSetType statesSetCells;

};



/**
	@author
*/
template < class E, class EPtr >
class SimulatedAnnealing
{
  public:
    typedef SimulatedAnnealing Self;

    SimulatedAnnealing();

    ~SimulatedAnnealing();

    GraphCellType <  EPtr > *GetEmbryo();
    void SetEmbryo ( GraphCellType<  EPtr > *p_embryo );

    bool AllNodesGood ( const std::set<EPtr> &movingCellsSet, long actualTime );

    void MinimizeStart ( long timeStart, long timeEnd, long timeWindow, long repeatMax, double porcentage, std::ostream &os );
    void MinimizeAddingCells ( long timeStart, long timeEnd, long timeWindow, bool addCell,long withMinimization, long direction, std::ostream &os );

    void FixShortTimeCells ( long timeWindow, std::ostream &os );


    void InicializateTree ( long time );
    double InsertCell ( EPtr cell, EPtr &newCell, long direction, long timeWindow, std::ostream &os );
    bool   TryAddCell ( EPtr cell, long direction, long timeWindow, std::ostream &os );
    
    vector<double> CalulatePosition(EPtr cell,long direction,long numberSteps);
    bool ThereIsNewbourne(vector< double > position, long time);
    bool ThereIsOrphan(vector< double > position, long time);
    bool TryAddCellNoMinimization ( EPtr cell, long direction, long timeWindow, std::ostream &os );




    void InicializateEnergy ( long time );
    void Minimize ( const std::set<EPtr> &movingCellsSet, long MaxIteration, double initialProb, double finalProb , std::ostream &os );


    void NeigbourSolution ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy );
    void NewSolutionAddLink ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy );
    void NewSolutionRemoveLink ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy );
    void NewSolutionReplaceLink ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy );

    void NewSolutionAddLinkBack ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy );
    void NewSolutionRemoveLinkBack ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy );



    double Probability ( double DeltaEnergy, double temperature );
    double Temperature ( double percentage , double initialProb, double  finalProb, double  averageDeltaEnergy );

    Solution<EPtr> ActualSolution ( const std::set<EPtr> &movingCellsSet );
    bool ActualSolutionisBetterThan ( Solution < EPtr > &bestSolution, double actualEnergy );

    bool applySolution ( Solution < EPtr > &solution, const std::set<EPtr> &movingCellsSet );

    std::vector<EPtr> CellsHighestEnergy ( long time );

    void SetNumerator ( long *pNumerator )
    {
      numerator = pNumerator;
    };
    long *numerator;
    
    void SetNumeratorIncrement ( long pNumeratorIncrement )
    {
      numeratorIncrement = pNumeratorIncrement;
    };
    long numeratorIncrement;


    void SetNumberNearestCells ( long p_NumberNearestCells )
    {
      this->m_NumberNearestCells = p_NumberNearestCells;
    };
    long GetNumberNearestCells()
    {
      return this->m_NumberNearestCells;
    };
    long m_NumberNearestCells;

    void SetIterationByCell ( long p_IterationByCell )
    {
      this->m_IterationByCell = p_IterationByCell;
    };
    long GetIterationByCell()
    {
      return this->m_IterationByCell;
    };
    long m_IterationByCell;

    void SetProbabilityStart ( double p_ProbabilityStart )
    {
      this->m_ProbabilityStart = p_ProbabilityStart;
    };
    double GetProbabilityStart()
    {
      return this->m_ProbabilityStart;
    };
    double m_ProbabilityStart;

    void SetProbabilityEnd ( double p_ProbabilityEnd )
    {
      this->m_ProbabilityEnd = p_ProbabilityEnd;
    };
    double GetProbabilityEnd()
    {
      return this->m_ProbabilityEnd;
    };
    double m_ProbabilityEnd;




    void printStat ( long actualTime, std::ostream &os );

    void TestEnergy ( );
    void printInfo ( std::set< EPtr > movingCellsSet, std::ostringstream &info );

  public:
    std::set< EPtr >  virtualCellsAdded;


  private:
    EPtr PickCell ( std::set<EPtr> movingCellsSet );
    list<double> lastDeltaEnergy;


    AllowedMovesType AccionToPerform ( EPtr a_Cell );

  private:
    friend class boost::serialization::access;
    template<class Archive> void serialize ( Archive & ar, const unsigned int version )
    {
      ar & m_embryo;
    }

  private:

    GraphCellType <  EPtr > *m_embryo;
};


template< class E, class EPtr >
SimulatedAnnealing<  E,  EPtr >::SimulatedAnnealing() :  lastDeltaEnergy ( 50, 0.0 ), virtualCellsAdded ( std::set< EPtr >() ) 
{



}


template< class E, class EPtr >
SimulatedAnnealing<  E,  EPtr >::~SimulatedAnnealing()
{
}


template< class E, class EPtr >
void SimulatedAnnealing<  E,  EPtr >::InicializateEnergy ( long time )
{
  m_embryo->InicializateNeighbourEdges ( time );
}


template< class E, class EPtr >
bool SimulatedAnnealing<  E,  EPtr >::AllNodesGood ( const std::set<EPtr> &movingCellsSet, long actualTime )
{
  bool result = true;
  for ( typename std::set< EPtr >::iterator it = movingCellsSet.begin(); it != movingCellsSet.end();++it )
  {
    EPtr aCell = *it;
    if ( actualTime == aCell->GetT() )
    {
      result &= this->GetEmbryo()->GoodSuccesor ( aCell ) ;
      result &= this->GetEmbryo()->GoodPredesesor ( aCell ) ;
      if ( !result )
      {
        break;
      }
    }

  }
  return result;
}

template< class E, class EPtr >
void SimulatedAnnealing<  E,  EPtr >::Minimize ( const std::set<EPtr> &movingCellsSet,  long MaxIteration, double initialProb, double finalProb , std::ostream &os )
{

  Solution< EPtr > bestSolution;
  double actualEnergy = this->GetEmbryo()->GetEnergy ( movingCellsSet );
  double porcAnt = 0;

  long timesWithoutChangeBest = 0;
  for ( long k = 0;k < MaxIteration; ++k )
  {
//    if (AllNodesGood(movingCellsSet,actualTime))
//    {
//      break;
//    }

    EPtr a_Cell;
    a_Cell = this->PickCell ( movingCellsSet );

//    if (!this->GetEmbryo()->GoodSuccesor(a_Cell))
    {
      double m_ramdom;
      m_ramdom = ( ( double ) rand() / ( ( double ) ( RAND_MAX ) + ( double ) ( 1 ) ) );

      vector< boost::shared_ptr < CommandType > > undoSequence;
      undoSequence.resize ( 0 );

      double averageDeltaEnergy = ( double ) std::accumulate ( lastDeltaEnergy.begin(), lastDeltaEnergy.end(), 0 ) / lastDeltaEnergy.size();
      double temp = Temperature ( ( double ) k / MaxIteration, initialProb, finalProb, averageDeltaEnergy );
      AllowedMovesType accion = this->AccionToPerform ( a_Cell );

      double deltaEnergy = 0;
      undoSequence.resize ( 0 );

      switch ( accion )
      {
        case AMT_MOVE:
        {
          this->NeigbourSolution ( a_Cell, undoSequence, deltaEnergy );
          break;
        }
        case AMT_ADD:
        {
          this->NewSolutionAddLink ( a_Cell, undoSequence, deltaEnergy );
          break;
        }
        case AMT_REPLACE:
        {
          this->NewSolutionReplaceLink ( a_Cell, undoSequence, deltaEnergy );
          break;
        }
        case AMT_DELETE:
        {
          this->NewSolutionRemoveLink ( a_Cell, undoSequence, deltaEnergy );
          break;
        }
        case AMT_DELETE_BACK:
        {
          this->NewSolutionRemoveLinkBack ( a_Cell, undoSequence, deltaEnergy );
          break;
        }
        case AMT_ADD_BACK:
        {
          this->NewSolutionAddLinkBack ( a_Cell, undoSequence, deltaEnergy );
          break;
        }
        case AMT_NOTHING:
        {
          break;
        }
        default:
        {
          break;
        }
      }

      //////ANALISE IF ACCEPT THE SOLUTION
      lastDeltaEnergy.push_back ( deltaEnergy );
      lastDeltaEnergy.pop_front();

      if ( Probability ( deltaEnergy, temp ) > m_ramdom )
      {
        //Accept the new solution
        actualEnergy += deltaEnergy;
      }
      else
      {
        //Do not accept the solution
        for_each ( undoSequence.begin(), undoSequence.end(), boost::bind ( &CommandType::execute, _1 ) );
      }

      if ( ActualSolutionisBetterThan ( bestSolution, actualEnergy ) )
      {
        bestSolution = this->ActualSolution ( movingCellsSet );
        timesWithoutChangeBest = 0;
      }
      else
      {
        timesWithoutChangeBest += 1;
      }


      if ( ( bestSolution.GetEnergy() + abs ( bestSolution.GetEnergy() *0.1 ) ) <  actualEnergy )
      {
        applySolution ( bestSolution, movingCellsSet );
        actualEnergy = bestSolution.GetEnergy();
      }
      //////END ANALISE IF ACCEPT THE SOLUTION
    }

    double porc;
    porc = k * 100.0 / MaxIteration ;

    if ( porc >= ( porcAnt + 1.0 ) )
    {
      porcAnt = porc;
    }

  }
  applySolution ( bestSolution, movingCellsSet );
}

template< class E, class EPtr >
void SimulatedAnnealing<  E,  EPtr >::NewSolutionRemoveLink ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy )
{
  std::set< EPtr > nodesMoved;
  vector< boost::shared_ptr< CommandType > > doSequence;

  typename std::list< EPtr > daughtersCells;
  typename std::list< EPtr >::iterator daughtersCellsIt;

  this->m_embryo->GetSuccessor ( s_Cell, daughtersCells );

  // If the cell has daughter
  if ( daughtersCells.size() >= 1 )
  {
    int randNumber;

    randNumber =  rand() % daughtersCells.size();
    daughtersCellsIt = daughtersCells.begin();
    std::advance ( daughtersCellsIt, randNumber ) ;
    EPtr m_Cell = *daughtersCellsIt;


    boost::shared_ptr<CommandType> spDoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );

    boost::shared_ptr<CommandType> spDoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );


    doSequence.push_back ( spDoCmd1 );
    undoSequence.push_back ( spDoCmd2 );

    nodesMoved.insert ( s_Cell );

    std::list< EPtr > cellLife = this->GetEmbryo()->GetCellLife ( s_Cell );
//      std::copy(cellLife.begin(), cellLife.end(), std::inserter(nodesMoved, nodesMoved.begin()));



  }

  deltaEnergy = 0;
  deltaEnergy -= this->GetEmbryo()->GetEnergy ( nodesMoved );

  // run the command sequence
  for_each ( doSequence.begin(), doSequence.end(), boost::bind ( &CommandType::execute, _1 ) );
  deltaEnergy += this->GetEmbryo()->GetEnergy ( nodesMoved );

}


template< class E, class EPtr >
void SimulatedAnnealing<  E,  EPtr >::NewSolutionRemoveLinkBack ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy )
{
  deltaEnergy = 0;
  std::set< EPtr > nodesMoved;
  vector< boost::shared_ptr< CommandType > > doSequence;

  typename std::list< EPtr > mothersCells;
  typename std::list< EPtr >::iterator mothersCellsIt;

  this->m_embryo->GetPredecessor ( s_Cell, mothersCells );

  // If the cell has mother
  if ( mothersCells.size() >= 1 )
  {
    int randNumber;

    randNumber =  rand() % mothersCells.size();
    mothersCellsIt = mothersCells.begin();
    std::advance ( mothersCellsIt, randNumber ) ;
    EPtr m_Cell = *mothersCellsIt;


    boost::shared_ptr<CommandType> spDoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(),  m_Cell, s_Cell ) ) );

    boost::shared_ptr<CommandType> spDoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(),  m_Cell, s_Cell ) ) );


    doSequence.push_back ( spDoCmd1 );
    undoSequence.push_back ( spDoCmd2 );

    nodesMoved.insert ( s_Cell );
    nodesMoved.insert ( m_Cell );

    std::list< EPtr > cellLife = this->GetEmbryo()->GetCellLife ( s_Cell );
//      std::copy(cellLife.begin(), cellLife.end(), std::inserter(nodesMoved, nodesMoved.begin()));


    deltaEnergy -= this->GetEmbryo()->GetEnergy ( nodesMoved );

    // run the command sequence
    for_each ( doSequence.begin(), doSequence.end(), boost::bind ( &CommandType::execute, _1 ) );
    deltaEnergy += this->GetEmbryo()->GetEnergy ( nodesMoved );

  }
}


template< class E, class EPtr >
void SimulatedAnnealing<  E,  EPtr >::NewSolutionReplaceLink ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy )
{
  int randNumber;
  std::set< EPtr > nodesMoved;

  vector< boost::shared_ptr< CommandType > > doSequence;

  typename std::list< EPtr > daughtersCells;
  typename std::list< EPtr >::iterator daughtersCellsIt;

  this->m_embryo->GetSuccessor ( s_Cell, daughtersCells );

  // If the cell has daughter
  if ( daughtersCells.size() > 0 )
  {
    randNumber =  rand() % daughtersCells.size();
    //randNumber = 0;
    daughtersCellsIt = daughtersCells.begin();
    std::advance ( daughtersCellsIt, randNumber ) ;
    EPtr m_Cell = *daughtersCellsIt;

    std::list< EPtr > nearCells;
    typename std::list< EPtr >::iterator nearCellsIt;

    EmbryoStateType < EPtr > *embryoState = this->GetEmbryo()->GetEmbryoState ( m_Cell->GetT() );
    embryoState->GetNearestNCells ( m_Cell, 0,m_NumberNearestCells, nearCells );

    if ( nearCells.size() > 0 )
    {
      randNumber =  rand() % nearCells.size();
      //randNumber = 0;
      nearCellsIt = nearCells.begin();
      std::advance ( nearCellsIt, randNumber );
      EPtr n_Cell = *nearCellsIt;


      std::list< EPtr > motherCells;
      typename std::list< EPtr >::iterator motherCellsIt;
      this->m_embryo->GetPredecessor ( n_Cell, motherCells );

      if ( motherCells.size() > 0 )
      {
        //I have the 4 cells
        /*
        *    s *------------*m  ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *     v *------------*n  ;

        Switch the egdes;

        After the movementes;
        *    s *\     *m  ;
        *        \    |   ;
        *         \   |   ;
        *          \  |   ;
        *           \ |   ;
        *   v *      \*n   ;
        */

        randNumber =  rand() % motherCells.size();
        //randNumber = 0;

        motherCellsIt = motherCells.begin();
        std::advance ( motherCellsIt, randNumber );
        EPtr v_Cell = *motherCellsIt;


        if ( ( v_Cell != s_Cell ) && ( m_Cell != n_Cell ) )
        {

          {
            boost::shared_ptr<CommandType> spDoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );

            boost::shared_ptr<CommandType> spDoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), v_Cell, n_Cell ) ) );

            boost::shared_ptr<CommandType> spDoCmd3 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );


            doSequence.push_back ( spDoCmd1 );
            doSequence.push_back ( spDoCmd2 );
            doSequence.push_back ( spDoCmd3 );

            boost::shared_ptr<CommandType> spUndoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spUndoCmd3 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), v_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spUndoCmd4 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );

            undoSequence.push_back ( spUndoCmd2 );
            undoSequence.push_back ( spUndoCmd3 );
            undoSequence.push_back ( spUndoCmd4 );

            nodesMoved.insert ( s_Cell );

            std::list< EPtr > cellLifeS = this->GetEmbryo()->GetCellLife ( s_Cell );
//          std::copy(cellLifeS.begin(), cellLifeS.end(), std::inserter(nodesMoved, nodesMoved.begin()));

            nodesMoved.insert ( v_Cell );

            std::list< EPtr > cellLifeV = this->GetEmbryo()->GetCellLife ( v_Cell );
//          std::copy(cellLifeV.begin(), cellLifeV.end(), std::inserter(nodesMoved, nodesMoved.begin()));

          }
        }
      }
      else
      {
        //I have the 3 cells
        /*
        *    s *------------*m  ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *                   *n  ;

        Switch the egdes;

        After the movementes;
        *    s *------------*n  ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *                   *m   ;
        */

        if ( ( m_Cell != n_Cell ) )
        {

          {
            boost::shared_ptr<CommandType> spDoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );
            boost::shared_ptr<CommandType> spDoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );

            doSequence.push_back ( spDoCmd1 );
            doSequence.push_back ( spDoCmd2 );


            boost::shared_ptr<CommandType> spUndoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spUndoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );

            undoSequence.push_back ( spUndoCmd1 );
            undoSequence.push_back ( spUndoCmd2 );

            nodesMoved.insert ( s_Cell );

            std::list< EPtr > cellLifeS = this->GetEmbryo()->GetCellLife ( s_Cell );
//          std::copy(cellLifeS.begin(), cellLifeS.end(), std::inserter(nodesMoved, nodesMoved.begin()));

          }
        }
      }
    }

  }
  else
  {
    //I have the 3 cells
    /*
    *    s *            *m  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *   v *---------------*n ;

    Switch the egdes;

    After the movementes;
    *    s *------------*n  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *   v *             *m  ;

    */

    std::list< EPtr > nearCells;
    typename std::list< EPtr >::iterator nearCellsIt;
    EmbryoStateType < EPtr > *embryoState = this->GetEmbryo()->GetEmbryoState ( s_Cell->GetT() );
    embryoState->GetNearestNCells ( s_Cell, 0,m_NumberNearestCells, nearCells );

    if ( nearCells.size() > 0 )
    {
      randNumber =  rand() % nearCells.size();
      nearCellsIt = nearCells.begin();
      std::advance ( nearCellsIt , randNumber ) ;
      EPtr v_Cell =  *nearCellsIt;

      std::list< EPtr > daughtersCells;
      typename std::list< EPtr >::iterator daughtersCellsIt;
      this->m_embryo->GetSuccessor ( v_Cell, daughtersCells );

      if ( daughtersCells.size() > 0 )
      {
        randNumber =  rand() % daughtersCells.size();
        daughtersCellsIt = daughtersCells.begin();
        std::advance ( daughtersCellsIt, randNumber ) ;
        EPtr n_Cell = *daughtersCellsIt;


        if ( ( v_Cell != s_Cell ) )
        {


          {

            boost::shared_ptr< CommandType > spDoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), v_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spDoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );

            doSequence.push_back ( spDoCmd1 );
            doSequence.push_back ( spDoCmd2 );


            boost::shared_ptr<CommandType> spUndoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spUndoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), v_Cell, n_Cell ) ) );

            undoSequence.push_back ( spUndoCmd1 );
            undoSequence.push_back ( spUndoCmd2 );

            nodesMoved.insert ( s_Cell );

            std::list< EPtr > cellLifeS = this->GetEmbryo()->GetCellLife ( s_Cell );
//          std::copy(cellLifeS.begin(), cellLifeS.end(), std::inserter(nodesMoved, nodesMoved.begin()));

            nodesMoved.insert ( v_Cell );

            std::list< EPtr > cellLifeV = this->GetEmbryo()->GetCellLife ( v_Cell );
//          std::copy(cellLifeV.begin(), cellLifeV.end(), std::inserter(nodesMoved, nodesMoved.begin()));

          }
        }
      }
    }
  }


  deltaEnergy = 0;
  deltaEnergy -= this->GetEmbryo()->GetEnergy ( nodesMoved );

//  	for ( typename std::set<EPtr>::iterator it = nodesMoved.begin();it != nodesMoved.end(); it++ )
//  	{
//  		EPtr a = ( *it );
//  		std::cout << a->GetIdDb() << " ";
  //
//  	}
//  	std::cout << std::endl;

  // run the command sequence
  for_each ( doSequence.begin(), doSequence.end(), boost::bind ( &CommandType::execute, _1 ) );

  deltaEnergy += this->GetEmbryo()->GetEnergy ( nodesMoved );
}


template< class E, class EPtr >
void SimulatedAnnealing<  E,  EPtr >::NewSolutionAddLink ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy )
{
  deltaEnergy = 0;
  std::set< EPtr > nodesMoved;

  vector< boost::shared_ptr< CommandType > > doSequence;

  typename std::list< EPtr > daughtersCells;
  typename std::list< EPtr >::iterator daughtersCellsIt;

  this->m_embryo->GetSuccessor ( s_Cell, daughtersCells );

  // If is possible to add a link
  if ( daughtersCells.size() < 2 )
  {
    set< long > times = this->GetEmbryo()->GetTimes();
    set< long >::iterator timeIt;
    timeIt = times.find ( s_Cell->GetT() );
    if ( timeIt != times.end() )
    {
      ++timeIt;

      if ( timeIt != times.end() )
      {
        EmbryoStateType < EPtr > *embryoStateT = this->GetEmbryo()->GetEmbryoState ( s_Cell->GetT() );
        EmbryoStateType < EPtr > *embryoStateNext = this->GetEmbryo()->GetEmbryoState ( *timeIt );

        std::list< EPtr > list;
        embryoStateNext->GetNearestNCells ( s_Cell, 0,m_NumberNearestCells , list );

        for ( typename std::list< EPtr >::iterator listIt = list.begin(); listIt != list.end();++listIt )
        {
          std::list< EPtr > motherCells;
          typename std::list< EPtr >::iterator motherCellsIt;
          this->m_embryo->GetPredecessor ( *listIt, motherCells );

          if ( motherCells.size() == 0 )
          {
            boost::shared_ptr<CommandType> spDoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, *listIt ) ) );

            boost::shared_ptr<CommandType> spDoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, *listIt ) ) );


            doSequence.push_back ( spDoCmd1 );
            undoSequence.push_back ( spDoCmd2 );

            nodesMoved.insert ( s_Cell );

            std::list< EPtr > cellLife = this->GetEmbryo()->GetCellLife ( s_Cell );
//            std::copy(cellLife.begin(), cellLife.end(), std::inserter(nodesMoved, nodesMoved.begin()));


            break;
          }
        }
      }
    }
    deltaEnergy -= this->GetEmbryo()->GetEnergy ( nodesMoved );
    // run the command sequence
    for_each ( doSequence.begin(), doSequence.end(), boost::bind ( &CommandType::execute, _1 ) );
    deltaEnergy += this->GetEmbryo()->GetEnergy ( nodesMoved );
  }
}


template< class E, class EPtr >
void SimulatedAnnealing<  E,  EPtr >::NewSolutionAddLinkBack ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy )
{
  deltaEnergy = 0;
  std::set< EPtr > nodesMoved;

  vector< boost::shared_ptr< CommandType > > doSequence;

  typename std::list< EPtr > motherCells;
  typename std::list< EPtr >::iterator motherCellsIt;

  this->m_embryo->GetPredecessor ( s_Cell, motherCells );

  // If is possible to add a link
  if ( motherCells.size() < 1 )
  {
    set< long > times = this->GetEmbryo()->GetTimes();
    set< long >::iterator timeIt;
    timeIt = times.find ( s_Cell->GetT() );
    if ( timeIt != times.begin() )
    {
      --timeIt;

      if ( timeIt != times.begin() )
      {
        EmbryoStateType < EPtr > *embryoStateT = this->GetEmbryo()->GetEmbryoState ( s_Cell->GetT() );
        EmbryoStateType < EPtr > *embryoStatePrev = this->GetEmbryo()->GetEmbryoState ( *timeIt );

        std::list< EPtr > list;
        embryoStatePrev->GetNearestNCells ( s_Cell, 0, m_NumberNearestCells , list );

        for ( typename std::list< EPtr >::iterator listIt = list.begin(); listIt != list.end();++listIt )
        {
          std::list< EPtr > daughtersCells;
          typename std::list< EPtr >::iterator daughterCellsIt;
          this->m_embryo->GetSuccessor ( *listIt, daughtersCells );

          if ( daughtersCells.size() != 2 )
          {
            boost::shared_ptr<CommandType> spDoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), *listIt , s_Cell ) ) );

            boost::shared_ptr<CommandType> spDoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), *listIt, s_Cell ) ) );


            doSequence.push_back ( spDoCmd1 );
            undoSequence.push_back ( spDoCmd2 );

            nodesMoved.insert ( s_Cell );
            nodesMoved.insert ( *listIt );

            std::list< EPtr > cellLife = this->GetEmbryo()->GetCellLife ( s_Cell );
//            std::copy(cellLife.begin(), cellLife.end(), std::inserter(nodesMoved, nodesMoved.begin()));


            break;
          }
        }
      }
    }
    deltaEnergy -= this->GetEmbryo()->GetEnergy ( nodesMoved );
    // run the command sequence
    for_each ( doSequence.begin(), doSequence.end(), boost::bind ( &CommandType::execute, _1 ) );
    deltaEnergy += this->GetEmbryo()->GetEnergy ( nodesMoved );
  }
}


template< class E, class EPtr >
void SimulatedAnnealing<  E,  EPtr >::NeigbourSolution ( EPtr s_Cell, vector< boost::shared_ptr < CommandType > > &undoSequence, double &deltaEnergy )
{
  int randNumber;
  std::set< EPtr > nodesMoved;

  vector< boost::shared_ptr< CommandType > > doSequence;

  typename std::list< EPtr > daughtersCells;
  typename std::list< EPtr >::iterator daughtersCellsIt;

  this->m_embryo->GetSuccessor ( s_Cell, daughtersCells );

  // If the cell has daughter
  if ( daughtersCells.size() > 0 )
  {
    randNumber =  rand() % daughtersCells.size();
    //randNumber = 0;
    daughtersCellsIt = daughtersCells.begin();
    std::advance ( daughtersCellsIt, randNumber ) ;
    EPtr m_Cell = *daughtersCellsIt;

    std::list< EPtr > nearCells;
    typename std::list< EPtr >::iterator nearCellsIt;

    EmbryoStateType < EPtr > *embryoState = this->GetEmbryo()->GetEmbryoState ( m_Cell->GetT() );
    embryoState->GetNearestNCells ( m_Cell, 0, m_NumberNearestCells, nearCells );

    if ( nearCells.size() > 0 )
    {
      randNumber =  rand() % nearCells.size();
      //randNumber = 0;
      nearCellsIt = nearCells.begin();
      std::advance ( nearCellsIt, randNumber );
      EPtr n_Cell = *nearCellsIt;


      std::list< EPtr > motherCells;
      typename std::list< EPtr >::iterator motherCellsIt;
      this->m_embryo->GetPredecessor ( n_Cell, motherCells );

      if ( motherCells.size() > 0 )
      {
        //I have the 4 cells
        /*
        *    s *------------*m  ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *     v *------------*n  ;

        Switch the egdes;

        After the movementes;
        *    s *------------*n  ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *   v *------------*m   ;

        */

        randNumber =  rand() % motherCells.size();
        //randNumber = 0;

        motherCellsIt = motherCells.begin();
        std::advance ( motherCellsIt, randNumber );
        EPtr v_Cell = *motherCellsIt;


        if ( ( v_Cell != s_Cell ) && ( m_Cell != n_Cell ) )
        {

          {
            boost::shared_ptr<CommandType> spDoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );

            boost::shared_ptr<CommandType> spDoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), v_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spDoCmd3 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spDoCmd4 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), v_Cell, m_Cell ) ) );

            doSequence.push_back ( spDoCmd1 );
            doSequence.push_back ( spDoCmd2 );
            doSequence.push_back ( spDoCmd3 );
            doSequence.push_back ( spDoCmd4 );


            boost::shared_ptr<CommandType> spUndoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), v_Cell, m_Cell ) ) );
            boost::shared_ptr<CommandType> spUndoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spUndoCmd3 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), v_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spUndoCmd4 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );

            undoSequence.push_back ( spUndoCmd1 );
            undoSequence.push_back ( spUndoCmd2 );
            undoSequence.push_back ( spUndoCmd3 );
            undoSequence.push_back ( spUndoCmd4 );

            nodesMoved.insert ( s_Cell );

            std::list< EPtr > cellLifeS = this->GetEmbryo()->GetCellLife ( s_Cell );
//            std::copy(cellLifeS.begin(), cellLifeS.end(), std::inserter(nodesMoved, nodesMoved.begin()));

            nodesMoved.insert ( v_Cell );

            std::list< EPtr > cellLifeV = this->GetEmbryo()->GetCellLife ( v_Cell );
//            std::copy(cellLifeV.begin(), cellLifeV.end(), std::inserter(nodesMoved, nodesMoved.begin()));

          }
        }
      }
      else
      {
        //I have the 3 cells
        /*
        *    s *------------*m  ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *                   *n  ;

        Switch the egdes;

        After the movementes;
        *    s *------------*n  ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *                   |   ;
        *                   *m   ;

        */

        if ( ( m_Cell != n_Cell ) )
        {

          {
            boost::shared_ptr<CommandType> spDoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );
            boost::shared_ptr<CommandType> spDoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );

            doSequence.push_back ( spDoCmd1 );
            doSequence.push_back ( spDoCmd2 );


            boost::shared_ptr<CommandType> spUndoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spUndoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );

            undoSequence.push_back ( spUndoCmd1 );
            undoSequence.push_back ( spUndoCmd2 );

            nodesMoved.insert ( s_Cell );

            std::list< EPtr > cellLifeS = this->GetEmbryo()->GetCellLife ( s_Cell );
//            std::copy(cellLifeS.begin(), cellLifeS.end(), std::inserter(nodesMoved, nodesMoved.begin()));

          }
        }
      }
    }

  }
  else
  {
    //I have the 3 cells
    /*
    *    s *            *m  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *   v *---------------*n ;

    Switch the egdes;

    After the movementes;
    *    s *------------*n  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *    |                  ;
    *   v *             *m  ;

    */

    std::list< EPtr > nearCells;
    typename std::list< EPtr >::iterator nearCellsIt;
    EmbryoStateType < EPtr > *embryoState = this->GetEmbryo()->GetEmbryoState ( s_Cell->GetT() );
    embryoState->GetNearestNCells ( s_Cell, 0, m_NumberNearestCells, nearCells );

    if ( nearCells.size() > 0 )
    {
      randNumber =  rand() % nearCells.size();
      nearCellsIt = nearCells.begin();
      std::advance ( nearCellsIt , randNumber ) ;
      EPtr v_Cell =  *nearCellsIt;

      std::list< EPtr > daughtersCells;
      typename std::list< EPtr >::iterator daughtersCellsIt;
      this->m_embryo->GetSuccessor ( v_Cell, daughtersCells );

      if ( daughtersCells.size() > 0 )
      {
        randNumber =  rand() % daughtersCells.size();
        daughtersCellsIt = daughtersCells.begin();
        std::advance ( daughtersCellsIt, randNumber ) ;
        EPtr n_Cell = *daughtersCellsIt;


        if ( ( v_Cell != s_Cell ) )
        {


          {

            boost::shared_ptr< CommandType > spDoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), v_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spDoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );

            doSequence.push_back ( spDoCmd1 );
            doSequence.push_back ( spDoCmd2 );


            boost::shared_ptr<CommandType> spUndoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, n_Cell ) ) );
            boost::shared_ptr<CommandType> spUndoCmd2 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), v_Cell, n_Cell ) ) );

            undoSequence.push_back ( spUndoCmd1 );
            undoSequence.push_back ( spUndoCmd2 );

            nodesMoved.insert ( s_Cell );

            std::list< EPtr > cellLifeS = this->GetEmbryo()->GetCellLife ( s_Cell );
//            std::copy(cellLifeS.begin(), cellLifeS.end(), std::inserter(nodesMoved, nodesMoved.begin()));

            nodesMoved.insert ( v_Cell );

            std::list< EPtr > cellLifeV = this->GetEmbryo()->GetCellLife ( v_Cell );
//            std::copy(cellLifeV.begin(), cellLifeV.end(), std::inserter(nodesMoved, nodesMoved.begin()));

          }
        }
      }
    }
  }


  deltaEnergy = 0;
  deltaEnergy -= this->GetEmbryo()->GetEnergy ( nodesMoved );

//  	for ( typename std::set<EPtr>::iterator it = nodesMoved.begin();it != nodesMoved.end(); it++ )
//  	{
//  		EPtr a = ( *it );
//  		std::cout << a->GetIdDb() << " ";
//
//  	}
//  	std::cout << std::endl;

  // run the command sequence
  for_each ( doSequence.begin(), doSequence.end(), boost::bind ( &CommandType::execute, _1 ) );

  deltaEnergy += this->GetEmbryo()->GetEnergy ( nodesMoved );
}




template< class E, class EPtr >
double SimulatedAnnealing<  E,  EPtr >::Probability ( double DeltaEnergy, double temperature )
{
  double resu = 0;
  if ( DeltaEnergy <= 0 )
  {
    resu = 1;
  }
  else
  {
    resu =  exp ( ( -DeltaEnergy ) / temperature ) ;
  }
  return resu;
}

template< class E, class EPtr >
inline double SimulatedAnnealing<  E,  EPtr >::Temperature ( double percentage , double initialProb, double  finalProb, double averageDeltaEnergy )
{
  double probActual = percentage * ( finalProb - initialProb ) + initialProb;
  double t_inicial = -averageDeltaEnergy / log ( probActual );
  return ( t_inicial );
}



template< class E, class EPtr >
EPtr SimulatedAnnealing<  E,  EPtr >::PickCell ( std::set<EPtr> movingCellsSet )
{
  typename std::set<EPtr>::iterator it = movingCellsSet.begin();

  long randNumber =  rand() % movingCellsSet.size();

  advance ( it, randNumber );

  EPtr cell = *it;

  return cell;
}


template< class E, class EPtr >
AllowedMovesType SimulatedAnnealing<  E,  EPtr >::AccionToPerform ( EPtr a_Cell )
{
  std::set<AllowedMovesType> possibleActions;
  std::set<AllowedMovesType>::iterator possibleActionsIt;

  std::list< EPtr > p_daughters;
  this->GetEmbryo()->GetSuccessor ( a_Cell, p_daughters ) ;

  std::list< EPtr > p_mother;
  this->GetEmbryo()->GetPredecessor ( a_Cell, p_mother ) ;

  //IF THE CELL HAS NOT SUCCESSORS
  if ( p_daughters.size() == 0 )
  {
    possibleActions.insert ( AMT_ADD );
    possibleActions.insert ( AMT_REPLACE );
  }
  if ( p_daughters.size() == 1 )
  {
    possibleActions.insert ( AMT_MOVE );
    possibleActions.insert ( AMT_REPLACE );
    possibleActions.insert ( AMT_ADD );
    possibleActions.insert ( AMT_DELETE );
  }
  if ( p_daughters.size() > 1 )
  {
    possibleActions.insert ( AMT_MOVE );
    possibleActions.insert ( AMT_REPLACE );
    possibleActions.insert ( AMT_DELETE );
  }

  if ( p_mother.size() == 0 )
  {
    possibleActions.insert ( AMT_ADD_BACK );
  }
  if ( p_mother.size() == 1 )
  {
    possibleActions.insert ( AMT_DELETE_BACK );
  }


  int randNumber =  rand() % possibleActions.size();
  possibleActionsIt = possibleActions.begin();
  std::advance ( possibleActionsIt, randNumber ) ;
  return ( *possibleActionsIt );
}

template< class E, class EPtr >
inline
GraphCellType <  EPtr > *SimulatedAnnealing<  E,  EPtr >::GetEmbryo()
{
  return this->m_embryo;
}


template< class E, class EPtr >
void SimulatedAnnealing<  E,  EPtr >::SetEmbryo ( GraphCellType <  EPtr > * p_embryo )
{
  this->m_embryo = p_embryo;
}



template< class E, class EPtr >
void SimulatedAnnealing<  E,  EPtr >::TestEnergy ( )
{
  long cantCells;
  EmbryoStateType < EPtr > *embryoState = this->GetEmbryo()->GetEmbryoState ( 0 );


  typename EmbryoStateType < EPtr >::iterator itCell = embryoState->begin();

  while ( itCell != embryoState->end() )
  {
    vector< boost::shared_ptr< CommandType > > doSequence;
    vector< boost::shared_ptr < CommandType > > undoSequence;


    EPtr m_Cell, *s_Cell;
    s_Cell = *itCell;

    typename std::list< EPtr > daughtersCells;
    typename std::list< EPtr >::iterator daughtersCellsIt;
    this->m_embryo->GetDaughters ( s_Cell, daughtersCells );

    if ( daughtersCells.size() > 0 )
    {
      daughtersCellsIt = daughtersCells.begin();
      std::advance ( daughtersCellsIt, 0 ) ;
      m_Cell = *daughtersCellsIt;


      boost::shared_ptr<CommandType> spDoCmd1 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::RemoveRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );

      doSequence.push_back ( spDoCmd1 );

      boost::shared_ptr<CommandType> spUndoCmd4 ( new CommandType ( boost::bind ( &EmbryoType<EPtr>::AddRelationCell, this->GetEmbryo(), s_Cell, m_Cell ) ) );

      undoSequence.push_back ( spUndoCmd4 );

      std::set< EPtr > nodesMoved;

      nodesMoved.insert ( s_Cell );

      std::list< EPtr > cellLifeS = this->GetEmbryo()->GetCellLife ( s_Cell );
//      std::copy(cellLifeS.begin(), cellLifeS.end(), std::inserter(nodesMoved, nodesMoved.begin()));


      double EmbryoEnergy = this->GetEmbryo()->GetEmbryoEnergy ( 0 );
      double nodesEnergy = this->GetEmbryo()->GetEnergy ( nodesMoved );


      // run the command sequence
      for_each ( doSequence.begin(), doSequence.end(), boost::bind ( &CommandType::execute, _1 ) );


      double EmbryoEnergyAfterChange = this->GetEmbryo()->GetEmbryoEnergy ( 0 );
      double NodesEnergyAfterChange = this->GetEmbryo()->GetEnergy ( nodesMoved );

      for_each ( undoSequence.begin(), undoSequence.end(), boost::bind ( &CommandType::execute, _1 ) );

      double EmbryoEnergyAfterUndo = this->GetEmbryo()->GetEmbryoEnergy ( 0 );
      double NodesEnergyAfterUndo = this->GetEmbryo()->GetEnergy ( nodesMoved );

      if ( nodesEnergy == NodesEnergyAfterUndo )
      {
        if ( EmbryoEnergyAfterUndo == EmbryoEnergy )
        {
          long a = daughtersCells.size();

        }
        else
        {
          long a = daughtersCells.size();

        }
      }
      else
      {
        long a = daughtersCells.size();

      }
    }

    itCell++;
  }
}



template< class E, class EPtr >
Solution<EPtr> SimulatedAnnealing<  E,  EPtr >::ActualSolution ( const std::set< EPtr > &movingCellsSet )
{

  Solution< EPtr > result;

  double energyEmbryo = this->GetEmbryo()->GetEnergy ( movingCellsSet );
  result.SetEnergy ( energyEmbryo );

  typename set< EPtr >::iterator itBegin = movingCellsSet.begin();
  typename set< EPtr >::iterator itEnd = movingCellsSet.end();

  for ( typename std::set< EPtr >::iterator cellIt = itBegin; cellIt != itEnd ; ++cellIt )
  {
    EPtr s_Cell, m_Cell;

    s_Cell = *cellIt;

    std::list< EPtr > p_daughters;
    this->GetEmbryo()->GetSuccessor ( s_Cell, p_daughters ) ;

    typename std::list< EPtr >::iterator daughtersIt;

    for ( daughtersIt = p_daughters.begin(); daughtersIt != p_daughters.end() ; ++daughtersIt )
    {
      m_Cell = *daughtersIt ;
      result.Addlink ( s_Cell, m_Cell );
    }
  }

  return result;
}


template< class E, class EPtr >
bool SimulatedAnnealing<  E,  EPtr >::ActualSolutionisBetterThan ( Solution < EPtr > &bestSolution, double actualEnergy )
{
  double bestEnergy = bestSolution.GetEnergy();


  if ( bestEnergy <= actualEnergy )
  {
    return false;
  }
  else
  {
    return true;
  }

}


template< class E, class EPtr >
bool SimulatedAnnealing<  E,  EPtr >::applySolution ( Solution < EPtr > &solution, const std::set<EPtr> &movingCellsSet )
{

  typename set< EPtr >::iterator itBegin = movingCellsSet.begin();
  typename set< EPtr >::iterator itEnd = movingCellsSet.end();

  for ( typename std::set< EPtr >::iterator cellIt = itBegin; cellIt != itEnd ; ++cellIt )
  {
    EPtr s_Cell, m_Cell;
    s_Cell = *cellIt;

    std::list< EPtr > p_daughters;
    this->GetEmbryo()->GetSuccessor ( s_Cell, p_daughters ) ;

    typename std::list< EPtr >::iterator daughtersIt;

    for ( daughtersIt = p_daughters.begin(); daughtersIt != p_daughters.end() ; ++daughtersIt )
    {
      m_Cell = *daughtersIt ;
      this->GetEmbryo()->RemoveRelationCell ( s_Cell, m_Cell );
    }
  }

  typename Solution < EPtr >::iterator it;
  for ( it = solution.begin(); it != solution.end();++it )
  {
    this->GetEmbryo()->AddRelationCell ( it->first, it->second );
  }
}

template <typename E, typename EPtr>
void SimulatedAnnealing<  E,  EPtr >::printInfo ( std::set< EPtr > movingCellsSet, std::ostringstream &info )
{
  BOOST_FOREACH ( EPtr cell, movingCellsSet )
  {
    info  << cell->GetX() << " " << cell->GetY() << " " << cell->GetZ() << " " << cell->GetT() << std::endl;

    typename std::list< EPtr > daughtersCells;
    typename std::list< EPtr >::iterator daughtersCellsIt;

    this->m_embryo->GetSuccessor ( cell, daughtersCells );

    BOOST_FOREACH ( EPtr m_cell, daughtersCells )
    {
      info  << m_cell->GetX() << " " << m_cell->GetY() << " " << m_cell->GetZ() << " " << m_cell->GetT();

      double a_speed = sqrt ( pow ( m_cell->GetX() - cell->GetX(), 2 ) + pow ( m_cell->GetY() - cell->GetY(), 2 ) + pow ( m_cell->GetZ() - cell->GetZ(), 2 ) ) ;


      info << " " << a_speed << std::endl;
    }
    info  << "-------------" << std::endl;

  }

  info  << "+++++++++++" << std::endl;
}



template <typename E, typename EPtr>
void SimulatedAnnealing<  E,  EPtr >::MinimizeStart ( long timeStart, long timeEnd, long timeWindow, long repeatMax , double porcentage, std::ostream &os )
{
  for ( long i = timeStart ; i <  timeEnd  ;i++ )
  {
    os << "Minimizing Time: " << i << " of: " << timeEnd << std::endl;

    { 
//    long number = this->GetEmbryo()->GetNumberCells( i );
//    long numberDyingCells = this->GetEmbryo()->GetNumberDyingCells( i );
//    long numberMotherCells = this->GetEmbryo()->GetNumberMotherCells( i );
//    long numberNewbourneCells = this->GetEmbryo()->GetNumberNewbourneCells( i );
//    long numberMovingCells = this->GetEmbryo()->GetNumberMovingCells( i );
//    os << "Before Time: " << i << " Number " << number << " Dying " << numberDyingCells <<   " Mother " << numberMotherCells << " Newborn " << numberNewbourneCells << " Moving "<< numberMovingCells << std::endl;
    }

    

    long iteration = 0;
    long numberWrite = 0;

    for ( int repeat = 0;repeat < repeatMax;++repeat )
    {
      long index = 0;
      std::vector< EPtr > positionWithHighEnergy = CellsHighestEnergy ( i );
      BOOST_FOREACH ( EPtr cell, positionWithHighEnergy )
      {
        long time = cell->GetT();
        EmbryoStateType < EPtr > *embryoState = this->GetEmbryo()->GetEmbryoState ( time );
        std::set< EPtr > movingCellsList = this->GetEmbryo()->GetNearestCells ( cell, m_NumberNearestCells, cell->GetT() );
//        std::set< EPtr > movingCellsList = this->GetEmbryo()->GetNearestCells ( cell, 4, time - timeWindow  , time + timeWindow + 1);


        std::set< EPtr > movingCellsSet;
        std::copy ( movingCellsList.begin(), movingCellsList.end(), std::inserter ( movingCellsSet, movingCellsSet.begin() ) );

        double deltaEnergy = 0;
        deltaEnergy = ( -this->GetEmbryo()->GetEnergy ( movingCellsSet ) );
        this->Minimize ( movingCellsSet, movingCellsSet.size() *m_IterationByCell, m_ProbabilityStart, m_ProbabilityEnd , os );
        deltaEnergy += this->GetEmbryo()->GetEnergy ( movingCellsSet ) ;

        if ( double ( index ) / ( double ) positionWithHighEnergy.size() > porcentage )
          break;

        ++index;
        ++iteration;

      }
    }
    
    {
//      long number = this->GetEmbryo()->GetNumberCells( i );
//      long numberDyingCells = this->GetEmbryo()->GetNumberDyingCells( i );
//      long numberMotherCells = this->GetEmbryo()->GetNumberMotherCells( i );
//      long numberNewbourneCells = this->GetEmbryo()->GetNumberNewbourneCells( i );
//      long numberMovingCells = this->GetEmbryo()->GetNumberMovingCells( i );
//      os << "After  Time: " << i << " Number " << number << " Dying " << numberDyingCells  << " Mother " << numberMotherCells << " Newborn " << numberNewbourneCells << " Moving "<< numberMovingCells << std::endl;
    }

  }
}



template <typename E, typename EPtr>
    void SimulatedAnnealing<  E,  EPtr >::MinimizeAddingCells ( long timeStart, long timeEnd, long timeWindow, bool addCell, long withMinimization,long direction, std::ostream &os )
{
  long cant = 0;
  std::set< long > times = this->GetEmbryo()->GetTimes();
  if ( times.size() == 0 )
  {
    return;
  }

  os << "Start MinimizeAddingCells: Direction " << direction << std::endl;

  long cellsAdded = 0;
  BOOST_FOREACH ( long time, times )
  {
    if ( timeStart <= time && time < timeEnd )
    {
      if ( ( direction == -1 && time > * ( times.begin() ) ) || ( direction ==  1 && time < * ( times.rbegin() ) ) )
      {
        std::set< EPtr > cells;
        if ( direction == -1 )
        {
          cells = this->GetEmbryo()->GetOrphansCellsCells ( time );
        }
        else
        {
          cells = this->GetEmbryo()->GetDyingCells ( time );
        }

        os << "Start MinimizeAddingCells: Direction " << direction << "Time " << time << " Cells: " << cells.size() << std::endl;

        BOOST_FOREACH ( EPtr cell, cells )
        {
          
          if (withMinimization)
          {
            if ( TryAddCell ( cell, direction, timeWindow, os ) )
            {
              cellsAdded ++;
            }
          }
          else
          {
            if ( TryAddCellNoMinimization ( cell, direction, timeWindow, os ) )
            {
              cellsAdded ++;
            }
          }
          
        }
      }
    }
  }
os << "Cells Added Direction: " << direction << " "<< cellsAdded << std::endl;
}





template <typename E, typename EPtr>
bool SimulatedAnnealing<  E,  EPtr >::TryAddCell ( EPtr cell, long direction, long timeWindow, std::ostream &os )
{

  bool result;

  std::set< EPtr > virtualCells;

  long age = this->GetEmbryo()->GetCellLife ( cell ).size();
  long iterations = age / 5;
  bool finish = false;

  EPtr CellClone = cell;

  while ( !finish )
  {
    EPtr newCell;
    double deltaEnergy = InsertCell ( CellClone, newCell, direction, timeWindow, os );
    virtualCells.insert ( newCell );

    long numberSuccCell = this->GetEmbryo()->GetNumberSuccessor ( CellClone );
    long numberSuccNewCell = this->GetEmbryo()->GetNumberSuccessor ( newCell );
    long cellLifeCell = this->GetEmbryo()->GetCellLife ( CellClone ).size();
//    long cellLifeNewCell = this->GetEmbryo()->GetCellLife ( newCell ).size();

    if ( iterations < 0 )
    {
      std::for_each ( virtualCells.begin(),
                      virtualCells.end(),
                      boost::bind ( &GraphCellType<  EPtr >::RemoveCell, this->GetEmbryo(), _1 ) );

      finish = true;
      result = false;
    }
    else if ( numberSuccCell > 0 && numberSuccNewCell > 0 && cellLifeCell > ( virtualCells.size() + age - 1 ) )
    {
      finish = true;
      result = true;
      
      //Insert the virtual cells in the added container;
      std::copy ( virtualCells.begin(), virtualCells.end(), std::inserter ( virtualCellsAdded, virtualCellsAdded.begin() ) );
      
      os << "Cell Added: Time: " << cell->GetT() << " Direccion: " << direction << " Size Virtual Path: " << virtualCells.size() << std::endl;
    }

    CellClone = newCell;
    iterations--;
  }
  return result;
}





template <typename E, typename EPtr>
    vector<double>  SimulatedAnnealing<  E,  EPtr >::CalulatePosition(EPtr cell,long direction,long numberSteps)
{
  std::set< EPtr > cellsToStimateDireccion = this->GetEmbryo()->GetNearestCells ( cell, 4, cell->GetT());
  
  std::vector<double> directionVector ( 3, 0 );
  for ( typename std::set< EPtr >::iterator cellsToStimateDireccionIt = cellsToStimateDireccion.begin();cellsToStimateDireccionIt != cellsToStimateDireccion.end();cellsToStimateDireccionIt++ )
  {
    std::vector<double> directionVectorTMP ( 3, 0 );

    directionVectorTMP = this->GetEmbryo()->GetDirection ( *cellsToStimateDireccionIt );
    directionVector[0] += directionVectorTMP[0];
    directionVector[1] += directionVectorTMP[1];
    directionVector[2] += directionVectorTMP[2];
  }

  if ( cellsToStimateDireccion.size() != 0 )
  {
    directionVector[0] = directionVector[0] / cellsToStimateDireccion.size() * direction * numberSteps;
    directionVector[1] = directionVector[1] / cellsToStimateDireccion.size() * direction * numberSteps;
    directionVector[2] = directionVector[2] / cellsToStimateDireccion.size() * direction * numberSteps;
  }
  
  return directionVector;
}


template <typename E, typename EPtr>
    bool  SimulatedAnnealing<  E,  EPtr >::ThereIsNewbourne(vector< double > position, long time)
{
  double position3d[3] = {position[0],position[1],position[2],};
  std::set< EPtr > cells = this->GetEmbryo()->GetNearestCells ( position3d, 2, time,time);
  
  for ( typename std::set< EPtr >::iterator cellsIt = cells.begin();cellsIt != cells.end();cellsIt++ )
  {
    if (this->GetEmbryo()->IsNewBourne(*cellsIt))
    {
       return true;
    }
  }
  return false;
}

template <typename E, typename EPtr>
    bool  SimulatedAnnealing<  E,  EPtr >::ThereIsOrphan(vector< double > position, long time)
{
  double position3d[3] = {position[0],position[1],position[2],};
  std::set< EPtr > cells = this->GetEmbryo()->GetNearestCells ( position3d, 2, time,time);
  
  for ( typename std::set< EPtr >::iterator cellsIt = cells.begin();cellsIt != cells.end();cellsIt++ )
  {
    if (this->GetEmbryo()->GetNumberSuccessor (*cellsIt) == 0)
    {
      return true;
    }
  }
  return false;
}




template <typename E, typename EPtr>
    bool SimulatedAnnealing<  E,  EPtr >::TryAddCellNoMinimization ( EPtr cell, long direction, long timeWindow, std::ostream &os )
{
  bool result;
  long age = this->GetEmbryo()->GetCellLife ( cell ).size();
  long iterations = age / 5;
  bool finish = false;

  EPtr CellClone = cell;
  long numberSteps = 1;
  bool addCell = false;
  while ( !finish )
  {
    vector< double >  position = CalulatePosition(cell,direction,numberSteps);
    if (direction == 1)
    {
      addCell = ThereIsNewbourne(position,cell->GetT() + numberSteps * direction);
    }
    else
    {
      addCell = ThereIsOrphan(position,cell->GetT() + numberSteps * direction);
    }
    
    if (addCell)
    {
      break;
    }
    numberSteps++;
    if ((iterations--)<0)
    {
      finish = true;
    }
  }
    
  if (addCell == true)
  {
    return (TryAddCell ( cell, direction, 1, os ));
  }
  
  return false;
}


template <typename E, typename EPtr>
double SimulatedAnnealing<  E,  EPtr >::InsertCell ( EPtr cell, EPtr &newCell, long direction, long timeWindow, std::ostream &os )
{
  std::set< EPtr > cellsToMin = this->GetEmbryo()->GetNearestCells ( cell, 4, cell->GetT() );
  
  double deltaEnergy = 0;
  deltaEnergy = -this->GetEmbryo()->GetEnergy ( cellsToMin );

  std::set< EPtr > cellsToStimateDireccion = cellsToMin;
  std::vector<double> directionVector ( 3, 0 );
  for ( typename std::set< EPtr >::iterator cellsToStimateDireccionIt = cellsToStimateDireccion.begin();cellsToStimateDireccionIt != cellsToStimateDireccion.end();cellsToStimateDireccionIt++ )
  {
    std::vector<double> directionVectorTMP ( 3, 0 );

    directionVectorTMP = this->GetEmbryo()->GetDirection ( *cellsToStimateDireccionIt );
    directionVector[0] += directionVectorTMP[0];
    directionVector[1] += directionVectorTMP[1];
    directionVector[2] += directionVectorTMP[2];
  }

  if ( cellsToStimateDireccion.size() != 0 )
  {
    directionVector[0] = directionVector[0] / cellsToStimateDireccion.size() * direction;
    directionVector[1] = directionVector[1] / cellsToStimateDireccion.size() * direction;
    directionVector[2] = directionVector[2] / cellsToStimateDireccion.size() * direction;
  }


  EPtr newCellTMP ( new E ( cell->GetX() + directionVector[0], cell->GetY() + directionVector[1], cell->GetZ() + directionVector[2], cell->GetT() + direction, *this->numerator, *this->numerator ) );

  
  *this->numerator += this->numeratorIncrement;

  newCell = newCellTMP;
  newCell->SetVirtual(1);
  cellsToMin.insert ( newCell );
  this->GetEmbryo()->AddCell ( newCell );

  if ( direction == 1 )
  {
    this->GetEmbryo()->AddRelationCell ( cell, newCell );
  }
  else
  {
    this->GetEmbryo()->AddRelationCell ( newCell, cell );
  }

  this->Minimize ( cellsToMin,  cellsToMin.size() * m_IterationByCell, m_ProbabilityStart, m_ProbabilityEnd, os );
  deltaEnergy += this->GetEmbryo()->GetEnergy ( cellsToMin );


  return deltaEnergy;
}



template <typename E, typename EPtr>
void SimulatedAnnealing<  E,  EPtr >::FixShortTimeCells ( long timeWindow, std::ostream &os )
{
  //Collect the cells life of the complete cells (cells that have mothers and dauthers).
  std::map< long, std::set< EPtr > > mapAgeCell;
  std::multiset< long > cellAge;

  os << "FixShortTimeCells" << std::endl;

  for ( typename  EmbryoType< EPtr >::iterator it = this->GetEmbryo()->begin(); it != this->GetEmbryo()->end(); it++ )
  {
    EPtr cell = *it;

    if ( GetEmbryo()->HasMother ( cell ) && GetEmbryo()->HasDaughters ( cell ) )
    {
      long age = GetEmbryo()->GetCellLife ( cell ).size();
      cellAge.insert ( age );

      mapAgeCell[age].insert ( cell );
    }
  }



  //Calculate the age that select the x% of cells (percentile);
  double percentage = 0.40;
  long numberOfcell = cellAge.size();
  std::multiset< long >::iterator itCellAge = cellAge.begin();

  double tmp = percentage * numberOfcell;

  std::advance ( itCellAge , ( long ) tmp );
  long limitAge = *itCellAge ;

  os << "numberOfcell: " << cellAge.size() << std::endl;
  os << "limitAge: " << limitAge << std::endl;

  typename std::map< long, std::set< EPtr > >::iterator itAges;


   for ( itAges = mapAgeCell.begin(); itAges != mapAgeCell.end(); ++itAges )
   {
     std::cout << "Age: " << itAges->first << " Count: " << itAges->second.size() << std::endl;
   }


  for ( itAges = mapAgeCell.begin();itAges != mapAgeCell.end(); ++itAges )
  {
    if ( itAges->first < limitAge )
    {
      std::cout << "Age: " << itAges->first << " Count: " << itAges->second.size() << std::endl;

      typename std::set< EPtr >::iterator itCells;
      for ( itCells = itAges->second.begin();itCells != itAges->second.end();++itCells )
      {

        EPtr cellMother1, cellDautherA1, cellDautherB1;
        EPtr cellMother2, cellDautherA2, cellDautherB2;

        EPtr cell = *itCells;
        {
          std::list< EPtr > p_mother;
          this->GetEmbryo()->GetPredecessor ( cell, p_mother );
          if ( p_mother.size() > 0 )
          {
            cellMother1 = * ( p_mother.begin() );

            std::list< EPtr > p_daughters;
            GetEmbryo()->GetSuccessor ( cellMother1, p_daughters );
            if ( p_daughters.size() == 2 )
            {
              cellDautherA1 = * ( p_daughters.begin() );
              cellDautherB1 = * ( p_daughters.rbegin() );
            }
          }


          {
            std::list< EPtr > cellLife = GetEmbryo()->GetCellLife ( cell );
            cellMother2 = * ( cellLife.rbegin() );

            std::list< EPtr > p_daughters;
            GetEmbryo()->GetSuccessor ( cellMother2, p_daughters );


            if ( p_daughters.size() == 2 )
            {
              cellDautherA2 = * ( p_daughters.begin() );
              cellDautherB2 = * ( p_daughters.rbegin() );
            }
          }
          if ( cellMother1.get() && cellDautherA1.get() && cellDautherB1.get() &&
               cellMother2.get() && cellDautherA2.get() && cellDautherB2.get() )
          {
            double speedA1, speedB1;
            double speedA2, speedB2;

            speedA1 = GetEmbryo()->GetSpeed ( cellDautherA1 );
            speedB1 = GetEmbryo()->GetSpeed ( cellDautherB1 );

            speedA2 = GetEmbryo()->GetSpeed ( cellDautherA2 );
            speedB2 = GetEmbryo()->GetSpeed ( cellDautherB2 );

            EPtr cellClon;

            if ( abs ( speedA1 - speedB1 ) > abs ( speedA2 - speedB2 ) )
            {
              if ( speedA1 > speedB1 )
              {
                GetEmbryo()->RemoveRelationCell ( cellMother1, cellDautherA1 );
                cellClon = cellDautherA1;
              }
              else
              {
                GetEmbryo()->RemoveRelationCell ( cellMother1, cellDautherB1 );
                cellClon = cellDautherB1;
              }
            }
            else
            {
              if ( speedA2 > speedB2 )
              {

                GetEmbryo()->RemoveRelationCell ( cellMother2, cellDautherA2 );
                cellClon = cellDautherA2;
              }
              else
              {
                GetEmbryo()->RemoveRelationCell ( cellMother2, cellDautherB2 );
                cellClon = cellDautherB2;
              }
            }

//            std::cout << "IdMother1: " << cellMother1->GetIdDb() << " IdMother2: " << cellMother2->GetIdDb() << std::endl;
//            std::cout << "Speed A1: " << speedA1 << " SpeedB1: " << speedB1 << std::endl;
//            std::cout << "Speed A2: " << speedA2 << " SpeedB2: " << speedB2 << std::endl;

            if ( TryAddCell ( cellClon, -1, timeWindow, os ) )
            {
//              std::cout << "Cell Added" << std::endl;
            }
            else
            {
//              std::cout << "Cell Not Added" << std::endl;
            }
          }
        }
      }
    }
  }
}



template< class EPtr >
class lessEnergy: public binary_function<bool, EPtr, EPtr>
{
  public:
    lessEnergy ( GraphCellType<  EPtr > *p_embryo ) : embryo ( p_embryo ) {};

    bool operator() ( const EPtr cell1, const EPtr cell2 ) const
    {
      double cellEnergy1 = embryo->GetEnergy ( ( EPtr ) cell1 );
      double cellEnergy2 = embryo->GetEnergy ( ( EPtr ) cell2 );

      if ( cellEnergy1 < cellEnergy2 )
      {
        return true;
      }
      else
      {
        return false;
      }
    }
    GraphCellType< EPtr > *embryo;
};



template <typename E, typename EPtr>
std::vector< EPtr > SimulatedAnnealing<  E,  EPtr >::CellsHighestEnergy ( long time )
{
  std::vector< EPtr > resu ( 0 );

  try
  {
    EmbryoStateType < EPtr > *embryoState = this->GetEmbryo()->GetEmbryoState ( time );
    typename EmbryoStateType< EPtr >::iterator it;
    for ( it = embryoState->begin();it != embryoState->end();++it )
    {
      EPtr aCell = *it;
      resu.push_back ( aCell );
    }
    std::make_heap ( resu.begin(), resu.end(), lessEnergy<EPtr> ( this->GetEmbryo() ) );
  }
  catch ( ... )
  {

  }

  return resu;
}

template <typename E, typename EPtr>
void SimulatedAnnealing<  E,  EPtr >::printStat ( long actualTime, std::ostream &os )
{
  time_t rawtime;
  struct tm * timeinfo;

  os.precision ( 5 );


  os << "\t" << actualTime;

  double averageDeltaEnergy = ( double ) std::accumulate ( lastDeltaEnergy.begin(), lastDeltaEnergy.end(), 0 ) / lastDeltaEnergy.size();
  os << "\t" << averageDeltaEnergy ;

  os << "\t" << this->GetEmbryo()->GetEmbryoEnergy ( actualTime );
  os << "\t" << this->GetEmbryo()->GetNumberCells ( actualTime );
  os << "\t" << this->GetEmbryo()->GetNumberMovingCells ( actualTime );
  os << "\t" << this->GetEmbryo()->GetNumberDyingCells ( actualTime );
  os << "\t" << this->GetEmbryo()->GetNumberMotherCells ( actualTime );
  os << "\t" << this->GetEmbryo()->GetNumberNewbourneCells ( actualTime ) << " \t ";

  std::time ( &rawtime );
  timeinfo = localtime ( &rawtime );
  os << asctime ( timeinfo ) << std::flush;
}

#endif
