/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef Parallelio_h
#define Parallelio_h

#include <boost/graph/use_mpi.hpp>
#include <boost/serialization/shared_ptr.hpp>
#include <boost/serialization/list.hpp>
#include <boost/tokenizer.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/format.hpp>
#include <boost/graph/distributed/adjacency_list.hpp>
#include <boost/graph/distributed/mpi_process_group.hpp>
#include <boost/graph/iteration_macros.hpp>
#include <boost/graph/distributed/graphviz.hpp>
#include <boost/graph/distributed/distributed_graph_utility.hpp>
#include <boost/graph/distributed/mpi_process_group.hpp>
#include <boost/property_map/parallel/distributed_property_map.hpp>
#include <boost/tokenizer.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/mpi.hpp>

#include <fstream>
#include <iostream>
#include <string>


template <typename EPtr, typename E>
long ParallelIOAppendEmbryoExecute ( ParallelGraphCellType<EPtr, E> &embryo, long time, ofstream &fout );



long ParallelIOSaveHeader ( ofstream &fout )
{
  try
  {
    fout << "#id_center; " <<  "id_center_origin; " <<  "time; " <<  "id_center_mother; " <<  "x; " <<  "y; " <<  "z; " <<  "vx; " <<  "vy; " <<  "vz; " <<  "track_probability; " <<  "center_probability; " <<  std::endl;
    std::cout << "Header Saved" << std::endl;
  }
  catch ( ... )
  {
    std::cout << "Fail to save Header " << std::endl;
    return 1;
  }
  return 0;
}



//,

template < class EPtr, typename E >
long ParallelIOSaveLineageTree ( ParallelGraphCellType< EPtr, E > &embryo, long timeBegin, long timeEnd, std::ofstream &fout )
{
  /*  try
    {
      std::set< long > times = embryo.GetTimes();
      for ( std::set< long >::iterator  timeIt = times.begin();timeIt != times.end();timeIt++ )
      {
        if ( timeBegin <= ( *timeIt ) && ( *timeIt ) <= timeEnd )
        {
          if (ParallelIOAppendEmbryoExecute< EPtr > ( embryo, *timeIt,fout ))
          {
            std::cout << "Fail to save Lineage Tree: " << std::endl;
            return 1;
          }
        }
      }
    }
    catch ( ... )
    {
      std::cout << "Fail to save Lineage Tree: " << std::endl;
      return 1;
    }
    return 0;*/
}


template <typename EPtr, typename E>
long ParallelIOAppendEmbryoExecute ( ParallelGraphCellType<EPtr, E> &embryo, long time, ofstream &fout )
{
  /*  EmbryoStateType< EPtr > *embryoState;
    embryoState = embryo.GetEmbryoState ( time );

    std::vector < long > vec_id_center;
    std::vector < long > vec_id_center_origin;
    std::vector < long > vec_time;
    std::vector < long > vec_id_center_mother;
    std::vector < double > vec_x;
    std::vector < double > vec_y;
    std::vector < double > vec_z;
    std::vector < double > vec_vx;
    std::vector < double > vec_vy;
    std::vector < double > vec_vz;
    std::vector < double > vec_track_probability;
    std::vector < double > vec_center_probability;


    long i = 0;
    for ( typename EmbryoStateType<EPtr>::iterator it = embryoState->begin();it != embryoState->end();++it )
    {
      EPtr cell = *it;

      vec_id_center.push_back ( cell->GetIdDbNew() );
      vec_id_center_origin.push_back ( cell->GetIdDb() );
      vec_time.push_back ( cell->GetT() );

      std::list< EPtr > p_mother;
      embryo.GetPredecessor ( cell, p_mother );
      if ( p_mother.size() == 0 )
      {
        vec_id_center_mother.push_back ( -1 );
        vec_track_probability.push_back ( -1 );
      }
      else
      {
        EPtr cellMother = * ( p_mother.begin() );
        vec_id_center_mother.push_back ( cellMother->GetIdDbNew() );
        EdgePropertyType edgeProb = embryo.GetEdgeProperty(cellMother,cell);
        vec_track_probability.push_back ( edgeProb.GetProbability() );
      }

      vec_x.push_back ( cell->GetX() );
      vec_y.push_back ( cell->GetY() );
      vec_z.push_back ( cell->GetZ() );
      vec_vx.push_back ( cell->GetDX() );
      vec_vy.push_back ( cell->GetDY() );
      vec_vz.push_back ( cell->GetDZ() );
      vec_center_probability.push_back ( cell->GetProb() );
      i++;
    }

    try
    {
      for (long j = 0;j<i;j++)
      {
        fout << vec_id_center[i] << "; ";
        fout << vec_id_center_origin[i]<< "; ";
        fout << vec_time[i]<< "; ";
        fout << vec_id_center_mother[i]<< "; ";
        fout << vec_x[i]<< "; ";
        fout << vec_y[i]<< "; ";
        fout << vec_z[i]<< "; ";
        fout << vec_vx[i]<< "; ";
        fout << vec_vy[i]<< "; ";
        fout << vec_vz[i]<< "; ";
        fout << vec_track_probability[i]<< "; ";
        fout << vec_center_probability[i]<< "; ";
        fout << std::endl;
      }
    }
    catch ( ... )
    {
      std::cout << "Fail to save Lineage Tree " << std::endl;
      return 1;
    }
    std::cout << "Tree Uploaded, Time :"<< time << " Number:"<< i  <<  std::endl;
    return 0;*/
}


template <typename E, typename EPtr >
void ParallelreadSisterInfo ( long timeBegin, long timeEnd, ParallelGraphCellType< EPtr, E > *myEmbryo, std::string pattern )
{
  for ( int i = timeBegin;i <= timeEnd;i++ )
  {
    boost::format fmter ( pattern );
    fmter % i;
    string s  = fmter.str();
    std::cout << s  <<  std::endl;
    try
    {
      ifstream ifs ( s.c_str() );
      if ( ifs )
      {
        while ( !ifs.eof() )
        {
          int cell_NumberP, cell_NumberCh1, cell_NumberCh2;
          ifs >> cell_NumberP >> cell_NumberCh1 >> cell_NumberCh2;
          if ( cell_NumberP > 0 && cell_NumberCh1 > 0 && cell_NumberCh2 > 0 )
          {
            if ( ifs.eof() )
            {

              break;
            }
            EmbryoStateType< EPtr > *embryoState;
            embryoState = myEmbryo->GetEmbryoState ( i + 1 );
            if ( embryoState )
            {
              EPtr cell1 = embryoState->GetCellByNumber ( cell_NumberCh1 );
              std::cout << cell1  <<  std::endl;
              if ( cell1 )
              {
                cell1->SetIsNewBourne ( true );
              }
              EPtr cell2 = embryoState->GetCellByNumber ( cell_NumberCh2 );
              std::cout << cell2  <<  std::endl;
              if ( cell2 )
              {
                cell2->SetIsNewBourne ( true );
              }
            }
          }
        }
      }
    }

    catch ( ... )
    {
      std::cout << "Can not read file:"  << s.c_str() << std::endl;
    }

  }
}


template < typename EPtr, typename E >
long ParallelIOReadCenters ( long timeBegin, long timeEnd, ParallelGraphCellType< EPtr, E > *myEmbryo, std::string pattern, double spacingX, double spacingY, double spacingZ )
{
  long IDDB = 0;
  long cant = 0;

  for ( int i = timeBegin;i <= timeEnd;i++ )
  {
    boost::format fmter ( pattern );
    fmter % i;
    string s  = fmter.str();
    try
    {
      ifstream ifs ( s.c_str() );
      if ( ifs )
      {
        int cell_Number = 1;
        while ( !ifs.eof() )
        {
          int  point[3];
          ifs >> point[0] >> point[1] >> point[2];

          if ( ifs.eof() )
          {
            break;
          }
          EPtr  cell ( new E ( point[0]*spacingX, point[1]*spacingY, point[2]*spacingZ, i, cell_Number++, IDDB++ ) );
          myEmbryo->AddCell ( cell.get() );
          cant++;
        }
      }

    }
    catch ( ... )
    {
      std::cout << "Can not read file:"  << s.c_str() << std::endl;
    }
  }
  return cant;
}


template < typename EPtr, typename E >
    long ParallelIOReadCentersVTK ( boost::mpi::communicator &world,long timeBegin, long timeEnd, ParallelGraphCellType< EPtr, E > *myEmbryo, std::string pattern, long rank )
{
  long IDDB = 0;

  for ( int t = timeBegin;t <= timeEnd;t++ )
  {
    if ( rank == 0 )
    {
      boost::format fmter ( pattern );
      fmter % t;
      string s  = fmter.str();
      try
      {
        ifstream ifs ( s.c_str() );
        if ( ifs )
        {
          std::cout << "Reading file:"  << s.c_str() << std::endl;

          std::string str1, str2, str3, str4, str5;

          std::getline ( ifs, str1 );
          std::getline ( ifs, str2 );
          std::getline ( ifs, str3 );
          std::getline ( ifs, str4 );



          if ( str1.find_first_of ( "# vtk DataFile Version" ) == std::string::npos ||
               str3.find_first_of ( "ASCII" ) == std::string::npos ||
               str4.find_first_of ( "DATASET" ) == std::string::npos
             )
          {
            std::cout << "Error reading file:" << s << std::endl;

            std::cout << str1.find_first_of ( "# vtk DataFile Version" ) << std::endl;
            std::cout << str3.find_first_of ( "ASCII" ) << std::endl;
            std::cout << str4.find_first_of ( "DATASET" ) << std::endl;

            return 0;
          }

          ifs >>  str5;
          long numberPoints;

          ifs >>  numberPoints;

          ifs >>  str5;
          for ( long i = 0;i < numberPoints;++i )
          {
            double  point[3];
            ifs >> point[0] >> point[1] >> point[2];
            EPtr  cell ( new E ( point[0], point[1], point[2], t, i, IDDB++ ) );
            myEmbryo->AddCell ( cell.get() );
            if ( ifs.eof() )
            {
              break;
            }
          }
        }
        else
        {
          std::cout << "Can not read file:"  << s.c_str() << std::endl;
        }

      }
      catch ( ... )
      {
        std::cout << "Can not read file:"  << s.c_str() << std::endl;
      }
      myEmbryo->Synchronize();
    }
    else
    {
      myEmbryo->Synchronize();
    }
  }
  
  broadcast ( world, IDDB, 0 );
  
  return IDDB;
}




template < class E, class EPtr >
    long ParallelIOLoadLineageTree ( boost::mpi::communicator &world,ParallelGraphCellType< EPtr, E > &myEmbryo, std::string fileName, long rank )
{

  long linesNumber = 1;
  long MaxIDDB = 0;
  
  std::list< long > fromCell;
  std::list< long > toCell;
  
  if ( rank == 0 )
  {

    std::cout << "Loading Data from File....." << std::endl;


    std::ifstream fin ( fileName.c_str() );
    if ( fin )
    {
      std::string data;
      while ( getline ( fin, data ) )
      {
//        typedef boost::tokenizer< boost::escaped_list_separator<char> > tokenizer ;
//        tokenizer toker ( data );
        typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
        boost::char_separator<char> sep(",");
        tokenizer tokens(data, sep); 
        tokenizer::iterator iterator = tokens.begin() ;
        
        try
        {
          long id_center = boost::lexical_cast<long> ( *iterator++ );
          long id_center_mother = boost::lexical_cast<long> ( *iterator++ );
          float X = boost::lexical_cast<float> ( *iterator++ );
          float Y = boost::lexical_cast<float> ( *iterator++ );
          float Z = boost::lexical_cast<float> ( *iterator++ );
          long Time = boost::lexical_cast<long> ( *iterator++ );
          float DX = boost::lexical_cast<float> ( *iterator++ );
          float DY = boost::lexical_cast<float> ( *iterator++ );
          float DZ = boost::lexical_cast<float> ( *iterator++ );
          long Virtual = boost::lexical_cast<long> ( *iterator++ );
          long Name = boost::lexical_cast<long> ( *iterator++ );
          long CenterNumber = boost::lexical_cast<long> ( *iterator++ );
          double Deformation = boost::lexical_cast<double> ( *iterator++ );
          double Distance = boost::lexical_cast<double> ( *iterator++ );

          E cell ( X, Y, Z, Time, CenterNumber, id_center );
          cell.SetDX ( DX );
          cell.SetDY ( DY );
          cell.SetDZ ( DZ );
          cell.SetName ( Name );
          cell.SetVirtual ( Virtual );
          cell.SetDeformation ( Deformation );

          fromCell.push_back ( id_center_mother );
          toCell.push_back ( id_center );

          myEmbryo.AddCell ( &cell );
          
          MaxIDDB = (MaxIDDB>id_center)?MaxIDDB:id_center;
          
        }
        catch ( boost::bad_lexical_cast& lex )
        {
          std::cout << lex.what()<< std::endl;
          std::cout << "Lines readed: " << linesNumber << std::endl;
          
          return 0;
        }
        linesNumber++;
      }
    }
    else
    {
      std::cout << "Error: can not read File "<< fileName.c_str() << std::endl;
      return 0;
    }

    myEmbryo.Synchronize();
  }
  else
  {
    myEmbryo.Synchronize();
  }
  
  myEmbryo.Synchronize();

  
  
  //ADD The cells the cells should be local;
  broadcast ( world, fromCell, 0 );
  broadcast ( world, toCell, 0 );
  
  
  std::list< long >::iterator itFrom = fromCell.begin();
  std::list< long >::iterator itTo = toCell.begin();
  for ( ;itFrom != fromCell.end();++itFrom, ++itTo )
  {
    if (myEmbryo.IsLocal(*itFrom))
    {
      myEmbryo.AddRelationIdDb ( *itFrom, *itTo );
    }
  }

  myEmbryo.Synchronize();
  
  broadcast ( world, MaxIDDB, 0 );
  return MaxIDDB;
}


struct Point
{
  long T;
  float X;
  float Y;
  float Z;
  float DX ;
  float DY;
  float DZ;
  friend bool operator< ( const Point& x, const Point& y );
  friend class boost::serialization::access;
  template<class Archive> void serialize ( Archive & ar, const unsigned int version )
  {
    ar & T & X & Y & Z & DZ & DY & DZ;
  }
} ;

bool operator< ( const Point& x, const Point& y )
{
  if ( x.T < y.T && x.X < y.X && x.Y < y.Y && x.Z < y.Z )
    return true;
  return false;
}


template <typename E, typename EPtr >
int ParallelIOReadVectorField ( boost::mpi::communicator &world, long timeBegin, long timeEnd, ParallelGraphCellType< EPtr, E > &embryo, std::string pattern )
{
//  std::multimap<long, Point> mapTimeVector;
//  typedef std::multimap<long, Point>::iterator mapItType;
  std::map<long, long> mapCellNotFound;

  for ( int i = timeBegin;i <= timeEnd;i++ )
  {
    std::set< boost::shared_ptr< Point > > mapTimeVector;
    typedef std::set< boost::shared_ptr<Point> >::iterator mapItType;

    if ( world.rank() == 0 )
    {
      boost::format fmter ( pattern );
      fmter % i;
      string s  = fmter.str();
      try
      {
        ifstream ifs ( s.c_str(), ios::in );
        if ( ifs )
        {
          std::cout << "Reading:"  << s.c_str() << std::endl;
          while ( !ifs.eof() )
          {
            float X, Y, Z, DX, DY, DZ;
            ifs >> X >> Y >> Z >> DX >> DY >> DZ;


            boost::shared_ptr< Point > p ( new Point );

            p->T = i ;
            p->X = X ;
            p->Y = Y;
            p->Z = Z;
            p->DX = DX;
            p->DY = DY;
            p->DZ = DZ;

//          mapTimeVector.insert ( std::make_pair ( i, p ) );

            mapTimeVector.insert ( p );
          }
        }
      }
      catch ( ... )
      {
        std::cout << "Error Can not read file:"  << s.c_str() << std::endl;
        return 1;
      }
    }

    //Send the info to all the process.
    broadcast ( world, mapTimeVector, 0 );

    typedef typename ParallelGraphCellType< EPtr, E >::DistGraphType EmbryoTypeTEMP;


    BGL_FORALL_VERTICES_T ( v, embryo.graph, EmbryoTypeTEMP )
    {
      E &cell = embryo.graph[v];
      if ( cell.GetT() == i )
      {

//    std::pair<mapItType, mapItType> it_pair = mapTimeVector.equal_range ( cell.GetT() );

        bool found = false;

        float nearestVector[3] = {0, 0, 0};
        float near_dist = 100000;

        for ( mapItType it = mapTimeVector.begin(); it != mapTimeVector.end();++it )
//      for ( mapItType it = it_pair.first; it != it_pair.second;++it )
        {
          long oo = ( *it )->T;

          float p3[3];

          p3[0] = ( *it )->X;
          p3[1] = ( *it )->Y;
          p3[2] = ( *it )->Z;

          float dist = cell.GetDistance ( p3 );

          if ( dist < near_dist )
          {
            nearestVector[0] = ( *it )->DX;
            nearestVector[1] = ( *it )->DY;
            nearestVector[2] = ( *it )->DZ;

            near_dist = dist ;
          }
        }
        if ( near_dist > 5 )
        {
          if ( mapCellNotFound.find ( cell.GetT() ) == mapCellNotFound.end() )
          {
            mapCellNotFound[cell.GetT() ] = 0;
          }
          else
          {
            mapCellNotFound[cell.GetT() ] ++;
          }
//          std::cout << "Cell Not found: " << cell.GetT() << " " << cell.GetX() << " " << cell.GetY() << " " << cell.GetZ() << " " << cell.GetDX() << " " << cell.GetDY() << " " << cell.GetDZ() << std::endl ;

          cell.SetDX ( 0 );
          cell.SetDY ( 0 );
          cell.SetDZ ( 0 );
        }
        else
        {
          cell.SetDX ( nearestVector[0] );
          cell.SetDY ( nearestVector[1] );
          cell.SetDZ ( nearestVector[2] );
        }
      }
    }
  }


  for ( std::map<long, long>::iterator it = mapCellNotFound.begin();it != mapCellNotFound.end();++it )
  {
    std::cout << "Cells Not found in Vector Field files: " << it->first << " " << it->second << std::endl ;
  }
  return 0;
}




#endif
