/*  Difference of Gaussians - Difference of Gaussians Method for nucleus center detection in 3D data 
 *  Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 *  Barbara Rizzi brizzi08@gmail.com
 *  Paul Bourgine paul.bourgine@polytechnique.edu
 *  
 *  To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 *  You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
*/

//itk lib
#include "itkImage.h"

//vtk lib
#include <vtkSmartPointer.h>
#include <vtkPolyDataWriter.h>

//c++ lib
#include <time.h>
#include <unistd.h>

//custom headers
#include "centers_detection.h"
#include "timer.h"
#include "utilities.h"

using namespace std;


/**************************************** main program ******************************************/

int main( int argc, char * argv[] )
{	
  
    Parameters param( argc, argv );
    
    string fullNameNuclei = param.getFullFileNameNuclei();
    string fullNameMembranes = param.getFullFileNameMembranes();
    string fullCentersName = param.getFullFileNameCenters();
    string dirOutput = param.getTmpDir();
    double weight = param.getWeight();
    double stdSmall = param.getStdSmall();
    double stdBig = param.getStdBig();
    double threshold = param.getThreshold();
    
    cout<<"\nRunning CentersDetection....\n"<<endl; 
    cout<<"fullNameNuclei "<<fullNameNuclei<<endl;
    cout<<"fullNameMembranes "<<fullNameMembranes<<endl;
    cout<<"fullCentersName "<<fullCentersName<<endl;
    cout<<"dirOutput "<<dirOutput<<endl;
    cout<<"weight "<<weight<<endl;
    cout<<"stdSmall "<<stdSmall<<endl;
    cout<<"stdBig "<<stdBig<<endl;
    cout<<"threshold "<<threshold<<endl;

    Timer globalTimer;
    
    const unsigned int Dim = 3;
    typedef float InputPixelType;
    
    typedef itk::Image< InputPixelType,  Dim > InputImageType;
    InputImageType::Pointer image;
    Timer sectionTimer;
    
    // Reading the data, combining nuclei and membranes signal if required, and rescaling in [0,1]
    image=ReadingPreprocessingData< InputImageType >::ReadingPreprocessing(fullNameNuclei, fullNameMembranes, weight);
    printf("\n Reading Data time: %e secs\n",sectionTimer.timePassed());

    // Performing the convolution with two Gaussian having std 'stdSmall' and 'stdBig'
    // If the convolved data exist already, they are read from file and the convolution is not performed
    // Otherwise, after the convolution the data are saved in a temporary directory
    string nameStdSmall;
    string nameStdBig;
    {
	stringstream ss;
	ss << stdSmall; 
	nameStdSmall = dirOutput + "/" +  itksys::SystemTools::GetFilenameWithoutExtension(fullNameNuclei) + "_std_" + ss.str() + ".vtk";
	cout<<"nameStdSmall "<<nameStdSmall<<endl;
	ss.str(""); // reset the string to empty
	ss.clear(); // clear the state flag
	ss << stdBig;
	nameStdBig = dirOutput + "/" + itksys::SystemTools::GetFilenameWithoutExtension(fullNameNuclei) + "_std_" + ss.str() + ".vtk";
	cout<<"nameStdBig "<<nameStdBig<<endl;
    }

    // Performing the difference of the images convolved with the two Gaussians
    DifferenceOfGaussian< InputImageType > difference;
    image=difference.PerformDifferenceOfGaussian(image, stdSmall, stdBig, threshold, fexists(nameStdSmall), fexists(nameStdBig),nameStdSmall,nameStdBig);
	
    sectionTimer.reset();
    // Searching for local maxima identified as nuclei centers
    Centers< InputImageType > centers( image );
    
    vtkSmartPointer< vtkPolyData > nodes = centers.GetCenters(image);
    printf("\n Searching for local maxima then refining time: %e secs\n",sectionTimer.timePassed());
    cout<<"Saving output..."<<endl;
	 
    // Saving the centers as vtkPolyData
    sectionTimer.reset();
    vtkSmartPointer< vtkPolyDataWriter > pdWriter = vtkSmartPointer< vtkPolyDataWriter >::New();
    pdWriter->SetFileName( fullCentersName.c_str() );
    pdWriter->SetInput( nodes );	
    pdWriter->Update();

    printf("\n Saving centers time: %e secs\n",sectionTimer.timePassed());
    printf("\n CPU total time: %e secs\n",globalTimer.timePassed());
	
    return EXIT_SUCCESS;
}

