#!/bin/bash

# Geodesic Mean Curvature Flow - Geodesic Mean Curvature Flow Method for 3D image filtering 
# Written in 2015 by Karol Miikula mikula@math.sk
# Robert Cunderlik cunderli@svf.stuba.sk
# Robert Spir spir.robert@gmail.com
# Alessandro Sarti alessandro.sarti@ehess.fr
# Zuzana Kriva kriva@math.sk
# 
# To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
# You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.

PATHDESTINATION="$(O="$(cd "$(dirname "$0")" && pwd)/$(basename "$0")"; while [ ! -e "$O/.bioemergences" -a "$O" != "/" ]; do O="$(dirname "$O")"; done; echo "$O")"
BIWF_ALGOSDIR="${PATHDESTINATION}/Algorithmes"
MPI_HOME="${PATHDESTINATION}/lib/MPICH/mpich"
MPI="${MPI_HOME}/bin/mpiexec"
YOU_CAN_CLOSE="*** You can now close the window ***";

EXIT_OK=0
EXIT_INPUT_NOT_DIRECTORY=1
EXIT_NO_OUTPUT_PATH=2
EXIT_NOT_ENOUGH_MEMORY=3
EXIT_UNABLE_TO_FIND_OUTPUT=4

echo " === GMCF Filtering ==="
echo ""
echo "Starting at `date`"

algodir="${BIWF_ALGOSDIR}/Filtering/gmcfFiltering/"
source "${algodir}/Parameters.sh"
procs=$(getconf _NPROCESSORS_ONLN)   #initialize this with an automatic value based on the number of processors available
procs=$((procs - 2))
if [ $procs -lt 1 ]
then
  procs=1
fi

per_iter=$iter     #periodicity of writing, in this case, 1 write per 100 iterations


if [ -d "$1" ]; then
  inputDir="$1"
else
  echo "$1 is not a directory. "
  echo "${YOU_CAN_CLOSE}"
  exit ${EXIT_INPUT_NOT_DIRECTORY};
fi
echo "Input Directory : $inputDir"

if [ -n "$1" ]; then
  outputDir="$2"
else
  echo "You must specify an output path. "
  echo "${YOU_CAN_CLOSE}"
  exit ${EXIT_NO_OUTPUT_PATH};
fi

checkMemory=0
if [ "$3" -eq 1 ]
then
  checkMemory=1;
fi

echo "Output Directory : $outputDir"
mkdir -p "$outputDir"


#############################
# DO NOT EDIT BELOW THIS LINE
. "${BIWF_ALGOSDIR}/HelperFunctions.sh"

exe="${algodir}EXE/gmcf-par"

echo 'progression=0' > "${algodir}/.run"
ntotal=$(ls "${inputDir}"/*.vtk* | wc -l)
ncurrent=0

#run the algorithm on all files in the inputDir
for inputFile in "$inputDir"/*.vtk*
do
  baseName="$(basename "${inputFile}")"
  strippedName="${baseName%.*}"
  outputFile="${outputDir}/${strippedName}.vtk"

  if [ ! -e "${outputFile}" ];
  then
    echo "Compute  ${outputFile} "
    processVTK "$inputFile" "$outputDir"

    # Requires 19 arrays of the dataset size containing doubles (8-bytes)
    requirement=$((${n3} * ${n2} * ${n1} * 19 * 8))

    # free memory
    if [ "$(uname)" == "Darwin" ]
    then
      # Free pages
    	freeMemory=$(vm_stat | grep 'Pages free:' | grep -oE '[0-9]+')

      # 1 page = 4k
    	freeMemory=$((freeMemory * 4000))
    elif [ "$(uname)" == "Linux" ]
    then
	    freeMemory=$(free -b | head -n 2 | tail -n 1 | awk '{print $4}')
    fi

    echo "Free memory: ${freeMemory}";
    echo "Requirement: ${requirement}";

    if [ ${checkMemory} -eq 1 -a ${freeMemory} -lt ${requirement} ]
    then
      echo "ERROR: not enough memory. Your computer needs $((requirement - freeMemory)) bytes more."
      echo "${YOU_CAN_CLOSE}"
      exit ${EXIT_NOT_ENOUGH_MEMORY};
    fi

    input="${tempDir}/ $binaryData ${n3} ${n2} ${n1} ${K} ${epsilon} ${tau} ${sigma} ${iter} ${per_iter}"

    echo "inputVar: $input"
     
    echo ${input} | "${MPI}" -n ${procs} "${exe}" #logging later: $logDir/xxx-t$inputFile

    if [ $(ls -l "${tempDir}"/part_* | wc -l) -eq 0 ]
    then
      echo "Unable to find GMCF output"
      echo "${YOU_CAN_CLOSE}"
      exit ${EXIT_UNABLE_TO_FIND_OUTPUT};
    fi
        
    mv "$header" "${outputFile}"
    cat "$tempDir"/part_* >> "${outputFile}"

    rmdir "$output"
    rm -r "$tempDir"
  else
    echo "File ${outputFile} already exist "
  fi

  ncurrent=$((ncurrent + 1))
  echo "progression=$((${ncurrent} * 100 / ${ntotal}))" > "${algodir}/.run"
done

if [ -e "${algodir}/.run" ]
then
  rm -f "${algodir}/.run"
fi

echo "${YOU_CAN_CLOSE}"
exit ${EXIT_OK}
