/* GSubSurf - Generalized Subjective Surface Method for segmentation in 3D 
 * Written in 2015 by Karol Mikula mikula@math.sk
 * Mariana Remesikova remesikova@math.sk
 * Robert Spir spir.robert@gmail.com
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */

/* Program for multiscale analysis of 3D Caselles model,
semi-implicit approximation in scale
finite volume method in space
*/

#include <string.h>
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <fcntl.h>

#define  p 1
#define nnuc 20000

#define vyska 1.0

#define maxpgsi 1000
#define omega 1.2
#define tol 1.0e-6
#define pertol 2
#define odchod 1.0e-13

#define voronoi_intersections 500
#define CUBE 100
#define border 5

#define max(a,b)(((a) > (b)) ? (a) : (b))
#define min(a,b)(((a) < (b)) ? (a) : (b))


typedef  double pole[CUBE][CUBE][CUBE];

int    	skk,kk,pck,iz,s1[nnuc],s2[nnuc],s3[nnuc];
int 	i1,i2,jj1,j2,k1,k2,verbose,jad;
int	idim,jdim,kdim,ip,jp,kp,n1p,n2p,n3p;
int	mins1,maxs1,mins2,maxs2,mins3,maxs3;
int 	jj,Gi1,Gi2,Gj1,Gj2,Gk1,Gk2,kruh;

int    	ind, idw, ide, ids, idn, idb, idt;
int    	idws, idwb, idsb, idnt, idet, iden;
int	idwn, idwt, idst, idnb, ideb, ides;
int    	dwe, dsn, pp, pi;
int    	Gind, Gidw, Gide, Gids, Gidn, Gidb, Gidt;

int	n1,n2,n3,n11,embryo,nadstav,okrajXY,okrajZ,mocnina;
int	convolutionU,convolutionPMC,VTK;
double	sx,sy,sz,epsilon1,kruhz;

double  tau,sigma,h,h2,scale,r,koefconv,K,Vadv,Vcurv,koef,masa,r1r1,r2r2;
double	pom,pom1,pom2,pom3,apkonst;

double 	*u,*pmc;

static pole  uu,uprev,str1,str2,str3,ppmc;
static pole  aw,ae,as,an,ab,at,ap,b;

FILE   	*vystup,*vstup,*lfile;
char   	*name,*namezac,*namezac1,*namekoniec;
char 	*namecenters,*path1,*path2,*path3;
double 	*r1,*r2;


/****************************************************************/
double  sqr(double x)
{
  return x*x;
}            

double pm(double v)
{
     /* model perona-malik */
        return 1.0/(1.0+K*v);
     /* heat equation model */   
     /* return 1.0; */
}

double g(double v)
{
     /* model mean curvature flow */
        return 1.0/sqrt(epsilon1+v);
}

double ginv(double v)
{
     /* model mean curvature flow */
        return sqrt(epsilon1+v);
}
             
/****************************************************************/
void Writing3D()
{
 int ll,i,j,k;
 double min,max;
  
 min=max=uu[1][1][1];

 for (i=1;i<=idim;i++)
  for (j=1;j<=jdim;j++)
   for (k=1;k<=kdim;k++)
 {
 if (uu[i][j][k]<min) min=uu[i][j][k];
 else if (uu[i][j][k]>max) max=uu[i][j][k];
 }  

 if (verbose==1) printf("writing file %d\n",jj);
 sprintf(namekoniec,"_segm_%d.vtk",jj);
 strcat(name,namekoniec);
 vystup=fopen(name,"w");

 fprintf(vystup,"# vtk DataFile Version 3.0\nDistance in binary format\nBINARY\n");
 fprintf(vystup,"DATASET STRUCTURED_POINTS\nDIMENSIONS %i %i %i\n",kdim,jdim,idim);
 fprintf(vystup,"ORIGIN %lf %lf %lf\n",sx*(k1-1),sy*(jj1-1),sz*(i1-1));
 fprintf(vystup,"SPACING %lf %lf %lf\nPOINT_DATA %i\n",sx,sy,sz,idim*jdim*kdim);
 fprintf(vystup,"SCALARS scalars unsigned_char 1\nLOOKUP_TABLE default\n");
 
 for (i=1;i<=idim;i++)
  for (j=1;j<=jdim;j++)
   for (k=1;k<=kdim;k++)
	putc((unsigned char)((uu[i][j][k]-min)*255./(max-min)),vystup);

 fclose(vystup);
 strcpy(name,namezac);
}

/****************************************************************/
void ReadingImage()
{
 int ll,i,j,k,pocet;
 char auxchar[100];

  if (verbose==1) printf("reading image\n");

 //sprintf(namekoniec,".vtk");
 //strcat(path2,namekoniec);
 //printf("%s\n",path2);
 vystup=fopen(path2,"rb");

  if (VTK==1) for (i=1;i<=9;i++) 
  {
    fgets(auxchar,100,vystup);
//    printf("%s",auxchar);
  }

  for(i=p;i<=n11+p;i++)
    for(j=p;j<=n2+p;j++)
      for(k=p;k<=n3+p;k++)
        {
         ll=getc(vystup);
         if( i>=Gi1-1 && i<=min(Gi2+1,n11+p))
          if( j>=Gj1-1 && j<=Gj2+1 )
           if( k>=Gk1-1 && k<=Gk2+1 )
            {
		ip = i-Gi1+1;
		jp = j-Gj1+1;
		kp = k-Gk1+1;
		ind = ((ip*n2p+jp)*n3p+kp);
        	u[ind]=ll/255.;
	    }
        }  
    
    if((Gi2+1)>n11+p)
      {    
	pocet=0; pom=0.0;
        for(j=0;j<=jdim+1;j++)
          for(k=0;k<=kdim+1;k++)
            {
	     ind = (((n11+p-Gi1+1)*n2p+j)*n3p+k);
             pom+=u[ind];
             pocet++;
            }      
      pom = (double) pom/pocet;

      for(i=n11+p-Gi1+2;i<=idim+1;i++)
        for(j=0;j<=jdim+1;j++)
	  for(k=0;k<=kdim+1;k++)
	    {
	     ind = ((i*n2p+j)*n3p+k);
	     u[ind]=pom;
	    }
      }
  fclose(vystup);
}

/****************************************************************/
void ComputeSFRadii()
{
  int i,j;
  float average_dist=0,mind,dist;

  for (i=1;i<=jad;i++)
  {
    mind=1000;
    for (j=max(0,i-300);j<min(jad,i+300);j++) if (i!=j)
    {
       dist=sqrt(pow(s1[i]-s1[j],2)+pow(s2[i]-s2[j],2)+4*pow(s3[i]-s3[j],2));
       if (dist<mind) mind=dist;
    }
    average_dist+=mind;
  }
  average_dist/=jad;
  for (i=1;i<=jad;i++)
  {
    mind=1000;
    for (j=max(0,i-300);j<min(jad,i+300);j++) if (i!=j)
    {
       dist=sqrt(pow(s1[i]-s1[j],2)+pow(s2[i]-s2[j],2)+4*pow(s3[i]-s3[j],2));
       if (dist<mind && dist>0.5*average_dist && fabs(s3[i]-s3[j])<0.5*average_dist) mind=dist;
    }

    r1[i]=24;
    r2[i]=7;

if(embryo==1)
    {    
    if (r1[i]<=5) r1[i]=8;
    if (r1[i]<=8) r1[i]=10;
    else if (r1[i]<=12) r1[i]=11;
    else if (r1[i]<=20) r1[i]=13;
    else if (r1[i]<=30) r1[i]=15;
    else if (r1[i]<=35) r1[i]=17;
    else if (r1[i]<=40) r1[i]=21;
    else r1[i]=25;
    }
if(embryo==2)
    {
    if (r1[i]<=8) r1[i]=16;
    else if (r1[i]<=12) r1[i]=18;
    else if (r1[i]<=20) r1[i]=19;
    else if (r1[i]<=30) r1[i]=20;
    else if (r1[i]<=35) r1[i]=22;
    else if (r1[i]<=40) r1[i]=24;
    else r1[i]=25;
    }
  }
}

/****************************************************************/
void MakingSegmentationFunction()
{ 
  int i,j,k,l,c1[voronoi_intersections],c2[voronoi_intersections],c3[voronoi_intersections],nc;
  int minx, maxx, miny, maxy, minz, maxz, k11, j11, i11;
  double a,q,pomz,dist,coef[500],coef1,rad;

  for (i=0;i<CUBE;i++) for (j=0;j<CUBE;j++) for (k=0;k<CUBE;k++) uu[i][j][k]=0.;
  nc=0;
//  printf("%i %i %i\n",s1[jj-1],s2[jj-1],s3[jj-1]);
  for (i=1;i<=jad;i++)
  {
    dist=sqrt(pow(s1[jj]-s1[i],2)+pow(s2[jj]-s2[i],2)+pow(s3[jj]-s3[i],2));

    if (dist<r1[jj]+r1[i] && (dist>0.3*r1[jj] || fabs(s3[jj]-s3[i])>4.))
    {
      c1[nc]=s1[i];
      c2[nc]=s2[i];
      c3[nc]=s3[i];
      coef[nc]=(double)(r1[jj]/(r1[i]+r1[jj]));
      nc++;
    }
  }
//  printf("%i. function has intersection with %i other functions\n",jj,nc);

  a = 1./((kruh+vyska));
  q = 1./vyska;

  pom = kruh*kruh;
  pomz = kruhz*kruhz;

   for (i=1;i<=idim;i++)
    for (j=1;j<=jdim;j++)
     for (k=1;k<=kdim;k++)
       {
        pom1 = k-s1[jj]+k1;
        pom2 = j-s2[jj]+jj1;
        pom3 = i-s3[jj]+i1;
		//printf("%.3lf %.3lf %.3lf\n",pom1,pom2,pom3);
        if (( (double)(pom1*pom1)/(r1[jj]*r1[jj]) + (double)(pom2*pom2)/(r1[jj]*r1[jj]) + (double)(pom3*pom3)/(r2[jj]*r2[jj]) ) >=1.)
          uu[i][j][k] = a;
        else
        {
          /*if (nc==0)*/ uu[i][j][k]=q;
          /*else 
            for (l=0;l<nc;l++)
             {
        	coef1=sqrt((pow(s2[jj]-c2[l],2)+pow(s1[jj]-c1[l],2))/(pow(s3[jj]-c3[l],2)+pow(s2[jj]-c2[l],2)+pow(s1[jj]-c1[l],2)));
        	coef1=coef1*(coef[l]-0.5)+0.5;

    		pom1 = k+k1-Gk1;
    		pom2 = j+jj1-Gj1;
    		pom3 = i+i1-Gi1;

        	if ((1-coef1)*(pow(pom1-c1[l],2)+pow(pom2-c2[l],2)+pow(pom3-c3[l],2))>coef1*(pow(pom1-s1[jj],2)+pow(pom2-s2[jj],2)+pow(pom3-s3[jj],2)))
            	    uu[i][j][k]=q;
        	else
        	  {
            	    rad=(1.-coef[l])*r1[jj]/coef[l];
            	    if ((double)((pom1-c1[l])*(pom1-c1[l]))/(rad*rad)+(double)((pom2-c2[l])*(pom2-c2[l]))/(rad*rad)+(double)((pom3-c3[l])*(pom3-c3[l]))/(pomz)<=1.)
            	     {
                	uu[i][j][k]=a;
                	break;
            	     }
            	    else 
            		uu[i][j][k]=q;
        	  }
             }*/
          }
        }

  /*minx=n1+p;
  maxx=0;
  miny=n2+p;
  maxy=0;
  minz=n3+p;
  maxz=0;
   for (i=1;i<=idim;i++)
    for (j=1;j<=jdim;j++)
     for (k=1;k<=kdim;k++)
      {
        uu[i][j][k]=uu[i][j][k]-a;
        if (uu[i][j][k]>=1.0e-8)
        {
          if (k<minx) minx=k; else if (k>maxx) maxx=k;
          if (j<miny) miny=j; else if (j>maxy) maxy=j;
          if (i<minz) minz=i; else if (i>maxz) maxz=i;
        }
      }
      
  k11=max(k1+minx-border,k1); 
  k2=min(k1+maxx+border,k2);
  k1=k11;
  j11=max(jj1+miny-border,jj1); 
  j2=min(jj1+maxy+border,j2);
  jj1=j11;
  i11=max(i1+minz-border,i1); 
  i2=min(i1+maxz+border,i2);
  i1=i11;*/

  /*idim = i2-i1+1;
  jdim = j2-jj1+1;
  kdim = k2-k1+1;*/
  printf("makingsegmf: %d %d %d\n",idim,jdim,kdim);
  /*for (i=1;i<=idim;i++)
    for (j=1;j<=jdim;j++)
     for (k=1;k<=kdim;k++)
          uu[i][j][k]=uu[i+max(minz-border,0)][j+max(miny-border,0)][k+max(minx-border,0)];*/
printf("makingsegmf: %d %d %d\n",idim,jdim,kdim);
}





/****************************************************************/
void LinearSystemSolvingConvolution(void)
{
 int  i,j,k,pgsi=0;
 double delta=1.0e+10,deltain=0.,z;

 if (verbose==1) printf("solving system %d\n",kk); 

 for (i=1;i<=idim;i++)
  for (j=1;j<=jdim;j++)
   for (k=1;k<=kdim;k++)
    {
    ind = ((i*n2p+j)*n3p+k);
    idw = ind - dwe;
    ide = ind + dwe;
    ids = ind - dsn;
    idn = ind + dsn;
    idb = ind - 1;
    idt = ind + 1;

    pom = koefconv*(u[idw]+u[ide]+u[ids]+u[idn]+u[idb]+u[idt]);

    deltain=deltain+sqr(apkonst*u[ind]-pom-pmc[ind]);
    }
 if (verbose==1) printf("%d: deltain=%30.25lf\n",pgsi,deltain); 

 do
  {
   pgsi=pgsi+1;
   for (i=1;i<=idim;i++)
    for (j=1;j<=jdim;j++)
     for (k=1;k<=kdim;k++)
     {
     ind = ((i*n2p+j)*n3p+k); 
     idw = ind - dwe; 
     ide = ind + dwe;
     ids = ind - dsn;
     idn = ind + dsn;
     idb = ind - 1;
     idt = ind + 1;

     pom = koefconv*(u[idw]+u[ide]+u[ids]+u[idn]+u[idb]+u[idt]);

     z=(pmc[ind]+pom)/apkonst;
     u[ind]=u[ind]+omega*(z-u[ind]);
     }

   if (pgsi % pertol==0)
     {
       delta=0.;
       for (i=1;i<=idim;i++)
        for (j=1;j<=jdim;j++)
         for (k=1;k<=kdim;k++)
	  {
            ind = ((i*n2p+j)*n3p+k); 
    	    idw = ind - dwe; 
    	    ide = ind + dwe;
    	    ids = ind - dsn;
    	    idn = ind + dsn;
    	    idb = ind - 1;
    	    idt = ind + 1;

    	    pom = koefconv*(u[idw]+u[ide]+u[ids]+u[idn]+u[idb]+u[idt]);

    	    delta=delta+sqr(apkonst*u[ind]-pom-pmc[ind]);
	  }
      printf("%d: delta=%30.25lf\n",pgsi,delta);
     }
  } 
  while ((delta>tol*deltain) && (pgsi<maxpgsi));
  if (verbose==1) printf("%d:\tdelta: %30.25lf\n",pgsi,delta);

  if (verbose==1) printf("%d system solved with %d gs iterations\n",kk,pgsi);
} 


/****************************************************************/
void LinearSystemSolving(void)
{
 int  i,j,k,pgsi=0;
 double delta=1.0e+10,deltain=0.,z;

 if (verbose==1) printf("solving system %d\n",kk); 

 for (i=1;i<=idim;i++)
  for (j=1;j<=jdim;j++)
   for (k=1;k<=kdim;k++)
    {
    pom = (-aw[i][j][k]*uu[i-1][j][k]-ae[i][j][k]*uu[i+1][j][k]-
            as[i][j][k]*uu[i][j-1][k]-an[i][j][k]*uu[i][j+1][k]-
            ab[i][j][k]*uu[i][j][k-1]-at[i][j][k]*uu[i][j][k+1]+
            ap[i][j][k]*uu[i][j][k]-b[i][j][k]);

    deltain = deltain+pom*pom;
    }
 if (verbose==1) printf("%d: deltain=%30.25lf\n",pgsi,deltain); 

 do
  {
   pgsi=pgsi+1;
   for (i=1;i<=idim;i++)
    for (j=1;j<=jdim;j++)
     for (k=1;k<=kdim;k++)
     { 
     z=(b[i][j][k]+aw[i][j][k]*uu[i-1][j][k]+ae[i][j][k]*uu[i+1][j][k]+
        as[i][j][k]*uu[i][j-1][k]+an[i][j][k]*uu[i][j+1][k]+
        ab[i][j][k]*uu[i][j][k-1]+at[i][j][k]*uu[i][j][k+1])/ap[i][j][k];

     uu[i][j][k]=uu[i][j][k]+omega*(z-uu[i][j][k]);
     }

   if (pgsi % pertol==0)
     {
       delta=0.;
       for (i=1;i<=idim;i++)
       for (j=1;j<=jdim;j++)
       for (k=1;k<=kdim;k++)
       {
       pom = (-aw[i][j][k]*uu[i-1][j][k]-ae[i][j][k]*uu[i+1][j][k]-
               as[i][j][k]*uu[i][j-1][k]-an[i][j][k]*uu[i][j+1][k]-
               ab[i][j][k]*uu[i][j][k-1]-at[i][j][k]*uu[i][j][k+1]+
               ap[i][j][k]*uu[i][j][k]-b[i][j][k]);

       delta = delta+pom*pom;
       }
//      printf("%d: delta=%30.25lf\n",pgsi,delta);
     }
  } 
  while ((delta>tol*deltain) && (pgsi<maxpgsi));
  if (verbose==1) printf("%d:\tdelta: %30.25lf\n",pgsi,delta);

  if (verbose==1) printf("%d system solved with %d gs iterations\n",kk,pgsi);
} 

/****************************************************************/
void Coefficients0(void)
{
 int  i,j,k;
 double z,zw,ze,zs,zn,zb,zt;

 /* reflexions */
 for (i=1;i<=idim;i++)
  for (j=1;j<=jdim;j++)
  {
    pi=(i*n2p+j)*n3p;
    u[pi]=u[pi+2];
    u[pi+kdim+1]=u[pi+kdim-1];
  }

 for (j=1;j<=jdim;j++)
  for (k=0;k<=kdim+1;k++)
  {
    pi=j*n3p+k;
    u[pi]=u[pi+2*dwe];
    pi=((idim+1)*n2p+j)*n3p+k;
    u[pi]=u[pi-2*dwe];
  }

 for (i=0;i<=idim+1;i++)
  for (k=0;k<=kdim+1;k++)
  {
    pi=i*n2p*n3p+k;
    u[pi]=u[pi+2*n3p];
    pi+=(jdim+1)*n3p;
    u[pi]=u[pi-2*n3p];
  }

 if (convolutionU==1)
 {
 for (i=1;i<=idim;i++)
  for (j=1;j<=jdim;j++)
   for (k=1;k<=kdim;k++)
     {
      ind = ((i*n2p+j)*n3p+k);
      pmc[ind]=u[ind];
     }

  LinearSystemSolvingConvolution();
 }    

 for (i=1;i<=idim;i++)
 for (j=1;j<=jdim;j++)
 for (k=1;k<=kdim;k++)
 {
    ind = ((i*n2p+j)*n3p+k);
    idw = ind - dwe;
    ide = ind + dwe;
    ids = ind - dsn;
    idn = ind + dsn;
    idb = ind - 1;
    idt = ind + 1;

    idws = idw - dsn;
    idwb = idw - 1;
    idsb = ids - 1;

    idnt = idn + 1;
    idet = ide + 1;
    iden = ide + dsn;
    
    idwn = idw + dsn;
    idwt = idw + 1;
    ides = ide - dsn;
    ideb = ide - 1;
    idst = ids + 1;
    idnb = idn - 1;
    
    pom1 = (u[ind]-u[idw]);
    pom2 = 0.25*(u[idwn]+u[idn]-u[idws]-u[ids]);
    pom3 = 0.25*(u[idwt]+u[idt]-u[idwb]-u[idb]);

    zw = pom1*pom1+pom2*pom2+pom3*pom3;

    pom1 = (u[ide]-u[ind]);
    pom2 = 0.25*(u[idn]+u[iden]-u[ids]-u[ides]);
    pom3 = 0.25*(u[idt]+u[idet]-u[idb]-u[ideb]);
    
    ze = pom1*pom1+pom2*pom2+pom3*pom3;

    pom1 = (u[ind]-u[ids]);
    pom2 = 0.25*(u[ide]+u[ides]-u[idw]-u[idws]);
    pom3 = 0.25*(u[idt]+u[idst]-u[idb]-u[idsb]);

    zs = pom1*pom1+pom2*pom2+pom3*pom3;

    pom1 = (u[idn]-u[ind]);
    pom2 = 0.25*(u[iden]+u[ide]-u[idwn]-u[idw]);
    pom3 = 0.25*(u[idnt]+u[idt]-u[idnb]-u[idb]);

    zn = pom1*pom1+pom2*pom2+pom3*pom3;

    pom1 = (u[ind]-u[idb]);
    pom2 = 0.25*(u[ide]+u[ideb]-u[idw]-u[idwb]);
    pom3 = 0.25*(u[idn]+u[idnb]-u[ids]-u[idsb]);

    zb = pom1*pom1+pom2*pom2+pom3*pom3;

    pom1 = (u[idt]-u[ind]);
    pom2 = 0.25*(u[idet]+u[ide]-u[idwt]-u[idw]);
    pom3 = 0.25*(u[idnt]+u[idn]-u[idst]-u[ids]);

    zt = pom1*pom1+pom2*pom2+pom3*pom3;

    z=(zw+ze+zs+zn+zb+zt)/6.;
                            
    pmc[ind]=pm(z);
 }

 /* reflexions */

 for (i=1;i<=idim;i++)
  for (j=1;j<=jdim;j++)
  {
    pi=(i*n2p+j)*n3p;
    pmc[pi]=pmc[pi+2];
    pmc[pi+kdim+1]=pmc[pi+kdim-1];
  }

 for (j=1;j<=jdim;j++)
  for (k=0;k<=kdim+1;k++)
  {
    pi=j*n3p+k;
    pmc[pi]=pmc[pi+2*dwe];
    pi=((idim+1)*n2p+j)*n3p+k;
    pmc[pi]=pmc[pi-2*dwe];
  }

 for (i=0;i<=idim+1;i++)
  for (k=0;k<=kdim+1;k++)
  {
    pi=i*n2p*n3p+k;
    pmc[pi]=pmc[pi+2*n3p];
    pi+=(jdim+1)*n3p;
    pmc[pi]=pmc[pi-2*n3p];
  }

 if (convolutionPMC==1)
 {
 for (i=0;i<=idim+1;i++)
  for (j=0;j<=jdim+1;j++)
   for (k=0;k<=kdim+1;k++)
    { 
     ind = ((i*n2p+j)*n3p+k);
     u[ind]=pmc[ind];
    }
 
 LinearSystemSolvingConvolution();

 for (i=1;i<=idim;i++)
  for (j=1;j<=jdim;j++)
   for (k=1;k<=kdim;k++)
     {
       ind = ((i*n2p+j)*n3p+k);
       pmc[ind]=u[ind];
     }
 } 
}

/****************************************************************/
void Rescalepmc()
{
  double min, max, pmcv;
  int i,j,k,pocet;

  min=max=pmc[(n2p+1)*n3p+1];
  pocet = 0;

 for (i=1;i<=idim;i++)
 for (j=1;j<=jdim;j++)
 for (k=1;k<=kdim;k++)
  {
    pocet++;
    Gind = (((i+i1-Gi1)*n2p+(j+jj1-Gj1))*n3p+(k+k1-Gk1));

    ppmc[i][j][k]=pmc[Gind];
    
    if (ppmc[i][j][k]<min) min=ppmc[i][j][k];
    if (ppmc[i][j][k]>max) max=ppmc[i][j][k];
  }

// printf("\npmc-min: %.4lf\tpmc-max: %.4lf\n\n",min,max);

 for (i=1;i<=idim;i++)
 for (j=1;j<=jdim;j++)
 for (k=1;k<=kdim;k++)
  {
    pmcv=(ppmc[i][j][k]-min)/(max-min);
    ppmc[i][j][k]=pow(pmcv,mocnina);
  }
} 

/****************************************************************/
void Coefficients(void)
{
 int  i,j,k;
 double z,gradw,grade,grads,gradn,gradp,gradb,gradt;
 double cinw,cine,cins,cinn,cinb,cint,prod;
   
 if (verbose==1) printf("computing coefficients %d\n",kk);
 /* computing of coefficients */
 
 for (i=1;i<=idim+1;i++)
 for (j=1;j<=jdim+1;j++)
 for (k=1;k<=kdim+1;k++)
 {
  str1[i][j][k] = 0.25*(uu[i-1][j-1][k]+uu[i][j-1][k]+uu[i][j][k]+uu[i-1][j][k]);
  str2[i][j][k] = 0.25*(uu[i-1][j][k-1]+uu[i][j][k-1]+uu[i][j][k]+uu[i-1][j][k]);
  str3[i][j][k] = 0.25*(uu[i][j-1][k-1]+uu[i][j][k-1]+uu[i][j][k]+uu[i][j-1][k]);
 }

 for (i=1;i<=idim;i++)
 for (j=1;j<=jdim;j++)
 for (k=1;k<=kdim;k++)
 {
    pom1 = (uu[i][j][k]-uu[i-1][j][k]);
    pom2 = (str1[i][j+1][k]-str1[i][j][k]);
    pom3 = (str2[i][j][k+1]-str2[i][j][k]);

    gradw = pom1*pom1+pom2*pom2+pom3*pom3;

    pom1 = (uu[i+1][j][k]-uu[i][j][k]);
    pom2 = (str1[i+1][j+1][k]-str1[i+1][j][k]);
    pom3 = (str2[i+1][j][k+1]-str2[i+1][j][k]);

    grade = pom1*pom1+pom2*pom2+pom3*pom3;

    pom1 = (uu[i][j][k]-uu[i][j-1][k]);
    pom2 = (str1[i+1][j][k]-str1[i][j][k]);
    pom3 = (str3[i][j][k+1]-str3[i][j][k]);

    grads = pom1*pom1+pom2*pom2+pom3*pom3;

    pom1 = (uu[i][j+1][k]-uu[i][j][k]);
    pom2 = (str1[i+1][j+1][k]-str1[i][j+1][k]);
    pom3 = (str3[i][j+1][k+1]-str3[i][j+1][k]);

    gradn = pom1*pom1+pom2*pom2+pom3*pom3;

    pom1 = (uu[i][j][k]-uu[i][j][k-1]);
    pom2 = (str2[i+1][j][k]-str2[i][j][k]);
    pom3 = (str3[i][j+1][k]-str3[i][j][k]);

    gradb = pom1*pom1+pom2*pom2+pom3*pom3;

    pom1 = (uu[i][j][k+1]-uu[i][j][k]);
    pom2 = (str2[i+1][j][k+1]-str2[i][j][k+1]);
    pom3 = (str3[i][j+1][k+1]-str3[i][j][k+1]);

    gradt = pom1*pom1+pom2*pom2+pom3*pom3;

   //gradp=(gradw+grade+grads+gradn+gradb+gradt)/6.;
    z=ginv((gradw+grade+grads+gradn+gradb+gradt)/6.);

    prod = Vcurv*koef*ppmc[i][j][k]*z;                                                                                                            
    aw[i][j][k] = prod*g(gradw);
    ae[i][j][k] = prod*g(grade);
    as[i][j][k] = prod*g(grads);
    an[i][j][k] = prod*g(gradn);
    ab[i][j][k] = prod*g(gradb);
    at[i][j][k] = prod*g(gradt);
                                                                                                                                                                                                                                                       

    ap[i][j][k] = 1.0+aw[i][j][k]+ae[i][j][k]+as[i][j][k]+
                      an[i][j][k]+ab[i][j][k]+at[i][j][k];

/*   cinw=min(0.,Vadv*(ppmc[i][j][k]-ppmc[i-1][j][k])); 
   cine=min(0.,-Vadv*(ppmc[i+1][j][k]-ppmc[i][j][k]));
   cins=min(0.,Vadv*(ppmc[i][j][k]-ppmc[i][j-1][k])); 
   cinn=min(0.,-Vadv*(ppmc[i][j+1][k]-ppmc[i][j][k]));
   cinb=min(0.,Vadv*(ppmc[i][j][k]-ppmc[i][j][k-1])); 
   cint=min(0.,-Vadv*(ppmc[i][j][k+1]-ppmc[i][j][k]));
                                                   
   b[ind]=uprev[i][j][k]+koef*(
              cinw*(uu[i][j][k]-uu[i-1][j][k])+
              cine*(uu[i][j][k]-uu[i+1][j][k])+
              cins*(uu[i][j][k]-uu[i][j-1][k])+
              cinn*(uu[i][j][k]-uu[i][j+1][k])+
              cinb*(uu[i][j][k]-uu[i][j][k-1])+ 
              cint*(uu[i][j][k]-uu[i][j][k+1]));*/
  
                                                                                                                                                                 
   b[i][j][k]=uprev[i][j][k] - Vadv*tau*(
              max(0.,-0.5*(ppmc[i+1][j][k]-ppmc[i-1][j][k]))*(uu[i][j][k]-uu[i-1][j][k])+
              min(0.,-0.5*(ppmc[i+1][j][k]-ppmc[i-1][j][k]))*(uu[i+1][j][k]-uu[i][j][k])+
              max(0.,-0.5*(ppmc[i][j+1][k]-ppmc[i][j-1][k]))*(uu[i][j][k]-uu[i][j-1][k])+
              min(0.,-0.5*(ppmc[i][j+1][k]-ppmc[i][j-1][k]))*(uu[i][j+1][k]-uu[i][j][k])+
              max(0.,-0.5*(ppmc[i][j][k+1]-ppmc[i][j][k-1]))*(uu[i][j][k]-uu[i][j][k-1])+
              min(0.,-0.5*(ppmc[i][j][k+1]-ppmc[i][j][k-1]))*(uu[i][j][k+1]-uu[i][j][k]));    

 }
}
                                                                                                                                                                                                                
/****************************************************************/
double EllipticStep(void)
{ 
  int  i,j,k;
  double cput1,cput2,delta=0.,max,min;
  
  masa=0.;

 for (i=1;i<=idim;i++)
  for (j=1;j<=jdim;j++)
   for (k=1;k<=kdim;k++)
     uprev[i][j][k]=uu[i][j][k];

  cput1=clock()/(double)(CLOCKS_PER_SEC);
  Coefficients(); 
  cput2=clock()/(double)(CLOCKS_PER_SEC);
  if (verbose==1) printf("CPU time coeff: %e secs\n",cput2-cput1);
     
  cput1=clock()/(double)(CLOCKS_PER_SEC);
  LinearSystemSolving();
  cput2=clock()/(double)(CLOCKS_PER_SEC);
  if (verbose==1) printf("CPU time system: %e secs\n",cput2-cput1);

 for (i=1;i<=idim;i++)
  for (j=1;j<=jdim;j++)
   for (k=1;k<=kdim;k++)
     {
      delta=delta+sqr(uu[i][j][k]-uprev[i][j][k]);
      masa=masa+uu[i][j][k];
     }
       
 delta=sqrt(delta*h*h*h);
 if (verbose==1) printf("L_2 norma odchyliek iteracii = %e, masa riesenia = %e\n",delta,masa);
 return delta;
}

/****************************************************************/
int main(void)
{
  int 	i,j,k,kruhi,kruhj,kruhk;
  double cput_start, cput_end, cput1,cput2,zmena;

  name=(char *)malloc(200);
  namezac=(char *)malloc(200);
  namezac1=(char *)malloc(200);
  namekoniec=(char *)malloc(50);
  path1=(char *)malloc(200);
  path2=(char *)malloc(200);
  path3=(char *)malloc(200);


  scanf("%s %s %s",path1,path2,path3);
  scanf("%d %d %d",&n11,&n2,&n3);
  scanf("%d %d %d %d",&embryo,&nadstav,&okrajXY,&okrajZ);
  scanf("%lf %lf %lf %lf %d",&sx,&sy,&sz,&epsilon1,&mocnina);
  scanf("%d %d %d",&convolutionU,&convolutionPMC,&VTK);
  scanf("%lf %lf %lf %lf %lf %d %d %d",&K,&tau,&sigma,&Vadv,&Vcurv,&pck,&iz,&verbose);  
  
  printf("****** SEGMENTATION OF MEMBRANES *******\n%s\n%s\n%s\nK=%lf\ttau=%lf\tsigma=%lf\n",path1,path2,path3,K,tau,sigma);
  
  strcpy(namezac,path1);
  strcpy(name,namezac);

  n11 -= 1;
  n2 -= 1;
  n3 -= 1;
  n1 = n11 + nadstav;

  h=1.0;
  h2=h*h;
  r=h*h/tau;
  koef=tau/h2;
  koefconv=sigma/(h*h);
  apkonst=(1.0+6*koefconv);
    
  scale=0.0;
  skk=0;

  vstup=fopen(path3,"r"); 
  jad=0;
  while (!feof(vstup))
    {
    jad++;
    fscanf(vstup,"%d %d %d",&s1[jad],&s2[jad],&s3[jad]);
    }
	
  fclose(vstup);

  r1=(double *)malloc((jad+1)*sizeof(double));
  r2=(double *)malloc((jad+1)*sizeof(double));

  ComputeSFRadii();

  maxs1=mins1=s1[1];
  maxs2=mins2=s2[1];
  maxs3=mins3=s3[1];

  for(i=1;i<=jad;i++)
  {
    if (s1[i]<mins1) mins1=s1[i];
    if (s1[i]>maxs1) maxs1=s1[i];
    if (s2[i]<mins2) mins2=s2[i];
    if (s2[i]>maxs2) maxs2=s2[i];
    if (s3[i]<mins3) mins3=s3[i];
    if (s3[i]>maxs3) maxs3=s3[i];
  }
  
  Gk1 = max(p,mins1-okrajXY); 
  Gk2 = min(n3+p,maxs1+okrajXY); 
  Gj1 = max(p,mins2-okrajXY); 
  Gj2 = min(n2+p,maxs2+okrajXY); 
  Gi1 = max(p,mins3-okrajZ); 
  Gi2 = min(n1+p,maxs3+okrajZ); 

  printf("\nx:\t%d (%d) - (%d) %d\ny:\t%d (%d) - (%d) %d\nz:\t%d (%d) - (%d) %d\n\n",
			    Gk1,mins1,maxs1,Gk2,Gj1,mins2,maxs2,Gj2,Gi1,mins3,maxs3,Gi2);

  idim = Gi2-Gi1+1;
  jdim = Gj2-Gj1+1;
  kdim = Gk2-Gk1+1;
printf("%d %d %d\n",idim,jdim,kdim);
  n1p = idim + 2;
  n2p = jdim + 2;
  n3p = kdim + 2;

  dwe = n2p*n3p;
  dsn = n3p;

  u = (double *) malloc(n1p * n2p * n3p * sizeof(double));
  pmc = (double *) malloc(n1p * n2p * n3p * sizeof(double));

 for (i=0;i<=idim+1;i++)
  for (j=0;j<=jdim+1;j++)
   for (k=0;k<=kdim+1;k++)
     {
      ind = ((i*n2p+j)*n3p+k);
      u[ind]=0.; pmc[ind]=0.;
     }

  ReadingImage();
  Coefficients0();

  cput_start=clock()/(double)(CLOCKS_PER_SEC);

  free(u);

 for (i=0;i<CUBE;i++)
  for (j=0;j<CUBE;j++)
   for (k=0;k<CUBE;k++)
     {
      uu[i][j][k]=0.;ppmc[i][j][k]=0.0;
      aw[i][j][k]=0.;ae[i][j][k]=0.;as[i][j][k]=0.;an[i][j][k]=0.;
      ab[i][j][k]=0.;at[i][j][k]=0.;
      ap[i][j][k]=0.;b[i][j][k]=0.;
      uprev[i][j][k]=0.;str1[i][j][k]=0.;
      str2[i][j][k]=0.;str3[i][j][k]=0.;
     }

  for (jj=1;jj<=jad;jj++)
  {
   kruh = r1[jj];
   kruhz = r2[jj];

   if (kruh==0) continue;
   if (verbose==0) printf("\n\njj=%d, s1=%d, s2=%d, s3=%d, kruh=%d  kruhz=%lf\n",jj,s1[jj],s2[jj],s3[jj],kruh,kruhz);
   printf("%i. cell has segmentation radii %lf %lf\n",jj,r1[jj],r2[jj]);

   kruhk=(int)(r1[jj]+border);
   kruhj=(int)(r1[jj]+border);
   kruhi=(int)(r2[jj]+border);

   kk=skk;
   
   k1=max(s1[jj]-kruhk,Gk1);
   k2=min(s1[jj]+kruhk,Gk2);
   jj1=max(s2[jj]-kruhj,Gj1);
   j2=min(s2[jj]+kruhj,Gj2);
   i1=max(s3[jj]-kruhi,Gi1);
   i2=min(s3[jj]+kruhi,Gi2);

//  printf("\nx:\t%d  -  %d\ny:\t%d  -  %d\nz:\t%d  -  %d\n\n",k1,k2,jj1,j2,i1,i2);

   idim = i2-i1+1;
   jdim = j2-jj1+1;
   kdim = k2-k1+1;
printf("%d %d %d\n",idim,jdim,kdim);
   cput1=clock()/(double)(CLOCKS_PER_SEC);

   MakingSegmentationFunction();
   cput2=clock()/(double)(CLOCKS_PER_SEC);
   printf("CPU time Making Segmentation Function: %e secs\n",cput2-cput1);

//   Writing3D();
   Rescalepmc();

   cput1=clock()/(double)(CLOCKS_PER_SEC);

   for (kk=skk+1;kk<=pck;kk++)
   {
     scale=scale+tau;
     zmena=EllipticStep();
     if (kk % iz ==0)
     {
       Writing3D();
       printf("zmena: %lf\n",zmena);
     }
     if (verbose==1) printf("%3d .time step finished, zmena: %20.18lf \n",kk,zmena);
   }
   printf("%i. cell segmented in %i steps\n",jj,kk);
    cput2=clock()/(double)(CLOCKS_PER_SEC);
//     if (verbose==1) 
     printf("CPU time cell segmentation: %e secs\n",cput2-cput1);
  }

  cput_end=clock()/(double)(CLOCKS_PER_SEC);
  printf("CPU time needed: %e secs\n",cput_end-cput_start);

//  free(name);

  free(pmc);

}

