#!/bin/bash

# GSubSurf - Generalized Subjective Surface Method for segmentation in 3D 
# Written in 2015 by Karol Mikula mikula@math.sk
# Robert Spir spir.robert@gmail.com
# Mariana Remesikova remesikova@math.sk
# Alessandro Sarti alessandro.sarti@ehess.fr
# 
# To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
# You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.

PATHDESTINATION="$(O="$(cd "$(dirname "$0")" && pwd)/$(basename "$0")"; while [ ! -e "$O/.bioemergences" -a "$O" != "/" ]; do O="$(dirname "$O")"; done; echo "$O")"
BIWF_ALGOSDIR="${PATHDESTINATION}/Algorithmes"
MPI_HOME="${PATHDESTINATION}/lib/MPICH/mpich"
MPI="${MPI_HOME}/bin/mpiexec"
YOU_CAN_CLOSE="*** You can now close the window ***";

echo " === Nuclei Segmentation ==="
echo ""
echo "Starting at `date`"

algodir="${BIWF_ALGOSDIR}/Segmentation/NucleiSegmentation/"
source "${algodir}/Parameters.sh"

per_iter=$iter    #periodicity of writing
prolong=30
edgeXY=36
edgeZ=20

if [ -d "$1" ]; then
  inputDir="$1"
else
  echo "$1 is not a directory"
  echo "${YOU_CAN_CLOSE}"
  exit 1
fi

echo "Input VTK Directory : $inputDir"

if [ -d "$2" ]; then
  CenterPath="$2"
else
  echo "$2 is not a directory"
  echo "${YOU_CAN_CLOSE}"
  exit 1
fi
echo "Center Path : $CenterPath"

if [ -n "$2" ]; then
  outputDir="$3"
  if [ "$outputDir" == "" ]; then 
    echo "You must specify an output path"
    echo "${YOU_CAN_CLOSE}"
    exit 1
  fi
else
  echo "You must specify an output path"
  echo "${YOU_CAN_CLOSE}"
  exit 1
fi
echo "outputDir : $outputDir"
mkdir -p "${outputDir}"
mkdir -p "${outputDir}/polydata/"

. "${BIWF_ALGOSDIR}/HelperFunctions.sh"

exe="${algodir}EXE/segm_nuclei"

formatVTK=1
verb=0

echo 'progression=0' > "${algodir}/.run"

# SimAnn
if [ -e "${CenterPath}/Centers" -a -e "${CenterPath}/IDS" ]
then
  cp -R "${CenterPath}/Centers" "${outputDir}"
  cp -R "${CenterPath}/IDS" "${outputDir}"
# Otherwise we must convert VTK to TXT in a Centers folder, and create IDs
else
  echo "Creates a fake tracking"
  lineage="${outputDir}/fake.lineageTree"
  printf '' > "${lineage}"
  currentID=100000
  
  echo "Extracts Dimensions and Spacing"
  rawvtk="$(ls "${inputDir}"/*${channel}*.vtk* | head -n 1)"
  processVTK "${rawvtk}" "${outputDir}"

  rm -rf "${output}"
  rm -rf "${outputDir}/tmp"
  
  echo "Building fake lineage"
  mkdir -p "${outputDir}/Centers"
  for vtk in "${CenterPath}/"*.vtk
  do
    echo "${vtk}"
    # Extract timestep
    timestep=$(basename "${vtk}" | grep -oE '_t[0-9]+')
    timestep=${timestep:2}

    # Convertion
    txt="${outputDir}/Centers/Centers_t${timestep}.txt"
    cn=0
    for coord in $(tail -n +6 "${vtk}")
    do
      printf ${coord} >> "${txt}"

      cn=$((cn + 1))
      if [ $((cn % 3)) -eq 0 ]
      then
        printf '\n' >> "${txt}"
      else
        printf ' ' >> "${txt}"
      fi
    done

    # Adding each centers to the fake lineage
    while read line
    do
      X=$(echo "${line}" | awk '{print $1}')
      Y=$(echo "${line}" | awk '{print $2}')
      Z=$(echo "${line}" | awk '{print $3}')

      printf '%s,-1,%s,%s,%s,%s,0,0,0,0,0,0,0,0\n' ${currentID} ${X} ${Y} ${Z} $(echo ${timestep} | sed 's/^0*//') >> "${lineage}"
      currentID=$((currentID + 1))
    done < "${txt}"
  done

  rm -r "${outputDir}/Centers"

  #Export Lineage for Mov-IT
  emblineage="${outputDir}/fake.emb"

  if [ ! -e "${emblineage}" ];
  then
	  echo "Export $LineageOutput to $emblineage"   
  	 "${BIWF_ALGOSDIR}"/Converter/CreateMOV-IT/EXE/ConvertLNGtoEMB \
  	 --inputfile "${lineage}" \
  	 --outputfile "${emblineage}" \
  	 --SpacingX ${sx} \
  	 --SpacingY ${sy} \
  	 --SpacingZ ${sz} \
  	 --ImageSizeX ${n1} \
  	 --ImageSizeY ${n2} \
  	 --ImageSizeZ ${n3}  
  fi

  mkdir -p "${outputDir}/Centers"
  mkdir -p "${outputDir}/IDS"
  "${BIWF_ALGOSDIR}/Converter/ExportMOV-ITCenters/EXE/ExportMOVITCenters" --inputfile "${emblineage}" --outputcenters "${outputDir}"/Centers/Centers_t%03d.txt --outputID "${outputDir}"/IDS/IDS_t%03d.txt --ImageSizeX ${n1} --ImageSizeY ${n2} --ImageSizeZ ${n3}

  CenterPath="${outputDir}"
fi

#Firsr Times step
times=$(getStartTime "${CenterPath}/Centers/Centers_t%03d.txt")

ntotal=$(ls "${inputDir}"/*${channel}*.vtk* | wc -l)
ncurrent=0

# running 
for inputFile in "${inputDir}"/*${channel}*.vtk*
do
  namefile="$(basename "${inputFile}")"
  echo "Working with ${namefile}..."
  
  if [ ! -e "${outputDir}/${namefile}" ];
  then
    printf '\n---- Creating Segmentation File: %s/%s\n' "${outputDir}" "${namefile}"
    processVTK "${inputFile}" "${outputDir}"

    printf '\n---- Detected time is: %s\n' "${times}"
    outputFile="$outputDir/${strippedName}.vtk"
    centers="$(printf "${CenterPath}/Centers/Centers_t%03d.txt" ${times})"

    input="$output $tempDir/$binaryData ${centers} ${n3} ${n2} ${n1} ${embryoType} ${prolong} ${edgeXY} ${edgeZ} ${sx} ${sy} ${sz} ${epsilon} ${edgePower} ${convolutionU} ${convolutionPMC} ${formatVTK} ${K} ${tau} ${sigma} ${Vadv} ${Vcurv} ${iter} ${per_iter} ${verb}"
    printf '\n---- Run command:\n---- %s\n' "${input}"
    echo "${input}" | "${exe}"
    
    printf '\n---- Merging VTK Files\n'
    "${BIWF_ALGOSDIR}"/Converter/MergeVTK/EXE/MergeVTK --inputpattern "${output}"/_segm_%d.vtk --outputfile "${outputDir}/${namefile}" --SpacingX ${sx} --SpacingY ${sy} --SpacingZ ${sz} --ImageSizeX ${n1} --ImageSizeY ${n2} --ImageSizeZ ${n3}
    
    printf '\n---- Cleaning\n'
    rm -rf "${output}"
    rm -rf "${outputDir}/tmp"
  fi
  
  if [ ! -e "${outputDir}/polydata/${namefile}" ];
  then
    printf '\n---- Creating polydata files: %s/polydata/%s\n' "${outputDir}" "${namefile}"
    IDtime="$(printf "${CenterPath}/IDS/IDS_t%03d.txt" ${times})"
    "${BIWF_ALGOSDIR}"/Converter/ConvertVTKtoPolydata/EXE/ConvertVTKtoPolydata "${outputDir}/${namefile}" "${IDtime}" "${outputDir}/polydata/${namefile}"
  fi 

  times=$((times + 1));
  
  ncurrent=$((ncurrent + 1))
  echo "progression=$((${ncurrent} * 100 / ${ntotal}))" > "${algodir}/.run"
done

######
# Cleaning
###
if [ -e ~/temp.vtk ]
then
  rm -f ~/temp.vtk
fi

rm -fr "${outputDir}/Centers"
rm -fr "${outputDir}/IDS"
find "${outputDir}" -maxdepth 1 -type f -exec rm {} \;
mv "${outputDir}/polydata"/* "${outputDir}"
rm -fr "${outputDir}/polydata"

if [ -e "${algodir}/.run" ]
then
  rm -f "${algodir}/.run"
fi

echo "${YOU_CAN_CLOSE}"
