/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef __celluletype_h
#define __celluletype_h


#include "definitions.h"
#include <iostream>
#include <vector>

#include <boost/archive/text_iarchive.hpp>
#include <boost/archive/text_oarchive.hpp>

#include <boost/serialization/base_object.hpp>
#include <boost/serialization/utility.hpp>



/*!
 * @file CelluleType.h
 * @brief Contains class CelluleType, struct pointer_extractor, struct t_extractor, and typedef CelluleTypePtr
 * @see CelluleType
 * @see pointer_extractor
 * @see t_extractor
 * @see CelluleTypePtr
 */


/*! 
 * @class CelluleType CelluleType.h "CelluleType.h"
 * @brief The class implement the cell definition with a Center, a Membrane and a Nuclei.
 */

class  CelluleType
{
  //todo: Should these members be public? Some of them have getters and setters
  public:
    /*!  Id of the cell in the embryo */
    long IdDb;
    
    /*! timestep of this cell */
    long T;
    
    /*! Used to calculate the connect component*/
    long ComponentNumber;
    
    /*! Name of the center from the time it's born until it dies*/
    long Name;
    
    /*! Number of successors this cellule has */
    long NumberSuccessors;
    
    /*! Number of predecessors this cellule has */
    long NumberPredecessors;
    
    /*! unused parameter/ todo: delete this */
    long Newbourne;
    
    /*! Number of the center in the input file */
    long CenterNumber;
    
    /*! unused/ todo: delete */
    double m_Intensity;
    
    /*! is this cellule newly born? */
    bool IsNewBourne;
    
    /*! stores whether or not this cell is calculatable from the image data provided */
    bool hasIntensity;


    /*! is cell_number from the DataBase z_center and z_cell*/ //todo: what is this referring to?
  private:
    /*! is Coordinate X of this Cell */
    float X;
    /*!is Coordinate Y of this Cell */
    float Y;
    /*!is Coordinate Z of this Cell */
    float Z;
    /*! is VectorField X of this Cell */
    float DX;
    /*! is VectorField Y of this Cell */
    float DY;
    /*! is VectorField Z of this Cell */
    float DZ;

    /*! todo: unused, delete */    
    float Deformation;
    
    /*! todo: unused, delete */    
    long Virtual;
    
    /*! todo: unused, delete */    
    unsigned char picked;
  public:
    /*! default constructor */
    CelluleType ();
    
    /*! 
     * @brief secondary constructor
     * @param x 
     * @param y
     * @param z
     * @param t
     * @param CenterNumber
     * @param IdDb 
     */
    CelluleType ( double x, double y, double z, long t, long CenterNumber, long IdDb );

    
    /*! destructor */
    ~CelluleType();

    /*!
     * @brief set the IdDb
     * @see IdDb
     * @param pIdDb any non-negative number (please read on the dangers of unnecessary unsigned use in c and c++ if you're curious why this is unsigned)
     */
    void SetIdDb ( long pIdDb )
    {
      this->IdDb = pIdDb;
    };
    
    /*!
     *  @brief returns the IdDb of this CelluleType
     * 	@see IdDb
     */
    long GetIdDb() const
    {
      return this->IdDb;
    };

    /*! generates getter methods for private member Deformation */
    GetMacro ( Deformation, float );
    
    /*! generates setter methods for private member Deformation */
    SetMacro ( Deformation, float );

    /*! generates setter methods for private member IsNewBourne */
    SetMacro ( IsNewBourne, bool );
    
    /*! generates getter and setter methods for a private member Deformation */
    GetMacro ( IsNewBourne, bool );
        
    /*!
     * @brief dummy function/ todo: delete
     * @param radious todo: describe param
     */
    void CalcIntensity ( double radious );
    
    /*!
     * @brief returns the intensity of the image data within a radius of this cell's position
     * @param radious a standard radius in units relative to image spacing
     * 
     * todo: delete, this function is a dummy function that will always return 0
     */
    double GetIntensity ( double radious );
    
    /*!
     * @brief Checks if this CelluleType is equal to c
     * @param c The CelluleType to compare against
     * @return todo: describe if == is reference equality or value equality
     */
    bool operator== ( const CelluleType& c ) const;


    /*!
     * @brief returns true if f1 has a lesser timestep than f2
     * @param f1 a cell at a certain timestep
     * @param f2 also a cell at a certain timestep
     * @return If f1.T is less than f2.T returns true.
     */
    static bool CellTimeCompare ( const CelluleType &f1, const CelluleType &f2 )
    {
      return f1.GetT() < f2.GetT();
    }

    friend class boost::serialization::access;
    
    /*!
     * @brief serialization helper function for boost::serialization, users should not use this 
     * todo: make private or protected
     */
    template<class Archive> void serialize ( Archive & ar, const unsigned int version )
    {
//      ar & imageProvider;
      ar & hasIntensity;
      ar & m_Intensity;
      ar & IdDb;
      ar & CenterNumber;
      ar & Name;
      ar & X;
      ar & Y;
      ar & Z;
      ar & T;
      ar & DX;
      ar & DY;
      ar & DZ;
      ar & Deformation;
      ar & Virtual;
      ar & ComponentNumber;
      ar & NumberSuccessors;
      ar & NumberPredecessors;
      ar & Newbourne;
    }

    /*! generates setter methods for a private member CenterNumber */
    SetMacro ( CenterNumber, long );
    
    /*! generates getter methods for a private member CenterNumber */
    GetMacro ( CenterNumber, long );
    
    /*! generates getter methods for private member Name */
    GetMacro ( Name, long );
    
    /*! generates setter methods for private member Name */
    SetMacro ( Name, long );
    
    /*! generates getter methods for private member X */    
    GetMacro ( X, float );
    
    /*! generates setter methods for private member X */
    SetMacro ( X, float );
    
    /*! generates getter methods for private member Y */  
    GetMacro ( Y, float );
    
    /*! generates setter methods for private member Y */
    SetMacro ( Y, float );
    
    /*! generates getter methods for private member Z */    
    GetMacro ( Z, float );
    
    /*! generates setter methods for private member Z */    
    SetMacro ( Z, float );
    
    /*! generates getter methods for private member DX */    
    GetMacro ( DX, float );
    
    /*! generates setter methods for private member DX */    
    SetMacro ( DX, float );
    
    /*! generates getter methods for private member DY */    
    GetMacro ( DY, float );
    
    /*! generates setter methods for private member DY */    
    SetMacro ( DY, float );
    
    /*! generates getter methods for private member DZ */    
    GetMacro ( DZ, float );
    
    /*! generates setter methods for private member DZ */    
    SetMacro ( DZ, float );
    
    /*! generates getter methods for private member T */    
    GetMacro ( T, long );
    
    /*! generates setter methods for private member T */    
    SetMacro ( T, long );
    
    /*! generates getter methods for private member Virtual */    
    GetMacro ( Virtual, long );
    
    /*! generates setter methods for private member Name */
    SetMacro ( Virtual, long );
    
    /*!
     * @brief Fills vector with the cartesian coordinates of this cellule ({x,y,z} format)
     * @param vector a 3 float array that serves as the return of this function
     */
    void GetPosition ( float vector[3] );
    
    /*!
     * @brief returns the distance of this cell from vector
     * @param vector a point in cartesian space {x,y,z} format
     * @return the distance relative to the image spacing of the input data
     */
    float GetDistance ( float vector[3] );
    
    /*!
     * @brief fills vector with the vector of this cell
     * @param vector a blank array to be filled with the SCIENTIFIC/MATHEMATIC vector of this cell (format {dx,dy,dz})
     */
    void GetDirection ( float vector[3] );
    
    /*!
     * @brief returns the delta x, y, or z depending on vector
     * @param vector 0 = dx, 1 = dy, 2 = dz
     * @return the movement of this cell on an axis, with units relative to the image spacing of the input data
     */
    float GetDirection ( int vector );
    
    /*!
     * @brief sets the vector of this cell
     * @param vector a mathematical vector as represented by an array with {dx,dy,dz} format
     */
    void SetDirection ( float vector[3] );
    
    /*!
     * @brief sets the vector of this cell
     * @param vx x value of the vector
     * @param vy y value of the vector
     * @param vz z value of the vector
     */
    void SetDirection ( float vx, float vy, float vz );
    
    /*!
     * @brief returns the estimated vector of this cell
     * @param vector a blank 3 element array to fill with the estimated vector
     */    
    void GetEstimatedDirection ( float vector[3] );
    
    /*!
     * @brief determines if this cell falls within the bounds of bounds
     * @param bounds a std::vector type representing a 3d cartesian box of format {x_s,x_e,y_s,y_e,z_s,z_e} where _s is the startpoint for that axis and _e is the endpoint.
     */    
    bool IsInBounds ( std::vector< float > bounds );

};

/*! @typedef A specialization of shared_ptr for CelluleType */
typedef boost::shared_ptr<CelluleType>  CelluleTypePtr ;

/*!
 * @struct pointer_extractor "CelluleType.h"
 * @brief A simple wrapper to extract IdDb values from CelluleTypePtrs
 */
struct pointer_extractor
{
  /*! todo: describe typedef */
  typedef long result_type;

  /*! 
   * @brief override of the () operator that returns the IdDb of the CelluleTypePtr ref passed to it
   * @param e a managed pointer to a CelluleType
   * @return returns the IdDb of the CelluleType
   * @see CelluleType::IdDb
   * @see CelluleTypePtr
   */
  const result_type operator() ( const CelluleTypePtr& e ) const
  {
    return e->GetIdDb();
  }
  
  /*!
   * @brief override of the () operator that returns the IdDb of the CelluleTypePtr pointer passed to it
   * @param e A managed pointer to a CelluleType
   * @return returns the IdDb of the CelluleType
   * @see CelluleType::IdDb
   * @see CelluleTypePtr
   */
  result_type operator() ( CelluleTypePtr* e ) const
  {
    long  a = ( *e )->GetIdDb();
    return a;
  }
};


/*!
 * @struct t_extractor "CelluleType.h"
 * @brief A simple wrapper to extract T values from CelluleTypePtrs
 */
struct t_extractor
{
  /*! todo: describe typedef */
  typedef long result_type;
  
  /*!
   * @brief override of the () operator that returns the IdDb of the CelluleTypePtr referece passed to it
   * @param e A managed pointer to a CelluleType
   * @return returns the IdDb of the CelluleType
   * @see CelluleType::IdDb
   * @see CelluleTypePtr
   */
  const result_type operator() ( const CelluleTypePtr& e ) const
  {
    return e->GetT();
  }
  
  /*!
   * @brief override of the () operator that returns the IdDb of the CelluleTypePtr pointer passed to it
   * @param e A managed pointer to a CelluleType
   * @return returns the IdDb of the CelluleType
   * @see CelluleType::IdDb
   * @see CelluleTypePtr
   */  
  result_type operator() ( CelluleTypePtr* e ) const
  {
    long  a = ( *e )->GetT();
    return a;
  }
};
#endif