/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef __DbAccess_h__
#define __DbAccess_h__

#include <fstream>
#include <iostream>
#include <ostream>
#include <istream>
#include <string>
#include <fstream>
#include <stdio.h>
#include <stdlib.h>
#include <vector>
#include <set>


#include "EmbryoType.h"
#include "GraphCellType.h"
#include "EdgePropertyType.h"

#include <vector>
#include <list>
#include <map>

#include <fstream>
#include <algorithm>
#include <iterator>
#include <Convert.h>
//#include <boost/foreach.hpp>

#define CountFetch 100000


using namespace oracle::occi;


template <class T>
void findandreplace ( T& source, const T& find, const T& replace )
{
  size_t j;
  for ( ; ( j = source.find ( find ) ) != T::npos; )
  {
    source.replace ( j, find.length(), replace );
  }
}


template < class E >
std::string LongToString ( E l )   //convert long to string
{
  std:: stringstream s;
  s << l;
  return s.str();
}



template < typename E,typename EPtr, typename G >
    void downloadCells ( G *TotalBase,
                         long id_process_center,
                         long id_process_vectorfield,
                         long timeStart,
                         long timeEnd,
                         long timeModulo, std::set< EPtr > &container )
{
  std::string command;


  command = " select z_center.X  as X, z_center.Y as Y, z_center.Z  as Z, z_center.id_time as T, z_center.id_center as IdDB , z_center.cell_number as CellNumber, coalesce(z_vectorfield.vx,0)  as DX, coalesce(z_vectorfield.vy,0)   as DY, coalesce(z_vectorfield.vz,0)   as DZ from z_center left join z_vectorfield on z_center.id_center = z_vectorfield.id_center and z_vectorfield.id_process_vectorfield =  :id_process_vectorfield:  where  z_center.id_time >= :timeStart: and z_center.id_time <= :timeEnd: and z_center.id_process_center = :id_process_center: and mod(z_center.id_time,  :timeModulo:)  = 0 "; 
  

  
  findandreplace ( command, std::string ( ":id_process_vectorfield:" ), LongToString ( id_process_vectorfield ) );
  findandreplace ( command, std::string ( ":timeStart:" ), LongToString ( timeStart ) );
  findandreplace ( command, std::string ( ":timeEnd:" ), LongToString ( timeEnd ) );
  findandreplace ( command, std::string ( ":id_process_center:" ), LongToString ( id_process_center ) );
  findandreplace ( command, std::string ( ":timeModulo:" ), LongToString ( timeModulo ) );

  std::cout << "downloadCells" << std::endl;
  std::cout << command << std::endl;

  /////////////////
  std::map< std::string, long > colName;
  oracle::occi::ResultSet *resultSet;
  oracle::occi::Statement *SCellList;

  long i ;
  try
  {
    SCellList = TotalBase->GetConnection()->createStatement ( command );


    long T[CountFetch];
    float X[CountFetch];
    float Y[CountFetch];
    float Z[CountFetch];
    long centerNumber[CountFetch];
    long IDDB[CountFetch];
    float DX[CountFetch];
    float DY[CountFetch];
    float DZ[CountFetch];

    resultSet = SCellList->executeQuery();


    std::vector<MetaData> md = resultSet->getColumnListMetaData ();
    int numCols = md.size();

    for ( i = 0; i < numCols; i++ )
    {
      int ptype = md [ i ].getInt ( MetaData::ATTR_PTYPE );
      if ( ptype == MetaData::PTYPE_COL )
      {
        colName[  md[ i ].getString ( MetaData::ATTR_NAME ) ] = i + 1;
      }
    }



    resultSet ->setDataBuffer ( colName["T"], T, OCCIINT, sizeof ( long ) ) ;
    resultSet ->setDataBuffer ( colName["X"], X, OCCIFLOAT, sizeof ( float ) ) ;
    resultSet ->setDataBuffer ( colName["Y"], Y, OCCIFLOAT, sizeof ( float ) ) ;
    resultSet ->setDataBuffer ( colName["Z"], Z, OCCIFLOAT, sizeof ( float ) ) ;
    resultSet ->setDataBuffer ( colName["IDDB"], IDDB, OCCIINT, sizeof ( long ) ) ;
    resultSet ->setDataBuffer ( colName["CELLNUMBER"], centerNumber, OCCIINT, sizeof ( long ) ) ;
    resultSet ->setDataBuffer ( colName["DX"], DX, OCCIFLOAT, sizeof ( float ) ) ;
    resultSet ->setDataBuffer ( colName["DY"], DY, OCCIFLOAT, sizeof ( float) ) ;
    resultSet ->setDataBuffer ( colName["DZ"], DZ, OCCIFLOAT, sizeof ( float ) ) ;


    //////////////////////////

    while ( resultSet->next ( CountFetch ) )
    {
      for ( i = 0;i < resultSet->getNumArrayRows();++i )
      {
        EPtr cell(new E ( X[i], Y[i], Z[i], static_cast<long> ( T[i] ), static_cast<long> ( centerNumber[i] ), static_cast<long> ( IDDB[i] ) ));
        container.insert ( cell );
      }
    }
  }
  catch ( SQLException sqle )
  {
    std::cout << "Request to database failed: " << i << " " << sqle.what();
    exit(1);
  }

  try
  {
    SCellList->closeResultSet ( resultSet );
    TotalBase->GetConnection()->terminateStatement ( SCellList );
  }
  catch ( SQLException sqle )
  {
    std::cout << "Request to database failed: " << sqle.what();
    exit(1);
  }
}





template <typename G>
void downloadLinks ( G *TotalBase,
                     long id_process_tracking,
                     long id_process_center,
                     long timeStart,
                     long timeEnd,
                     std::list< long > &fromCell,
                     std::list< long > &toCell )
{
  std::string command;
//  command = " select  aParentCell.id_center as fromCell,aParentCell.id_time as fromTime, aCell.id_center as toCell,aCell.id_time as toTime from z_tree join z_center aCell on z_tree.id_cell = aCell.id_center join z_center aParentCell on z_tree.id_cellmother = aParentCell.id_center where z_tree.id_process_tracking = :id_process_tracking: and aParentCell.id_process_center = :id_process_center: and aCell.id_process_center = :id_process_center: and aParentCell.id_time >= :timeStart: and aParentCell.id_time <= :timeEnd:  ";

  command = " select  z_tree.id_cellmother as fromCell,aCell.id_center as toCell from z_tree join z_center aCell on z_tree.id_cell = aCell.id_center  where z_tree.id_process_tracking = :id_process_tracking:  and aCell.id_process_center = :id_process_center: and aCell.id_time >= :timeStart:  and aCell.id_time <= :timeEnd: and z_tree.id_cellmother is not NULL";


  findandreplace ( command, std::string ( ":id_process_tracking:" ), stringify ( id_process_tracking ) );
  findandreplace ( command, std::string ( ":id_process_center:" ), stringify ( id_process_center ) );
  findandreplace ( command, std::string ( ":timeStart:" ), stringify ( timeStart ) );
  findandreplace ( command, std::string ( ":timeEnd:" ), stringify ( timeEnd ) );

  std::cout << "downloadLinks" << std::endl;
  std::cout << command << std::endl;

  std::map< std::string, long > colName;
  oracle::occi::ResultSet *resultSet;
  oracle::occi::Statement *SCellList;

  /////////////////

  try
  {
    SCellList = TotalBase->GetConnection()->createStatement ( command );

    long from[CountFetch];
    long to[CountFetch];

    resultSet = SCellList->executeQuery();

    std::vector<MetaData> md = resultSet->getColumnListMetaData ();
    int numCols = md.size();

    for ( int i = 0; i < numCols; i++ )
    {
      int ptype = md [ i ].getInt ( MetaData::ATTR_PTYPE );
      if ( ptype == MetaData::PTYPE_COL )
      {
        colName[  md[ i ].getString ( MetaData::ATTR_NAME ) ] = i + 1;
      }
    }


    resultSet->setDataBuffer ( colName["FROMCELL"], from, OCCIINT, sizeof ( long ) ) ;
    resultSet->setDataBuffer ( colName["TOCELL"], to, OCCIINT, sizeof ( long ) ) ;

    //////////////////////////

    unsigned long i ;
    while ( resultSet->next ( CountFetch ) )
    {
      for ( i = 0;i < resultSet->getNumArrayRows();++i )
      {
        fromCell.push_back ( from[i] );
        toCell.push_back ( to[i] );
      }
    }
  }
  catch ( SQLException sqle )
  {
    std::cout << "Request to database failed: " << sqle.what();
    exit (-1);

  }

  try
  {
    SCellList->closeResultSet ( resultSet );
    TotalBase->GetConnection()->terminateStatement ( SCellList );
  }
  catch ( SQLException sqle )
  {
    std::cout << "Request to database failed: " << sqle.what();
    exit (-1);

  }
}



template < typename E,typename EPtr, typename G >
void downloadCellsTracking ( G *TotalBase,
                             long id_process_tracking,
                             long timeStart,
                             long timeEnd,
                             long timeModulo, std::set< EPtr > &container,
                             std::list< long > &fromCell,
                             std::list< long > &toCell )
{
  std::string command;

  command = "select tree.X as X, tree.Y as Y, tree.Z as Z, tree.time as T, tree.id_center as IdDB , tree.id_center_mother as Mother_IdDB,  tree.cell_number as CellNumber from z_center_tree tree where  tree.id_process_center_tree = :id_process_tracking: and tree.time >= :timeStart: and tree.time <= :timeEnd: ";


  findandreplace ( command, std::string ( ":timeStart:" ), LongToString ( timeStart ) );
  findandreplace ( command, std::string ( ":timeEnd:" ), LongToString ( timeEnd ) );
  findandreplace ( command, std::string ( ":id_process_tracking:" ), LongToString ( id_process_tracking ) );

  std::cout << "downloadTracking" << std::endl;
  std::cout << command << std::endl;

  /////////////////
  std::map< std::string, long > colName;
  oracle::occi::ResultSet *resultSet;
  oracle::occi::Statement *SCellList;

  try
  {
    SCellList = TotalBase->GetConnection()->createStatement ( command );

    long X[CountFetch];
    long Y[CountFetch];
    long Z[CountFetch];
    long T[CountFetch];
    long centerNumber[CountFetch];
    long IDDB[CountFetch];
    long Mother_IDDB[CountFetch];
    ub2 lengthMother[CountFetch];
    sb2 indMother[CountFetch];
    ub2 rcMother[CountFetch];


    resultSet = SCellList->executeQuery();

    std::vector<MetaData> md = resultSet->getColumnListMetaData ();
    int numCols = md.size();

    for ( int i = 0; i < numCols; i++ )
    {
      int ptype = md [ i ].getInt ( MetaData::ATTR_PTYPE );
      if ( ptype == MetaData::PTYPE_COL )
      {
        colName[  md[ i ].getString ( MetaData::ATTR_NAME ) ] = i + 1;
      }
    }

    
//     for ( int j = 0; j < CountFetch; j++ )
//     {
//       idLen[j] = sizeof ( vec_time[j] );
//       idLenDouble[j] = sizeof ( double );
//     }

    
    resultSet->setDataBuffer ( colName["X"], X, OCCIINT, sizeof ( long ) ) ;
    resultSet->setDataBuffer ( colName["Y"], Y, OCCIINT, sizeof ( long ) ) ;
    resultSet->setDataBuffer ( colName["Z"], Z, OCCIINT, sizeof ( long ) ) ;
    resultSet->setDataBuffer ( colName["T"], T, OCCIINT, sizeof ( long ) ) ;
    resultSet->setDataBuffer ( colName["IDDB"], IDDB, OCCIINT, sizeof ( long ) ) ;
    resultSet->setDataBuffer ( colName["MOTHER_IDDB"], Mother_IDDB, OCCIINT, sizeof ( long ),lengthMother,indMother,rcMother ) ;
    resultSet->setDataBuffer ( colName["CELLNUMBER"], centerNumber, OCCIINT, sizeof ( long ) ) ;
    //////////////////////////

    long i ;
    while ( resultSet->next ( CountFetch ) )
    {
      for ( i = 0;i < resultSet->getNumArrayRows();++i )
      {
        EPtr cell( new E ( X[i], Y[i], Z[i], T[i], centerNumber[i], IDDB[i] ));
        
        container.insert ( cell );

        if (indMother[i]!=-1)
        {
          fromCell.push_back ( Mother_IDDB[i] );
          toCell.push_back ( IDDB[i] );
        }
      }
    }
  }
  catch ( SQLException sqle )
  {

    std::cout << "Request to database failed: " << sqle.what();
    exit (-1);
  }

  try
  {
    SCellList->closeResultSet ( resultSet );
    TotalBase->GetConnection()->terminateStatement ( SCellList );
  }
  catch ( SQLException sqle )
  {
    std::cout << "Request to database failed: " << sqle.what();
    exit (-1);
  }
}


template < typename G >
float GetNew ( G *TotalBase, std::string table , std::string field, std::string where )
{
  float resu = 0;

  std::string command;

  command = "(SELECT max( :field: ) + 1 as value from :table: where 0 = 0  " + where + ")";

  findandreplace ( command, std::string ( ":field:" ), field );
  findandreplace ( command, std::string ( ":table:" ), table );

  std::cout << "New " << field << std::endl;
  std::cout << command << std::endl;

  /////////////////
  std::map< std::string, long > colName;
  oracle::occi::ResultSet *resultSet;
  oracle::occi::Statement *SCellList;

  try
  {
    SCellList = TotalBase->GetConnection()->createStatement ( command );
    float fieldValue[CountFetch];
    resultSet = SCellList->executeQuery();
    resultSet ->setDataBuffer ( 1, fieldValue, OCCIFLOAT, sizeof ( float ) ) ;

    //////////////////////////
    long i ;
    while ( resultSet->next ( CountFetch ) )
    {
      for ( i = 0;i < resultSet->getNumArrayRows();++i )
      {
        resu = fieldValue[i];
      }
    }
  }
  catch ( SQLException sqle )
  {

    std::cout << "Request to database failed: " << sqle.what();
    exit (-1);
  }

  try
  {
    SCellList->closeResultSet ( resultSet );
    TotalBase->GetConnection()->terminateStatement ( SCellList );
  }
  catch ( SQLException sqle )
  {
    std::cout << "Request to database failed: " << sqle.what();
    exit (-1);
  }
  return resu;
}





template < typename G >
std::vector< int > GetIdsVector ( G *TotalBase, long size )
{
  std::vector< int > resu ( size );
  std::stringstream command;
  command << "begin EMBRYOMICS.GET_ID_CENTER_TREE_LIST(:1, :2); end;";
                    
  try
  {
    //initialize in object mode since we are accessing VARRAY types
    Statement *stmt = TotalBase->GetConnection()->createStatement();

    try
    {
      //create anonymous block for executing stored proc/function
      stmt->setSQL ( command.str().c_str() );
      stmt->setNumber ( 1, size );
      stmt->registerOutParam ( 2, OCCIVECTOR, 0, "DENALI.TABOFNUMBER" );//varray


      stmt->execute(); //execute procedure

      //get the OUT parameters
      getVector ( stmt, 2, resu ); //there are overloaded getVector methods for each vector<T>
    }
    catch ( SQLException &ex )
    {
      //cleanup
      std::cout << "Error, cleaning up..." << std::endl;
      TotalBase->GetConnection()->terminateStatement ( stmt );
      exit (-1);
      
    }
    TotalBase->GetConnection()->terminateStatement ( stmt );

  }
  catch ( SQLException &ex )
  {
    std::cout << "Error Getting next Value : " << ex.getMessage() << std::endl;
    exit (-1);

  }
  return resu;
}


template < typename G >
long GetNext ( G *TotalBase, std::string sequence, long size )
{
  long resu;
  std::stringstream command;
  command << "begin ";
  command << " for i in 1.." << size << " loop " ;
  command << "      select " << sequence << ".nextval into :1 from  dual; " ;
  command << " end loop;";
  command << "end;";

  try
  {
    //initialize in object mode since we are accessing VARRAY types
    Statement *stmt = TotalBase->GetConnection()->createStatement();

    try
    {
      //create anonymous block for executing stored proc/function
      stmt->setSQL ( command.str().c_str() );

      //pass IN parameters by setXXX(), indicate OUT parameters
      //by registerOutParam
      stmt->registerOutParam ( 1, OCCIINT );

      stmt->execute(); //execute procedure

      //get the OUT parameters
      resu = stmt->getInt ( 1 );

    }
    catch ( SQLException &ex )
    {
      //cleanup
      std::cout << "Error, cleaning up..." << std::endl;
      TotalBase->GetConnection()->terminateStatement ( stmt );
      exit (-1);
    }
    TotalBase->GetConnection()->terminateStatement ( stmt );

  }
  catch ( SQLException &ex )
  {
    std::cout << "Error Getting next Value : " << ex.getMessage() << std::endl;
    exit (-1);
  }
  return resu -size;
}



template < typename EPtr, typename G >
    long SetNewIdDB ( G *TotalBase, GraphCellType<EPtr> &embryo)
{
  if (embryo.GetTimes().size()!=0)
  {
    SetNewIdDB ( TotalBase, embryo, *(embryo.GetTimes().begin()), *(embryo.GetTimes().rbegin()) );
  }
}



template < typename EPtr, typename G >
    long SetNewIdDB ( G *TotalBase, GraphCellType<EPtr> &embryo, long InitUpload, long EndUpload)
{
  try
  {
    std::map<long, long> cellsPerTime;
    std::set< long > times = embryo.GetTimes();

    for ( std::set< long >::iterator  timeIt = times.begin();timeIt != times.end();timeIt++ )
    {
      cellsPerTime[*timeIt] = 100000;
    }

    //Asing the new IDDB to all the cells!!
    std::set< EPtr > allCells = embryo.GetCells (InitUpload,EndUpload );

    std::vector<int> id_center_vec ;
    id_center_vec = GetNextVector ( TotalBase, embryo.size() );

    long i = 0;

    for ( typename std::set< EPtr >::iterator cellIt = allCells.begin(); cellIt != allCells.end();cellIt++ )
    {
      EPtr cell = *cellIt;
      cell->SetIdDbNew ( id_center_vec[i] );
      i++;

      //Count the cells per time
      cellsPerTime[ ( *cellIt )->GetT() ]++;
    }
    //Asing the new Name!!
    std::set<EPtr>  cellsToEnumerate = embryo.GetCellsToEnumerate();

    for ( typename std::set< EPtr >::iterator cell = cellsToEnumerate.begin(); cell != cellsToEnumerate.end();cell++ )
    {
      ( *cell )->SetCenterNumber ( cellsPerTime[ ( *cell )->GetT() ]++ );
      ( *cell )->SetIdDb ( 0 );
    }
  }
  catch ( SQLException sqle )
  {
    std::cout << "Request to database failed: " << sqle.what();
    exit (-1);
  }
  return 0;
}


template < typename E, typename G >
long UploadEmbryo ( G *TotalBase, EmbryoType<E> &embryo, long  id_process_center_tree, long timeBegin, long timeEnd )
{
  try
  {
    //Upload the tree!!!
    std::set< long > times = embryo.GetTimes();
    for ( std::set< long >::iterator  timeIt = times.begin();timeIt != times.end();timeIt++ )
    {
      if ( timeBegin <= ( *timeIt ) && ( *timeIt ) <= timeEnd )
      {
        UploadCentersTreeExecute ( TotalBase, embryo, id_process_center_tree, *timeIt );
      }
    }
  }
  catch ( SQLException sqle )
  {
    std::cout << "Request to database failed: " << sqle.what();
    exit (-1);
  }
  return 0;
}


template < typename EPtr, typename G >
    long UpdateEmbryo ( G *TotalBase, EmbryoType<EPtr> &embryo, long  id_process_center_tree, long timeBegin, long timeEnd )
{
  try
  {
    //Update the tree!!!
    std::set< long > times = embryo.GetTimes();
    for ( std::set< long >::iterator  timeIt = times.begin();timeIt != times.end();timeIt++ )
    {
      if ( timeBegin <= *timeIt && *timeIt < timeEnd )
      {
        UpdateCentersTreeExecute ( TotalBase, embryo, id_process_center_tree, *timeIt );
      }
    }
  }
  catch ( SQLException sqle )
  {
    std::cout << "Request to database failed: " << sqle.what();
    exit (-1);
  }
  return 0;
}




template <typename EPtr, typename G>
    long UpdateCentersTreeExecute ( G *TotalBase, EmbryoType<EPtr> &embryo, long id_process_center_tree, long time )
{
  EmbryoStateType< EPtr > *embryoState;
  embryoState = embryo.GetEmbryoState ( time );

  std::vector < long > vec_id_center;
  std::vector < long > vec_id_center_mother;
  std::vector < sb2 > vec_id_center_motherNull;
  std::vector < long > vec_id_process_center_tree;

  long i = 0;
  for ( typename EmbryoStateType<EPtr>::iterator it = embryoState->begin();it != embryoState->end();++it )
  {

    EPtr cell = *it;
    long id_center = cell->GetIdDb();
    long mother_id;


    std::list< EPtr > p_mother;
    embryo.GetPredecessor ( cell, p_mother );
    if ( p_mother.size() == 0 )
    {
      mother_id = -1;
    }
    else
    {
      EPtr cellMother = * ( p_mother.begin() );
      mother_id = cellMother->GetIdDb();
    }

    i++;

    try
    {
      Statement *stmt = TotalBase->GetConnection()->createStatement();
      
      
      stmt = TotalBase->GetConnection()->createStatement("UPDATE z_center_tree set id_center_mother=" + LongToString ( mother_id ) + " where id_process_center_tree= " + LongToString ( id_process_center_tree ) + " and id_center = " + LongToString(id_center) );
      stmt->execute();
      TotalBase->GetConnection()->terminateStatement(stmt);
    }

    catch ( SQLException sqle )
    {
      std::cout << "Request to database failed: " << sqle.what();
      exit (-1);
    }
  }
  std::cout << "Tree Updated " << i << " " << "id_process_center_tree: " << id_process_center_tree << "Time: " << time <<  std::endl;
  return 0;
}


template <typename E, typename G>
long UpdateCentersTreeExecute2 ( G *TotalBase, EmbryoType<E> &embryo, long id_process_center_tree, long time )
{
  EmbryoStateType< E > *embryoState;
  embryoState = embryo.GetEmbryoState ( time );

  std::vector < long > vec_id_center;
  std::vector < long > vec_time;
  std::vector < long > vec_cell_number;
  std::vector < long > vec_id_center_mother;
  std::vector < sb2 > vec_id_center_motherNull;
  std::vector < long > vec_id_process_center_tree;
  std::vector < double > vec_x;
  std::vector < double > vec_y;
  std::vector < double > vec_z;
  std::vector < double > vec_vx;
  std::vector < double > vec_vy;
  std::vector < double > vec_vz;
  std::vector < double > vec_track_probability;
  std::vector < double > vec_center_probability;

  long i = 0;
  for ( typename EmbryoStateType<E>::iterator it = embryoState->begin();it != embryoState->end();++it )
  {
    E* cell = *it;
    vec_id_center.push_back ( cell->GetIdDb() );
    vec_time.push_back ( cell->GetT() );
    vec_cell_number.push_back ( cell->GetCenterNumber() );

    std::list< E* > p_mother;
    embryo.GetPredecessor ( cell, p_mother );
    if ( p_mother.size() == 0 )
    {
      vec_id_center_mother.push_back ( -1 );
      vec_id_center_motherNull.push_back ( 0 );
    }
    else
    {
      E* cellMother = * ( p_mother.begin() );
      vec_id_center_mother.push_back ( -10 );
      vec_id_center_motherNull.push_back ( 0 );
    }

    vec_id_process_center_tree.push_back ( id_process_center_tree );
    vec_x.push_back ( cell->GetX() );
    vec_y.push_back ( cell->GetY() );
    vec_z.push_back ( cell->GetZ() );
    vec_vx.push_back ( cell->GetDX() );
    vec_vy.push_back ( cell->GetDY() );
    vec_vz.push_back ( cell->GetDZ() );
    vec_track_probability.push_back ( cell->GetProb() );
    vec_center_probability.push_back ( cell->GetProb() );
    i++;
  }


  try
  {
    Statement *stmt = TotalBase->GetConnection()->createStatement();

    stmt->setSQL ( "UPDATE z_center_tree set time = :2, cell_number= :3, id_center_mother= :4, x= :6 ,y = :7 ,z= :8 ,vx= :9 ,vy= :10, vz= :11, track_probability= :12 ,center_probability= :13 where id_center= :1 and id_process_center_tree= :5 " );

    stmt->setSQL ( "UPDATE z_center_tree set  id_center_mother= :1  where  id_process_center_tree= :2 " );

    std::vector< ub2 > idLen ( i );
    std::vector< ub2 > idLenDouble ( i );

    for ( int j = 0; j < i; j++ )
    {
      idLen[j] = sizeof ( vec_time[j] );
      idLenDouble[j] = sizeof ( double );
    }

//    stmt->setDataBuffer ( 1, &vec_id_center.front(), OCCIINT, sizeof ( vec_id_center[0] ), &idLen.front() );
//    stmt->setDataBuffer ( 2, &vec_time.front(), OCCIINT, sizeof ( vec_time[0] ), &idLen.front() );
//    stmt->setDataBuffer ( 3, &vec_cell_number.front(), OCCIINT, sizeof ( vec_cell_number[0] ), &idLen.front() );
    stmt->setDataBuffer ( 1, &vec_id_center_mother.front(), OCCIINT, sizeof ( vec_id_center_mother[0] ), &idLen.front(), &vec_id_center_motherNull.front() );
    stmt->setDataBuffer ( 2, &vec_id_process_center_tree.front(), OCCIINT, sizeof ( vec_id_process_center_tree[0] ), &idLen.front() );
//    stmt->setDataBuffer ( 6, &vec_x.front(), OCCIFLOAT, sizeof ( vec_x[0] ), &idLenDouble.front() );
//    stmt->setDataBuffer ( 7, &vec_y.front(), OCCIFLOAT, sizeof ( vec_y[0] ), &idLenDouble.front() );
//    stmt->setDataBuffer ( 8, &vec_z.front(), OCCIFLOAT, sizeof ( vec_z[0] ), &idLenDouble.front() );
//    stmt->setDataBuffer ( 9, &vec_vx.front(), OCCIFLOAT, sizeof ( vec_vx[0] ), &idLenDouble.front() );
//    stmt->setDataBuffer ( 10, &vec_vy.front(), OCCIFLOAT, sizeof ( vec_vy[0] ), &idLenDouble.front() );
//    stmt->setDataBuffer ( 11, &vec_vz.front(), OCCIFLOAT, sizeof ( vec_vz[0] ), &idLenDouble.front() );
//    stmt->setDataBuffer ( 12, &vec_track_probability.front(), OCCIFLOAT, sizeof ( ( vec_track_probability[0] ) ), &idLenDouble.front() );
//    stmt->setDataBuffer ( 13, &vec_center_probability.front(), OCCIFLOAT, sizeof ( vec_center_probability[0] ), &idLenDouble.front() );

    stmt->executeArrayUpdate ( i );       // data for two rows is inserted.
    TotalBase->GetConnection()->terminateStatement ( stmt );

  }
  catch ( SQLException sqle )
  {
    std::cout << "Request to database failed: " << sqle.what();
    exit (-1);
  }

  std::cout << "Tree Updated " << i << " " << "id_process_center_tree: " << id_process_center_tree << "Time: " << time <<  std::endl;
  return 0;
}


template <typename EPtr, typename G>
long UploadCentersTreeExecute ( G *TotalBase, EmbryoType<EPtr> &embryo, long id_process_center_tree, long time )
{
  EmbryoStateType< EPtr > *embryoState;
  embryoState = embryo.GetEmbryoState ( time );

  std::vector < long > vec_id_center;
  std::vector < long > vec_id_center_origin;
  std::vector < long > vec_time;
  std::vector < long > vec_cell_number;
  std::vector < long > vec_id_center_mother;
  std::vector < sb2 > vec_id_center_motherNull;
  std::vector < long > vec_id_process_center_tree;
  std::vector < double > vec_x;
  std::vector < double > vec_y;
  std::vector < double > vec_z;
  std::vector < double > vec_vx;
  std::vector < double > vec_vy;
  std::vector < double > vec_vz;
  std::vector < double > vec_track_probability;
  std::vector < double > vec_center_probability;

  
//  std::fstream tmp ("/tmp/id_centers.txt");

  long i = 0;
  for ( typename EmbryoStateType<EPtr>::iterator it = embryoState->begin();it != embryoState->end();++it )
  {
    EPtr cell = *it;
    
//    tmp << cell->GetIdDbNew() << std::endl;
    
    vec_id_center.push_back ( cell->GetIdDbNew() );
    vec_id_center_origin.push_back ( cell->GetIdDb() );
    vec_time.push_back ( cell->GetT() );
    vec_cell_number.push_back ( cell->GetCenterNumber() );

    std::list< EPtr > p_mother;
    embryo.GetPredecessor ( cell, p_mother );
    if ( p_mother.size() == 0 )
    {
      vec_id_center_mother.push_back ( -1 );
      vec_id_center_motherNull.push_back ( 0 );
      
      vec_track_probability.push_back ( -1 );

    }
    else
    {
      EPtr cellMother = * ( p_mother.begin() );
      vec_id_center_mother.push_back ( cellMother->GetIdDbNew() );
      vec_id_center_motherNull.push_back ( 0 );
      
      EdgePropertyType edgeProb = embryo.GetEdgeProperty(cellMother,cell);
      vec_track_probability.push_back ( edgeProb.GetProbability() );

    }

    vec_id_process_center_tree.push_back ( id_process_center_tree );
    vec_x.push_back ( cell->GetX() );
    vec_y.push_back ( cell->GetY() );
    vec_z.push_back ( cell->GetZ() );
    vec_vx.push_back ( cell->GetDX() );
    vec_vy.push_back ( cell->GetDY() );
    vec_vz.push_back ( cell->GetDZ() );
    vec_center_probability.push_back ( cell->GetProb() );
    i++;
  }


  try
  {

    Statement *stmt = TotalBase->GetConnection()->createStatement();

    stmt->setSQL ( "INSERT INTO z_center_tree (id_center,id_center_origin,time,cell_number,id_center_mother,id_process_center_tree,x,y,z,vx,vy,vz,track_probability,center_probability ) VALUES (:1,:2,:3,:4,:5,:6,:7,:8,:9,:10,:11,:12,:13,:14)" );

    std::vector< ub2 > idLen ( i );
    std::vector< ub2 > idLenDouble ( i );

    for ( int j = 0; j < i; j++ )
    {
      idLen[j] = sizeof ( vec_time[j] );
      idLenDouble[j] = sizeof ( double );
    }

    stmt->setDataBuffer ( 1, &vec_id_center.front(), OCCIINT, sizeof ( vec_id_center[0] ), &idLen.front() );
    stmt->setDataBuffer ( 2, &vec_id_center_origin.front(), OCCIINT, sizeof ( vec_id_center_origin[0] ), &idLen.front() );
    stmt->setDataBuffer ( 3, &vec_time.front(), OCCIINT, sizeof ( vec_time[0] ), &idLen.front() );
    stmt->setDataBuffer ( 4, &vec_cell_number.front(), OCCIINT, sizeof ( vec_cell_number[0] ), &idLen.front() );
    stmt->setDataBuffer ( 5, &vec_id_center_mother.front(), OCCIINT, sizeof ( vec_id_center_mother[0] ), &idLen.front(), &vec_id_center_motherNull.front() );
    stmt->setDataBuffer ( 6, &vec_id_process_center_tree.front(), OCCIINT, sizeof ( vec_id_process_center_tree[0] ), &idLen.front() );
    stmt->setDataBuffer ( 7, &vec_x.front(), OCCIFLOAT, sizeof ( vec_x[0] ), &idLenDouble.front() );
    stmt->setDataBuffer ( 8, &vec_y.front(), OCCIFLOAT, sizeof ( vec_y[0] ), &idLenDouble.front() );
    stmt->setDataBuffer ( 9, &vec_z.front(), OCCIFLOAT, sizeof ( vec_z[0] ), &idLenDouble.front() );
    stmt->setDataBuffer ( 10, &vec_vx.front(), OCCIFLOAT, sizeof ( vec_vx[0] ), &idLenDouble.front() );
    stmt->setDataBuffer ( 11, &vec_vy.front(), OCCIFLOAT, sizeof ( vec_vy[0] ), &idLenDouble.front() );
    stmt->setDataBuffer ( 12, &vec_vz.front(), OCCIFLOAT, sizeof ( vec_vz[0] ), &idLenDouble.front() );
    stmt->setDataBuffer ( 13, &vec_track_probability.front(), OCCIFLOAT, sizeof ( ( vec_track_probability[0] ) ), &idLenDouble.front() );
    stmt->setDataBuffer ( 14, &vec_center_probability.front(), OCCIFLOAT, sizeof ( vec_center_probability[0] ), &idLenDouble.front() );

    stmt->executeArrayUpdate ( i );       // data for two rows is inserted.
    TotalBase->GetConnection()->terminateStatement ( stmt );

  }
  catch ( SQLException sqle )
  {
    std::cout << "Request to database failed: " << sqle.what();
    exit (-1);
  }

  std::cout << "Tree Uploaded, Time :"<< time << " Number:"<< i << " " << "id_process_center_tree: " << id_process_center_tree <<  std::endl;
  return 0;
}






template <typename G>
std::vector< int > GetNextVector ( G *TotalBase, long size )
{
  std::vector< int >  resu;
  long id_toAsk = size;

  while ( id_toAsk > 0 )
  {
    const int cant = 999;
    std::vector< int >  ids;
    ids = GetIdsVector ( TotalBase, cant );
    std::copy ( ids.begin(), ids.end(), std::back_inserter ( resu ) );
    id_toAsk -= cant;
  }
  resu.resize ( size );
  return resu;
}







template <typename E, typename G>
        long DeleteCells ( G *TotalBase, long id_process_center_tree, std::set< E* > cellsToDelete)
{
  
  long i = 0;
  for ( typename std::set<E*>::iterator it = cellsToDelete.begin();it != cellsToDelete.end();++it )
  {
    
    E* cell = *it;
    long id_center = cell->GetIdDb();

    i++;

    try
    {
      Statement *stmt = TotalBase->GetConnection()->createStatement();
      
      stmt = TotalBase->GetConnection()->createStatement("delete from z_center_tree where id_process_center_tree = " + LongToString ( id_process_center_tree ) + " and id_center = " + LongToString(id_center) );
      stmt->execute();
      TotalBase->GetConnection()->terminateStatement(stmt);
    }

    catch ( SQLException sqle )
    {
      std::cout << "Request to database failed: " << sqle.what();
      exit (-1);
    }
  }
  std::cout << "Delete cells" << i << " " << "id_process_center_tree: " << id_process_center_tree <<  std::endl;
  return 0;
}


#endif
