/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */

#ifndef __EmbryoType_h
#define __EmbryoType_h


#include "EmbryoStateType.h"
#include <list>
#include <vector>
#include <set>
#include <functional>
#include <math.h>

#include <boost/function.hpp>
#include <boost/bind.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/foreach.hpp>

//#include "ImageProvider.h"
#include "EdgePropertyType.h"
#include "BoxType3d.h"
#include "SetPointsWithLocator.h"


#include <functional>

/*!
 * @file EmbryoType.h
 * @brief contains the EmbryoType class and the EmbryoType::iterator class.
 * @see EmbryoType
 * @see EmbryoType::iterator
 */

/*! 
 * @class EmbryoType "EmbryoType.h"
 * @brief An object representing an embryo in this calculation
 * @tparam EPtr A cell type like CelluleType or a descendant.
*/
template < typename EPtr>
class EmbryoType
{
	public:
		typedef EmbryoType Self;
  
        public:
		/*! Default constructor */
		EmbryoType() {};
		
		/*! Destructor */
		~EmbryoType() {};
	public:
		/*!
		 * Add a Cell to the embryo.
		 * @param p_Cell A Cell of type EPtr
		 */
		virtual void AddCell ( EPtr p_Cell ) = 0;
		/*!
		 * Link a cell other
		 * @param from_Cell cell from time t
		 * @param to_Cell cell from time t+n
		 * @return
		 */
                
                /*! Number of cells */
                virtual long size ( );

                
		virtual bool AddRelationCell ( EPtr from_Cell, EPtr to_Cell ) = 0;
		/*!
		 * Remove the links between cells
		 * @param from_Cell cell from time t
		 * @param to_Cell cell from time t+n
		 */
		virtual bool RemoveRelationCell ( EPtr from_Cell, EPtr to_Cell ) = 0;
		/**
		 * Remove all the relation of the cells of time step 
		 * @param time with cells of next timesteps
		 * @param time add the cell to this set. todo: what does this mean??
		 */
		virtual void RemoveAllRelations ( long time );

//                virtual bool ExistCell ( EPtr a_cell) = 0;
		
		/*!
		 * detects if there's an edge on the cell graph for the two input cells
		 * @param from_Cell the origin cell
		 * @param to_Cell the destination cell
		 * @return returns true if there is an edge in the cell graph between both cells
		 */
                virtual bool ExistEdge ( EPtr from_Cell, EPtr to_Cell ) = 0;


		/*!
		 * How many links p_Cell has in the nexts timesteps
		 * @param p_Cell A cell of type EPtr
		 * @return the number of successors
		 */
		virtual long  GetNumberSuccessor ( EPtr p_Cell );

		/*!
		 * How many links p_Cell has in the previous timesteps
		 * @param p_Cell A cell of type EPtr
		 * @return the number of predecessors
		 */
		virtual long  GetNumberPredecessor ( EPtr p_Cell );
                void GetMaxMin (std::vector<double> &min,std::vector<double> &max );
                
                
//                virtual void  AsignEdgeProperty ( ) = 0;
                virtual EdgePropertyType GetEdgeProperty ( EPtr from_Cell, EPtr to_Cell) = 0;
                virtual void SetEdgeProperty ( EPtr from_Cell, EPtr to_Cell, EdgePropertyType edgeProperty ) = 0;
                

		/*!
		 * Return the cells that are linked in the previous time step
		 * @param p_Cell a standard Cell type that we want the predecessors of.
		 * @param p_mother add the cells to this set
		 */
		virtual void GetPredecessor ( EPtr p_Cell, std::list< EPtr > &p_mother ) = 0;
		/*!
		 * Return the cells that are linked in the next time step
		 * @param p_Cell a standard Cell type that we want the successors of.
		 * @param p_daughters add the cells to this set.
		 */
		virtual void GetSuccessor ( EPtr p_Cell, std::list< EPtr > &p_daughters ) = 0;
		/*!
		 * Return a EmbryoStateType of the time p_time
		 * @param p_time the timestep to at which we want the the state of the embryo
		 * @see EmbryoStateType
		 */
                virtual EmbryoStateType< EPtr >* GetEmbryoState ( long p_time ) = 0;
		
		/*!
		 * Same as GetEmbryoState but using array indexing syntax
		 * @see GetEmbryoState
		 */
                virtual EmbryoStateType< EPtr >& operator[] (long p_time );
 
		/**
		 * Returns the set of timesteps of the cells that are inside the embryo
		 * @return
		 */
                virtual const std::set< long > &GetTimes() = 0;
                long  GetTimeMax();
                long  GetTimeMin();

                bool PredecesorMother( EPtr p_Cell );
                void RemoveEdgeSuccesors( EPtr p_Cell );
                void RemoveEdgePredecessors( EPtr p_Cell );

                /*!
		 * True if the cell is near to it's previous position
		 * @param cell a cell of EPtr type
		 */
                bool GoodSuccesor(EPtr cell);
		
		/*!
		 * True if the cell is near to it's next position
		 * @param cell a cell of EPtr type
		 */
                bool GoodPredesesor(EPtr cell);
                
                /*!
		 * Determine if the cell is an orphan or not
		 * @param a_Cell a standard cell of EPtr type.
		 */
                bool HasMother(EPtr a_Cell);
		
		/*!
		 * Determine if the cell is a parent or not
		 * @param a_Cell a standard cell of EPtr type.
		 */
                bool HasDaughters(EPtr a_Cell);
                
		/*!
		 * Returns the set of cells at time that are orphaned
		 * @param time the timestep to search in
		 */
                virtual std::set< EPtr > GetOrphansCellsCells ( long time );
		
		                
		/*!
		 * Dividing cells
		 * @param time the timestep to search in
		 * @return set of cells that are dividing
		 */
                virtual std::set< EPtr > GetDividingCells ( long time );
		/*!
		 * Moving cells
		 * @param time the timestep to search in
		 * @return set of cells that are moving
		 */
		virtual std::set< EPtr > GetMovingCells ( long time );
		/*!
		 * Dieing cells
		 * @param time the timestep to search in
		 * @return set of cells that are dieing
		 */
		virtual std::set< EPtr > GetDyingCells ( long time );
                
                virtual std::set< EPtr > GetAllCells (  );
                
                virtual bool IsNewBourne (EPtr cell );
		
		/*!
		 * Newborn cells
		 * @param time the timestep to search in
		 * @return set of cells that are Newborn
		 */
                virtual std::set< EPtr > GetNewbornCells ( long time );
		/*!
		 * Life of the cell
		 * @param a_Cell a cell of type EPtr
		 * @return a list of cells, from the time it born until it dies or divides. The set is time ordered.
		 */
		virtual std::list< EPtr > GetCellLife ( EPtr a_Cell );
		
		/*!
		 * Sister cell
		 * @param p_Cell A cell type EPtr
		 * @return returns the sister of the passed cell
		 */
                virtual EPtr GetSister( EPtr p_Cell );

                bool GetCell( double x, double y, double z, long t, EPtr &cell );

		/**
		 * Direction of the cell. Using the mother cell as reference
		 * @param a_Cell The cell to determine the direction of
		 * @return a vector (the scientific kind, not the computer science kind) 
		 */
		virtual std::vector< double > GetDirection ( EPtr a_Cell );
                
                /*!
                 * Nearest cells in different times
                 * @param position cartesian coordinates array, {x, y, z}
                 * @param nn count cells per times
                 * @param timeStart from (>=)
                 * @param timeEnd until (<=)
                 * @return the set of nearest cells
                 */
                virtual std::set< EPtr > GetNearestCells(double position[3], long nn, long timeStart, long timeEnd);
                /**
                 * Nearest cell in different times
                 * @param position cartesian
                 * @param nn count cells per times
                 * @param timeStart from (>=)
                 * @param timeEnd until (<=)
                 * @return std::set< EPtr >
                 */
                virtual std::set< EPtr > GetNearestCells (EPtr cell, long nn,long timeStart, long timeEnd );
                virtual std::set< EPtr > GetNearestCells (EPtr cell, long nn,long time );

		/*!
		 * Speed of the cell, using the mother cell as reference.
		 * @param a_Cell a cell of type EPtr
		 * @return the speed in units dependent on the spacing of the image
		 */
                virtual double GetSpeed ( EPtr a_Cell );

                /*!
		 * Gets the difference between timesteps in minutes
		 */	
		virtual double GetTimeStep ( );
		
		/**
		 * Set the real time between timesteps, calculated linearly from the start and end times provided
		 * @param timeStepA an arbitrary start timestep within our dataset
		 * @param minutesA the start time in real time at timeStepA
		 * @param timeStepB an arbitrary end timestep within our dataset
		 * @param minutesB the end time in real time at timeStepB
		 */
		virtual void SetRealTime ( long timeStepA, double minutesA, long timeStepB, double minutesB ) ;
		
		/**
		 * Get the real time at timeStep  
		 * @param timeStep 
		 * @return minutes
		 */
		virtual double GetRealTime ( long timeStep ) ;
		
		/*! returns the number of cells at timestep time */
                virtual long GetNumberCells( long time );
		
		/*! returns the number of dying cells at timestep time */
                virtual long GetNumberDyingCells( long time );
		
		/*! returns the number of mother cells at timestep time */
                virtual long GetNumberMotherCells( long time );
		
		/*! returns the number of newborn cells at timestep time */
                virtual long GetNumberNewbourneCells( long time );
		
		/*! returns the number of moving cells at timestep time */
                virtual long GetNumberMovingCells( long time );
                
                /*!
		 * get the cells within a box that are a part of the dataset for a processor.
		 * @param box the box to look in for cells
		 * @see BoxType3d
		 * @param regionNumber the processor number whose dataset we're looking in for cells
		 */
		virtual std::set< EPtr > GetCells(BoxType3d &box,long regionNumber);
		
		/*!
		 * Get all cells from InitTime to EndTime. The search includes cells from the times at InitTime and EndTime
		 * @param InitTime the starting timestep
		 * @param EndTime the ending timestep
		 */
                virtual std::set< EPtr > GetCells(long InitTime, long EndTime);
                
		/*!
		 * Definition of == for value equality of this EmbryoType vs another
		 * @param other1 the embryotype to compare this one to
		 */
                bool operator==(const EmbryoType< EPtr >& other1) const
                {
                  return(time_m == other1.time_m &&
                      time_b == other1.time_b);
                }

		/*!
		 * Get the real time minutes of the cell from the time it bourn until this cell.
		 * @param a_Cell a specific cell object representing a cell at a certain point in time
		 * @return minutes
		 */
		double GetRealTimeLiving ( EPtr a_Cell ) ;

		/*!
		 * Get the time steps of the cell from the time it bourn until this cell.
		 * @param a_Cell a specific cell object representing a cell at a certain point in time
		 * @return timesteps
		 */
                long GetTimeStepsLiving ( EPtr a_Cell ) ;

		/*!
		 * Sets the m_size and m_border settings of this object
		 */
                void SetDimension ( double size[6],double border){m_border = border;for (long i = 0;i<6;++i){m_size[i] = size[i];};};
		
		/*! getter that fills size with the contents of m_size */
                void GetDimension ( double size[6]){for (long i = 0;i<6;++i){size[i] = m_size[i];};}; 
		
		/*! getter for m_border */
                double GetBorder (){return m_border;};
		
		/*! returns true if pCell is */
                bool CellAtBorder(EPtr pCell);
                
		/*!
                 * Rename the name of the cells starting from the name 0. The cell has the same name from the time it born until it divides or disapears
                 */
                void RenameCells();
                
		/*! Dummy Function, does nothing at the moment */
                void SetImagePattern(std::string p_pattern){ 
//                  imageProvider.SetPattern(p_pattern); 
                };
//                ImageProvider *GetImageProvider(){ return &this->imageProvider; };
                
		friend class boost::serialization::access;
		/*! serialization function for boost serialization, not meant to be called by users */
		template<class Archive> void serialize ( Archive & ar, const unsigned int version )
		{
			ar & time_m;
			ar & time_b;
//                        ar & imageProvider;
		}

	private:
		double time_m;
		double time_b;
		/*! padding for the cartesian box limits, but unused at the moment */
                double m_border;
		
		/*! the cartesian box of this embryo where _s indicates the start of the box for an axis and _e indicates the end
		 * {x_s, x_e, y_s, y_e, z_s, z_e}
		 */
                double m_size[6];
//                ImageProvider imageProvider;
        public:
		/*!
		 * @class EmbryoType::iterator "EmbryoType.h"
		 * @brief allows one to iterate over the cells in this EmbryoType as time advances. Cells are arbitrarily ordered, but they are grouper per timestamp and you traverse the groups in timestep order.
		 */
                class iterator :
                    public std::iterator<std::forward_iterator_tag, EPtr> 
                    {
                      public:
                        iterator(EmbryoType *embryo,typename EmbryoStateType< EPtr >::iterator it,long time):embryo_(embryo),it_(it),time_(time),empty_( embryo->GetTimes().size()==0)
                        {
                          
                        };
                        
                        
                        ~iterator() {};

      // The assignment and relational operators are straightforward
                        iterator& operator=(const iterator& other)
                        {
                          it_ = other.it_;
                          time_ = other.time_;
                          embryo_ = other.embryo_;
                          empty_ = other.empty_ ;
                          return(*this);
                        }

                        bool operator==(const iterator& other)
                        {
                          if (empty_ == other.empty_ && empty_)
                            return true;
                          return(it_ == other.it_ && time_ == other.time_ && embryo_ == other.embryo_);
                        }

                        bool operator!=(const iterator& other)
                        {
                          if (empty_ == other.empty_ && empty_)
                            return false;

                          return(it_ != other.it_ || time_ != other.time_);
                        }

      // Update my state such that I refer to the next element 
                        iterator& operator++()
                        {
                          
                          std::set< long > times = embryo_->GetTimes();
                          long lastTime = *(times.rbegin());
                          
                          it_++;
                          do
                          {
                            if (it_ == embryo_->GetEmbryoState(time_)->end() && time_== lastTime)
                            {
                              break;
                            }
                            if (it_ == embryo_->GetEmbryoState(time_)->end() && time_!= lastTime)
                            {
                              typename std::set< long >::iterator timesIt2, timesItEnd;
                              timesIt2 = times.find ( time_ );
                              timesIt2++;
                              time_ = *timesIt2;
                              it_ = (embryo_->GetEmbryoState(time_)->begin());
                            }
                            
                            if (embryo_->GetTimeStepsLiving ( *it_)  == 0)
                            {
                              break;
                            }
                            
                            it_++;
                          }
                          while (true);
                          
                          return(*this);
                        }
                        
                        
                        iterator operator++(int)
                        {
                          iterator tmp(*this);
                          ++(*this);
                          return(*this);
                        }
                        
                        
                        
                        EPtr operator*()
                        {
                          EPtr cell = *it_; 
                          
                          return(*it_);
                        }

                        EPtr operator->()
                        {
                          return(*it_);
                        }

                      private:
                        typename EmbryoStateType< EPtr >::iterator it_;
                        long time_;
                        bool empty_;
                        EmbryoType< EPtr > *embryo_;
                    };

                    iterator begin()
                    {
                      std::set< long > times = this->GetTimes();
                      
                      BOOST_FOREACH(long time,times)
                      {
                        if (!(this->GetEmbryoState(time)->size())==0)
                        {
                          iterator tmp (this,this->GetEmbryoState(time)->begin(),time);
                          return(tmp);
                        }
                      }
                      
                      return(end());
                    }

                    iterator end()
                    {
                      std::set< long > times = this->GetTimes();
                      if (times.size() == 0)
                      {
                        SetPointsWithLocator < EPtr > embryoState;
                        return(iterator(this,embryoState.end(),0));
                      }
                      long lastTime = *(times.rbegin());
                      
                      return(iterator(this,this->GetEmbryoState(lastTime)->end(),lastTime));
                    }
};

template < typename EPtr>
bool EmbryoType< EPtr >::CellAtBorder(EPtr pCell)
{
  
  if (pCell->GetX() > (m_size[0] + m_border) &&
      pCell->GetX() < (m_size[1] - m_border) &&
     
      pCell->GetY() > (m_size[2] + m_border) &&
      pCell->GetY() < (m_size[3] - m_border) &&

      pCell->GetZ() > (m_size[4] + m_border) &&
      pCell->GetZ() < (m_size[5] - m_border)
     )
  {
    return false;
  }
  else
  {
    return true;
  }


}


template < typename EPtr>
bool compareTimeCell ( EPtr  o1, EPtr o2 )
{
	return ( o1->GetT() < o2->GetT() );
}


template < typename EPtr>
    std::set< EPtr > EmbryoType< EPtr >::GetAllCells (  )
{
  std::set< EPtr > cellsSet;
  std::set< long > times = this->GetTimes();

  
  for (std::set< long >::iterator timeIt = times.begin(); timeIt!= times.end(); ++timeIt)
  {
    long time = *timeIt;
    
     EmbryoStateType < EPtr > *embryoState = this->GetEmbryoState ( time );
     if (embryoState!=NULL)
     {
       for ( typename EmbryoStateType< EPtr >::iterator it = embryoState->begin(); it != embryoState->end(); ++it )
       {
         EPtr aCell;
         aCell = *it;
        
         cellsSet.insert(aCell);
       }
     }
  }
  
  return cellsSet;
}

template < typename EPtr>
    std::list< EPtr > EmbryoType< EPtr >::GetCellLife ( EPtr a_Cell )
{

	std::list< EPtr > cellLife;
	EPtr cell = a_Cell;
	do
	{
		cellLife.push_back ( cell );

		std::list< EPtr > p_mother;
		this->GetPredecessor ( cell,p_mother );

		if ( p_mother.size() == 0 )
		{
			//the just born
			break;
		}
		else
		{
			cell = * ( p_mother.begin() );
			// the cell is a mother
			std::list< EPtr > p_daughters;
			GetSuccessor ( cell,p_daughters );

			if ( p_daughters.size() == 2 )
			{
				break;
			}
		}
	}
	while ( 1 );

	// the cell is a mother
	std::list< EPtr > p_daughters;
	cell = a_Cell;
	GetSuccessor ( cell,p_daughters );
	if ( p_daughters.size() ==1 )
	{
		do
		{
			cell = * ( p_daughters.begin() );
			cellLife.push_back ( cell );

			p_daughters.clear();
			GetSuccessor ( cell,p_daughters );
		}
		while ( p_daughters.size() ==1 );
	}

	cellLife.sort ( compareTimeCell< EPtr > );


	return cellLife;
}



    template < typename EPtr>
        std::set< EPtr > EmbryoType< EPtr >::GetOrphansCellsCells ( long time )
{
  EmbryoStateType< EPtr > *embryoState;
  embryoState = this->GetEmbryoState ( time );

  std::set< EPtr > cellSet;

  typename EmbryoStateType< EPtr >::iterator it;
  for ( it = embryoState->begin();it != embryoState->end();++it )
  {
    std::list< EPtr > p_mother;
    GetPredecessor ( *it,p_mother );

    if ( p_mother.size() == 0 )
    {
      cellSet.insert ( *it );
    }
  }
  return cellSet;
}
    
    
    
template < typename EPtr>
std::set< EPtr > EmbryoType< EPtr >::GetDividingCells ( long time )
{
	EmbryoStateType< EPtr > *embryoState;
	embryoState = this->GetEmbryoState ( time );

	std::set< EPtr > cellSet;

	typename EmbryoStateType< EPtr >::iterator it;
	for ( it = embryoState->begin();it != embryoState->end();++it )
	{
		std::list< EPtr > p_daughters;
		GetSuccessor ( *it,p_daughters );

		if ( p_daughters.size() ==2 )
		{
			cellSet.insert ( *it );
		}
	}

	return cellSet;
}

template < typename EPtr>
std::set< EPtr > EmbryoType< EPtr >::GetMovingCells ( long time )
{
	EmbryoStateType< EPtr > *embryoState;
	embryoState = this->GetEmbryoState ( time );

	std::set< EPtr > cellSet;

	typename EmbryoStateType< EPtr >::iterator it;
	for ( it = embryoState->begin();it != embryoState->end();++it )
	{
		std::list< EPtr > p_daughters;
		GetSuccessor ( *it,p_daughters );

		if ( p_daughters.size() ==1 )
		{
			cellSet.insert ( *it );
		}
	}

	return cellSet;
}


template < typename EPtr>
std::set< EPtr > EmbryoType< EPtr >::GetDyingCells ( long time )
{
	EmbryoStateType< EPtr > *embryoState;
	embryoState = this->GetEmbryoState ( time );

	std::set< EPtr > cellSet;

	typename EmbryoStateType< EPtr >::iterator it;
	for ( it = embryoState->begin();it != embryoState->end();++it )
	{
		std::list< EPtr > p_daughters;
		GetSuccessor ( *it,p_daughters );

		if ( p_daughters.size() ==0 )
		{
			cellSet.insert ( *it );
		}
	}

	return cellSet;
}

template < typename EPtr>
std::set< EPtr > EmbryoType< EPtr >::GetNewbornCells ( long time )
{
	EmbryoStateType< EPtr > *embryoState;
	embryoState = this->GetEmbryoState ( time );

	std::set< EPtr > cellSet;

	typename EmbryoStateType< EPtr >::iterator it;
	for ( it = embryoState->begin();it != embryoState->end();++it )
	{
          if (IsNewBourne (*it))
		{
			cellSet.insert ( *it );
		}
	}
	return cellSet;
}


template < typename EPtr>
    bool EmbryoType< EPtr >::IsNewBourne ( EPtr cell )
{
  std::list< EPtr > p_mother;
  GetPredecessor ( cell,p_mother );

  if ( p_mother.size() ==0 )
  {
    return true;
  }
  else
  {
    std::list< EPtr > p_succ;
    GetSuccessor( *(p_mother.begin()),p_succ );
    if (p_succ.size()==2)
    {
      return true;
    }                  
  }
  return false;
}




template < typename EPtr>
    std::set< EPtr > EmbryoType< EPtr >::GetNearestCells (EPtr cell, long nn,long time )
{
  
  return this->GetNearestCells (cell, nn,time,time);
  
}
template < typename EPtr>
    std::set< EPtr > EmbryoType< EPtr >::GetNearestCells (EPtr cell, long nn,long timeStart, long timeEnd )
{
   double position[3];
   position[0] = cell->GetX();
   position[1] = cell->GetY();
   position[2] = cell->GetZ();

   return GetNearestCells (position, nn,timeStart, timeEnd );
}




template < typename EPtr>
    std::set< EPtr > EmbryoType< EPtr >::GetNearestCells (double position[3], long nn,long timeStart, long timeEnd )
{
  std::set< EPtr > movingCellsSet;
  std::set< long > times = this->GetTimes();

  
  for (long time=timeStart; time<= timeEnd; ++time)
  {
    
    if (times.find(time) != times.end())
    {
      std::list< EPtr > movingCellsList;
      EmbryoStateType < EPtr > *embryoState = this->GetEmbryoState ( time );
      if (embryoState!= NULL)
      {
        embryoState->GetNearestNCells ( position, nn, movingCellsList );
    
        std::copy ( movingCellsList.begin(), movingCellsList.end(), std::inserter ( movingCellsSet, movingCellsSet.begin() ) );
      }
    }
  }
  
  
  //Invalidate the locators that are far away than the one I'm working
  BOOST_FOREACH(long time,times)
  {
    if (fabs(timeStart - time)>100)
    {
      this->GetEmbryoState(time)->InvalidateLocator();
    }
  }

  
  return movingCellsSet;
}

template < typename EPtr>
std::vector<double> EmbryoType< EPtr >::GetDirection ( EPtr a_Cell )
{
	std::vector<double> direction ( 3 );

	std::list< EPtr > p_mother;
	GetPredecessor ( a_Cell,p_mother );

	if ( p_mother.size() == 1 )
	{
		direction[0] = a_Cell->GetX() - ( *p_mother.begin() )->GetX();
		direction[1] = a_Cell->GetY() - ( *p_mother.begin() )->GetY();
		direction[2] = a_Cell->GetZ() - ( *p_mother.begin() )->GetZ();
	}
	return direction;
}


template < typename EPtr>
double EmbryoType< EPtr >::GetSpeed ( EPtr a_Cell )
{

	std::vector<double> direction = this->GetDirection ( a_Cell );

        double speed = sqrt ( ( direction[0]*direction[0]+direction[1]*direction[1]+direction[2]*direction[2] ) );
        return speed / GetTimeStep();
}



template < typename EPtr>
void  EmbryoType< EPtr >::RemoveAllRelations ( long time )
{
	EmbryoStateType< EPtr > *embryoState;
	embryoState = this->GetEmbryoState ( time );

	typename EmbryoStateType< EPtr >::iterator s_CellIt;
	for ( s_CellIt = embryoState->begin();s_CellIt != embryoState->end();++s_CellIt )
	{
		std::list< EPtr > p_daughters;
		GetSuccessor ( *s_CellIt,p_daughters );

		typename std::list< EPtr >::iterator m_CellIt ;
		for ( m_CellIt = p_daughters.begin(); m_CellIt!= p_daughters.end(); ++m_CellIt )
		{
			this->RemoveRelationCell ( *s_CellIt,*m_CellIt );
		}
	}
}


template < typename EPtr>
long EmbryoType< EPtr >::GetNumberSuccessor ( EPtr p_Cell )
{
	std::list< EPtr > p_daugther;
	GetSuccessor ( p_Cell,p_daugther );

	return ( p_daugther.size() );
}


template < typename EPtr>
long EmbryoType< EPtr >::GetNumberPredecessor ( EPtr p_Cell )
{
	std::list< EPtr > p_mother;
	GetPredecessor ( p_Cell,p_mother );

	return ( p_mother.size() );
}



template < typename EPtr>
long EmbryoType< EPtr >::size (  ) 
{
  long numCells = 0;
  std::set< long > times = GetTimes() ;
  BOOST_FOREACH(long t, times)
  {
    numCells += GetNumberCells(t);
  }
  return numCells;
}

    
    template < typename EPtr>
        void EmbryoType< EPtr >::SetRealTime ( long timeStepA, double minutesA, long timeStepB, double minutesB )
{

  time_m = double ( minutesB - minutesA ) / double ( timeStepB - timeStepA );

  time_b = minutesB - time_m * timeStepB;

}

template < typename EPtr>
double EmbryoType< EPtr >::GetTimeStep ( )
{
  return time_m;
}

template < typename EPtr>
double EmbryoType< EPtr >::GetRealTime ( long timeStep )
{
	return ( timeStep*time_m+time_b );
}


template < typename EPtr>
double EmbryoType< EPtr >::GetRealTimeLiving ( EPtr a_Cell )
{
	long timesteps = this->GetTimeStepsLiving ( a_Cell );

	return timesteps * time_m;

}

template < typename EPtr>
long EmbryoType< EPtr >::GetTimeStepsLiving ( EPtr a_Cell )
{
	std::list< EPtr > cellLife;
	EPtr cell = a_Cell;
	do
	{
		cellLife.push_back ( cell );

		std::list< EPtr > p_mother;
		this->GetPredecessor ( cell,p_mother );

		if ( p_mother.size() == 0 )
		{
			//the just born
			break;
		}
		else
		{
			cell = * ( p_mother.begin() );
			// the cell is a mother
			std::list< EPtr > p_daughters;
			GetSuccessor ( cell,p_daughters );

			if ( p_daughters.size() == 2 )
			{
				break;
			}
		}
	}
	while ( 1 );

	return cellLife.size()-1;

}


template < typename EPtr>
long EmbryoType< EPtr >::GetNumberCells( long time )
{
  EmbryoStateType< EPtr > *embryoState;
  embryoState = this->GetEmbryoState ( time );

  return std::distance(embryoState->begin(), embryoState->end());
}

template < typename EPtr>
    long EmbryoType< EPtr >::GetNumberDyingCells( long time )
{
  std::set< EPtr > set = this->GetDyingCells(time );
  return set.size();
}
template < typename EPtr>
    long EmbryoType< EPtr >::GetNumberMotherCells( long time )
{
  std::set< EPtr > set = this->GetDividingCells ( time );
  return set.size();
}

template < typename EPtr>
    long EmbryoType< EPtr >::GetNumberNewbourneCells( long time )
{
  std::set< EPtr > set = this->GetNewbornCells(time );
  return set.size();
}


template < typename EPtr>
    void EmbryoType< EPtr >::RenameCells( )
{
  long name = 0;
  for (typename EmbryoType< EPtr >::iterator cellIt = this->begin(); cellIt != this->end();++cellIt)
  {
    EPtr aCell;
    aCell = *cellIt;
    
    std::list< EPtr > life = this->GetCellLife ( aCell );
    long i = 0;
    for (typename std::list< EPtr >::iterator lifeIt = life.begin(); lifeIt!=life.end();++lifeIt)
    {
      (*lifeIt)->SetName(name);
      i++;
    }
//    std::cout << i << std::endl;
    
    ++name;
  }
}







template < typename EPtr>
    long EmbryoType< EPtr >::GetNumberMovingCells( long time )
{
  std::set< EPtr > set = this->GetMovingCells(time );
  return set.size();
}

template < typename EPtr>
    inline
    long EmbryoType< EPtr >::GetTimeMax()
{
  std::set< long > times = this->GetTimes();
  return (*(times.rbegin()));
}
template < typename EPtr>
    inline
    long EmbryoType< EPtr >::GetTimeMin()
{
  std::set< long > times = this->GetTimes();
      return (*(times.begin()));
}



 template < typename EPtr>
     bool EmbryoType< EPtr >::PredecesorMother( EPtr p_Cell )
{
  std::list< EPtr > p_mother;
  GetPredecessor ( p_Cell,p_mother );

  if (p_mother.size()==1)
  {
    std::list< EPtr > p_dougther;
    GetSuccessor( *(p_mother.begin()),p_dougther );
    if (p_dougther.size()==2)
      return true;
  }


  return ( false );
}


template < typename EPtr>
    EPtr EmbryoType< EPtr >::GetSister( EPtr p_Cell )
{
  std::list< EPtr > life = this->GetCellLife ( p_Cell );

  std::list< EPtr > p_mother;
  GetPredecessor ( p_Cell,p_mother );
  
  if (p_mother.size()==1)
  {
    std::list< EPtr > p_dougther;
    GetSuccessor( *(p_mother.begin()),p_dougther );
    
    if (p_dougther.size()==2)
    {
      if (p_Cell->GetIdDb() == (*(p_dougther.begin()))->GetIdDb())
      {
        return *(p_dougther.rbegin());
      }
      else
      {
        return *(p_dougther.begin());
      }
    }
    else
    {
      throw 1;
    }
  }
  else
  {
    throw 1;
  }
}


template < typename EPtr>
    std::set<EPtr> EmbryoType< EPtr >::GetCells(long InitTime, long EndTime)
{
  std::set<EPtr> resu;
  for (std::set< long >::iterator timeIt = this->GetTimes().begin(); timeIt!= this->GetTimes().end(); ++timeIt)
  {
    if (InitTime <= *timeIt && *timeIt <= EndTime )
    {
      EmbryoStateType< EPtr > *embryoState = this->GetEmbryoState(*timeIt);
      for ( typename EmbryoStateType< EPtr >::iterator it = embryoState->begin(); it != embryoState->end(); ++it )
      {
        EPtr aCell= *it;
        resu.insert(aCell);
      }
    }
  }
  return resu;
}


template < typename EPtr>
    std::set<EPtr> EmbryoType< EPtr >::GetCells(BoxType3d &box,long regionNumber)
{
  std::set<EPtr> resu;
  for (std::set< long >::iterator timeIt = this->GetTimes().begin(); timeIt!= this->GetTimes().end(); ++timeIt)
  {
    long time = *timeIt;
    if (box.GetTimeInit()<=time && time <= box.GetTimeEnd())
    {
      EmbryoStateType< EPtr > *embryoState = this->GetEmbryoState(time);
      
      for ( typename EmbryoStateType< EPtr >::iterator it = embryoState->begin(); it != embryoState->end(); ++it )
      {
        EPtr aCell;
        aCell = *it;
        
        double x = aCell->GetX();
        double y = aCell->GetY();
        double z = aCell->GetZ();
        double t = aCell->GetT();
        
        if (box.IsInsideBuffer(x,y,z, regionNumber))
        {
          resu.insert(aCell);
        }
      }
    }
  }
  return resu;
}


template < typename EPtr>
    EmbryoStateType< EPtr > &EmbryoType< EPtr >::operator[] (long p_time )
{
  return *(GetEmbryoState ( p_time ));
}



template < typename EPtr>
    void EmbryoType< EPtr >::GetMaxMin (std::vector<double> &min,std::vector<double> &max )
{
  min.resize(3);
  min[0] = 10000000;
  min[1] = 10000000;
  min[2] = 10000000;
  
  max.resize(3);
  max[0] = -10000000;
  max[1] = -10000000;
  max[2] = -10000000;
  
  std::set< EPtr > allCells = GetAllCells ( );
  BOOST_FOREACH ( EPtr cell, allCells )
  {
    min[0] = (min[0]<cell->GetX())?min[0]:cell->GetX();
    min[1] = (min[1]<cell->GetY())?min[1]:cell->GetY();
    min[2] = (min[2]<cell->GetZ())?min[2]:cell->GetZ();
    
    max[0] = (max[0]>cell->GetX())?max[0]:cell->GetX();
    max[1] = (max[1]>cell->GetY())?max[1]:cell->GetY();
    max[2] = (max[2]>cell->GetZ())?max[2]:cell->GetZ();
  }
}


template < typename EPtr>
    void EmbryoType< EPtr >::RemoveEdgeSuccesors( EPtr p_Cell )
{
  std::list< EPtr > p_daughters;
  GetSuccessor ( p_Cell,p_daughters );
    for ( typename std::list< EPtr >::iterator p_daughtersIt = p_daughters.begin(); p_daughtersIt != p_daughters.end();++p_daughtersIt )
    {
      RemoveRelationCell(p_Cell,*p_daughtersIt);
    }
}

template < typename EPtr>
    void EmbryoType< EPtr >::RemoveEdgePredecessors( EPtr p_Cell )
{
  std::list< EPtr > p_mother;
  GetPredecessor( p_Cell,p_mother );
  for ( typename std::list< EPtr >::iterator p_motherIt = p_mother.begin(); p_motherIt != p_mother.end();++p_motherIt )
  {
    RemoveRelationCell(p_Cell,*p_motherIt);
  }
}


template < typename EPtr>
    bool EmbryoType< EPtr >::GetCell( double x, double y, double z, long t, EPtr &cell )
{
  double position[3] = {x,y,z};
  std::set< EPtr > listDestiny = this->GetNearestCells( position, 1, t,t );
  if (listDestiny.size()>0)
    {
      cell = *(listDestiny.begin());
      return  sqrt(pow(cell->GetX() - x,2) + pow(cell->GetY() - y,2) + pow(cell->GetZ() - z,2))<0.01;
    }
    else
    {
    return false;
    }

}


template < typename EPtr>
    bool EmbryoType< EPtr >::GoodPredesesor(EPtr a_Cell)
{
  if(GetSpeed(a_Cell)<(3))
  {
    return true;
  }
  else
  {
     return false;
  }
}


template < typename EPtr>
    bool EmbryoType< EPtr >::GoodSuccesor(EPtr a_Cell)
{
  std::list< EPtr > p_daughters;
  GetSuccessor ( a_Cell,p_daughters );

  if ( p_daughters.size() == 1 )
  {
    if(GetSpeed(*(p_daughters.begin()))<(3))
    {
      return true;
    }
  }
    
  return false;
}


template < typename EPtr>
    bool EmbryoType< EPtr >::HasMother(EPtr a_Cell)
{
  std::list< EPtr > cellLife;
  cellLife = this->GetCellLife ( a_Cell );

  long hasMother = this->GetNumberPredecessor ( *cellLife.begin() );
  return hasMother == 1;
}

template < typename EPtr>
    bool EmbryoType< EPtr >::HasDaughters(EPtr a_Cell)
{
  std::list< EPtr > cellLife;
  cellLife = this->GetCellLife ( a_Cell );

  long HasDaughters = this->GetNumberSuccessor( *cellLife.rbegin() );
  return HasDaughters == 2;
}


#endif
