/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef GraphCellSeccionType_h
#define GraphCellSeccionType_h

#include "GraphCellType.h"
#include <fstream>


#include <boost/config.hpp>
#include <boost/dynamic_bitset.hpp>
#include <boost/pending/indirect_cmp.hpp>
//#include <boost/range/irange.hpp>
#include <boost/serialization/map.hpp>
#include <boost/serialization/set.hpp>
#include <boost/pending/property_serialize.hpp>
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include <boost/pool/pool_alloc.hpp>
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include "boost/lambda/lambda.hpp"
#include "boost/lambda/bind.hpp"
#include "boost/lambda/construct.hpp"
#include <boost/foreach.hpp>


#include <strstream>
#include <sstream>


/**
	@author Camilo Melani
	\brief
 */




template < typename EPtr >
class GraphCellSeccionType : public GraphCellType < EPtr >
{
  public:
    typedef GraphCellSeccionType < EPtr >  Self;
    typedef GraphCellType < EPtr > Superclass;
  public:
    typedef typename std::pair<  long, EPtr > LinkBeforeType;
    typedef typename std::set<  LinkBeforeType > LinksBeforeType;
  private:

    friend class boost::serialization::access;
    template<class Archive> void serialize ( Archive & ar, const unsigned int version )
    {
      ar & boost::serialization::base_object<Superclass> ( *this );
      ar & linksBefore;
    }
  private:
    LinksBeforeType linksBefore;
  public:
    void RemoveCell ( long id );
    void RemoveCells ( std::set<long> &ids );
    bool AddRelationCell ( EPtr from_Cell, EPtr to_Cell );
    bool AddRelationIdDb ( long s, long v );
    bool RemoveRelationCell ( EPtr from_Cell, EPtr to_Cell );
    bool RemoveRelationIdDb ( long s, long v );
    int DegreeOut ( EPtr p_Cell );
    int DegreeIn ( EPtr p_Cell );

    LinksBeforeType GetLinksBefore ( );


    bool ExistEdge ( EPtr from_Cell, EPtr to_Cell );
    long GetPreviusIdDb ( EPtr to_Cell );
    bool  IncompleteLink ( EPtr cell );
    
    void GetMorphoFields ( std::set<EPtr> & cells, GraphCellSeccionType< EPtr > &newEmbryo );
    void CompleteMorphoField ( GraphCellSeccionType< EPtr > &newEmbryo );

  public:
    GraphCellSeccionType();
    ~GraphCellSeccionType();
};

template <typename EPtr>
GraphCellSeccionType< EPtr >::GraphCellSeccionType() : Superclass(), linksBefore ( LinksBeforeType() )
{
}


template <typename EPtr>
GraphCellSeccionType< EPtr >::~GraphCellSeccionType()
{
}

template <typename EPtr>
void GraphCellSeccionType< EPtr >::RemoveCell ( long id )
{
  Superclass::RemoveCell ( id );
  for ( typename LinksBeforeType::iterator it = linksBefore.begin();it != linksBefore.end(); ++it )
  {
    if ( it->first == id )
    {
      linksBefore.erase ( it );
    }
  }

}

template < typename EPtr >
void GraphCellSeccionType< EPtr >::RemoveCells ( std::set<long> &ids )
{
  for_each ( ids.begin(),
             ids.end(),
             boost::bind ( &GraphCellSeccionType< EPtr >::RemoveCell, this, _1 ) );
}


/**
 * @param from_Cell
 * @param to_Cell
 * @return true if the edge is added oo, false if the edge is not added.
 */
template <typename EPtr>
bool GraphCellSeccionType< EPtr >::AddRelationCell ( EPtr from_Cell, EPtr to_Cell )
{
  bool result = Superclass::AddRelationCell ( from_Cell, to_Cell ) ;
  if ( !result )
  {
      linksBefore.insert ( std::make_pair ( from_Cell->GetIdDb() , to_Cell ) );
    return true;
  }
  return result;

}


template <typename EPtr>
bool GraphCellSeccionType< EPtr >::AddRelationIdDb ( long s_id, long v_id )
{
  bool result;

  EPtr from_Cell;
  EPtr to_Cell;

  to_Cell = Superclass::GetCellIdDb ( v_id );

  try
  {
    from_Cell = Superclass::GetCellIdDb ( s_id );
    result = this->AddRelationCell ( from_Cell, to_Cell );
  }
  catch ( ... )
  {
    result = linksBefore.insert ( std::make_pair ( s_id, to_Cell ) );
  }

  return result;
}


template <typename EPtr>
bool GraphCellSeccionType< EPtr >::RemoveRelationCell ( EPtr from_Cell, EPtr to_Cell )
{
  return RemoveRelationIdDb ( from_Cell->GetIdDb(), to_Cell->GetIdDb() );
}


template <typename EPtr>
bool GraphCellSeccionType< EPtr >::RemoveRelationIdDb ( long s, long v )
{
  bool result = false;
  if ( Superclass::RemoveRelationIdDb ( s, v ) )
  {
    result = true;
  }
  else
  {
    for ( typename LinksBeforeType::iterator it = linksBefore.begin();it != linksBefore.end(); ++it )
    {
      if ( it->first == s && it->second->GetIdDb() == v )
      {
        linksBefore.erase ( it );
        result = true;
      }
    }
  }
  return result;
}

template < typename EPtr >
bool GraphCellSeccionType< EPtr >::ExistEdge ( EPtr from_Cell, EPtr to_Cell )
{
  bool result = Superclass::ExistEdge ( from_Cell, to_Cell );
  if ( !result )
  {
    for ( typename LinksBeforeType::iterator it = linksBefore.begin();it != linksBefore.end(); ++it )
    {
      if ( ( it->first == from_Cell->GetIdDb() && it->second->GetIdDb() == to_Cell->GetIdDb() ) )
      {
        result = true;
        return result;
      }
    }
  }
  return result;
}


template <typename EPtr>
int  GraphCellSeccionType< EPtr >::DegreeOut ( EPtr p_Cell )
{
  long result;
  result = Superclass::DegreeOut ( p_Cell );
  return result;
}


template <typename EPtr>
int  GraphCellSeccionType< EPtr >::DegreeIn ( EPtr p_Cell )
{
  long result;
  try
  {
    result = Superclass::DegreeIn ( p_Cell );
  }
  catch ( ... )
  {
    result = 0;
    long iddb = p_Cell->GetIdDb();
    for ( typename LinksBeforeType::iterator it = linksBefore.begin();it != linksBefore.end(); ++it )
    {
      if ( ( it->second->GetIdDb() == iddb ) )
      {
        ++result;
      }
    }
  }
  return result;
}


// template <typename EPtr>
//     GraphCellSeccionType< EPtr >::GetLinksBefore (LinksBeforeType &links )
// {
//   return linksBefore;
// }


template <typename EPtr>
    bool GraphCellSeccionType< EPtr >::IncompleteLink ( EPtr cell )
{
  for ( typename LinksBeforeType::iterator it = linksBefore.begin();it != linksBefore.end(); ++it )
  {
    if ( it->second->GetIdDb() == cell->GetIdDb() )
    {
      return true;
    }
  }
  return false;
}


template <typename EPtr>
    void GraphCellSeccionType< EPtr >::CompleteMorphoField ( GraphCellSeccionType< EPtr > &newEmbryo )
{
  for (typename LinksBeforeType::iterator it = newEmbryo.linksBefore.begin();it != newEmbryo.linksBefore.end();++it)
  {
    LinkBeforeType link = *it;
    
    long idDbFrom = newEmbryo.GetPreviusIdDb(link.second);
    EPtr cell = Superclass::GetCellIdDb(idDbFrom);
    
    std::list< EPtr > cellLife = this->GetCellLife ( cell );

    bool first;
    {
      EPtr previusCell;
      first = true;
      for ( typename std::list< EPtr >::iterator cell = cellLife.begin();cell != cellLife.end();++cell )
      {
        newEmbryo.AddCell ( *cell );
        if ( !first )
        {
          first = false;
          if ( ExistEdge ( *cell, previusCell ) )
          {
            newEmbryo.AddRelationCell ( *cell, previusCell );
          }
        }
        previusCell = *cell;
      }
    }
    newEmbryo.AddRelationCell ( cell, link.second );
  }
}

template <typename EPtr>
void GraphCellSeccionType< EPtr >::GetMorphoFields ( std::set<EPtr> & cells, GraphCellSeccionType< EPtr > &newEmbryo )
{
  Superclass::GetMorphoFields ( cells, newEmbryo );
  if ( newEmbryo.GetTimes().size() != 0 )
  {
    long minTime = * ( newEmbryo.GetTimes().begin() );
    for (typename EmbryoStateType<EPtr>::iterator it = newEmbryo[minTime].begin(); it != newEmbryo[minTime].end() ; ++it)
    {
        EPtr cell = *it;
        if ( IncompleteLink ( cell ) )
        {
          long iddbFrom = GetPreviusIdDb ( cell );
          newEmbryo.linksBefore.insert ( std::make_pair ( iddbFrom, cell ) );
        }
    }
  }
}

template <typename EPtr>
long GraphCellSeccionType< EPtr >::GetPreviusIdDb ( EPtr to_Cell )
{
  std::list< EPtr > p_mother ;
  GetPredecessor ( to_Cell, p_mother );
  if ( p_mother.size() == 1 )
  {
    return ( *p_mother.begin())->GetIdDb();
  }
  else
  {
    for ( typename LinksBeforeType::iterator it = linksBefore.begin();it != linksBefore.end(); ++it )
    {
      if ( ( it->second->GetIdDb() == to_Cell->GetIdDb() ) )
      {
        return it->first;
      }
    }
  }

  return -1;
}



#endif


