/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef __SetPointsWithLocator_h
#define __SetPointsWithLocator_h

#include "EmbryoStateType.h"
#include <iostream>
#include <map>
#include <set>
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include <boost/serialization/map.hpp>
#include <boost/serialization/set.hpp>
#include <boost/config.hpp>

#include <boost/dynamic_bitset.hpp>
#include <boost/pending/indirect_cmp.hpp>
//#include <boost/range/irange.hpp>
#include <boost/shared_ptr.hpp>



#include "SortPoints3.h"


/*! \brief A Set of point with a locator
 * The class implemente a set of point where the points can be access by the locator in a fast way. The points should provide the members GetX(),GetY(),GetZ().
*/
template < typename EPtr >
class SetPointsWithLocator: public EmbryoStateType < EPtr >
{
	public:
		typedef SetPointsWithLocator Self;
		typedef EmbryoStateType < EPtr > Superclass;
	private:

          typedef SortPoints3 < typename Superclass::SetType, EPtr > SortType;

	private:
	public:
          long size();
          virtual void GetNearestNCells ( EPtr aCell, long direction, long nn, std::list< EPtr > &list );

		/*! GetNearestNCells retrieve the nearest points in the set
                 * \param point[3] The point where returned cells should be near
                 * \param nn The number of retrieved cells
                 * \param list The list where the retrieved cells are stored*/
          void GetNearestNCells ( double point[3], long nn, std::list< EPtr > &list );
		/*! For all the cells of the time T the atribute of the cell Neigbourb is calculated
		 */
		void CalculateNeighbor();
                SetPointsWithLocator() :m_Locator ( boost::shared_ptr< SortType >() ) {};
		~SetPointsWithLocator(){}
		/*! Add a cell tho the set
                 */
                virtual void AddCell ( EPtr p_Cell );
		/*! Remove a cell tho the set
                 */
                virtual void RemoveCell ( EPtr p_Cell );
                
		/*! Delete from memory all the cells of the SetPointsWithLocator.
		 */
		virtual void DeallocateCells ( );
                
                
                void InvalidateLocator (  );
		/*!
		 * \return a iterator to the first element in the set
		 */
		typename SetPointsWithLocator::iterator begin() {return m_SetCells.begin();};
		/*!
		 * \return a iterator next to the last element in the set
		 */
		typename SetPointsWithLocator::iterator end() {return m_SetCells.end();};
	private:
                typename Superclass::SetType m_SetCells;
		boost::shared_ptr< SortType > m_Locator;

		friend class boost::serialization::access;
		template<class Archive> void serialize ( Archive & ar, const unsigned int version )
		{
			ar & boost::serialization::base_object<Superclass> ( *this );
			ar & m_SetCells;
			ar & m_Locator;
		}
};

template <typename EPtr>
    /**
     * 
     * @param aCell 
     * @param direction 
     * @param nn 
     * @param list 
     */
    void SetPointsWithLocator < EPtr > ::GetNearestNCells ( EPtr aCell, long direction, long nn, std::list< EPtr > &list )
{
  double point[3];

  point[0] = aCell->GetX() + (aCell->GetDX() * direction);
  point[1] = aCell->GetY() + (aCell->GetDY() * direction);
  point[2] = aCell->GetZ() + (aCell->GetDZ() * direction);
  
  this->GetNearestNCells ( point, nn , list );
}


template <typename EPtr>
void SetPointsWithLocator < EPtr > ::GetNearestNCells ( double point[3], long nn, std::list< EPtr > &list )
{
	list.clear();
	if ( m_Locator.get() == NULL )
	{
          boost::shared_ptr< SortType > m_Locator2(new SortType( &this->m_SetCells ));
          m_Locator = m_Locator2;
	}
	m_Locator->GetNearestNCells ( point, nn , list );
}



template <typename EPtr>
void SetPointsWithLocator<EPtr>::CalculateNeighbor()
{
	for ( typename SetPointsWithLocator<EPtr>::iterator it = begin(); it != end(); ++it )
	{
		double point[3];
		EPtr aCell;
		aCell = *it;

		point[0] = aCell->GetX();
		point[1] = aCell->GetY();
		point[2] = aCell->GetZ();

		std::list< EPtr > list;
		this->GetNearestNCells ( point, 10, list );

		aCell->SetNeighborCells ( list );

	}
}


template <typename EPtr>
    void SetPointsWithLocator<EPtr>::AddCell ( EPtr p_Cell )
{
  m_SetCells.insert ( p_Cell );

  InvalidateLocator();

};

template <typename EPtr>
    void SetPointsWithLocator<EPtr>::InvalidateLocator (  )
{
  m_Locator.reset();
};




template <typename EPtr>
    void SetPointsWithLocator<EPtr>::RemoveCell ( EPtr p_Cell )
{
  m_SetCells.erase ( p_Cell );

  InvalidateLocator();
};


template <typename EPtr>
void SetPointsWithLocator<EPtr>::DeallocateCells ( )
{
	m_SetCells.clear();
        InvalidateLocator();
}

template <typename EPtr>
long  SetPointsWithLocator< EPtr >::size()
{
  return this->m_SetCells.size();
}

#endif
