/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef LineageStatisticType_h
#define LineageStatisticType_h

#include "EmbryoType.h"
#include "EmbryoStateType.h"
#include "BoxType.h"


#include <functional>
#include <numeric>
#include <boost/foreach.hpp>



/**
	@author
 */
template <class EPtr>
class LineageStatisticType
{

  private:
    typedef LineageStatisticType Self;
  public:

    LineageStatisticType() {};

    ~LineageStatisticType() {};

    EmbryoType < EPtr > *GetEmbryo()
    {
      return this->m_embryo;
    };
    void SetEmbryo ( EmbryoType< EPtr > *p_embryo )
    {
      this->m_embryo = p_embryo;
    };

    void CellsTimeLife ( std::ostream& os );

    void PlotDivisionIntensityStat ( std::string fileName );
    void PlotDivisionSpeedStat ( std::string fileName );
    void CellAgeByTimeStep ( std::string fileName );
    void CellsInByTimeStep ( std::string fileName );
    void CellsOutByTimeStep ( std::string fileName );


  public:

  private:

  private:

    EmbryoType < EPtr > *m_embryo;
};

template <class EPtr>
void LineageStatisticType< EPtr >::CellsTimeLife ( std::ostream& os )
{
  std::set< long > times = this->GetEmbryo()->GetTimes();
  typename std::set< long >::iterator timesIt;

  std::multiset< long > cellsLifeTime;

  for ( timesIt = times.begin();timesIt != times.end();++timesIt )
  {
    long time = *timesIt;

    EmbryoStateType< EPtr > *embryoState;
    embryoState = this->GetEmbryo()->GetEmbryoState ( time );



    for ( typename EmbryoStateType< EPtr >::iterator stateIt = embryoState->begin();stateIt != embryoState->end();++stateIt )
    {
      EPtr aCell = *stateIt;

      if ( this->GetEmbryo()->GetTimeStepsLiving ( aCell ) == 0 )
      {
        std::list< EPtr > aCellLife = this->GetEmbryo()->GetCellLife ( aCell );


        bool inside = true;
        for ( typename std::list< EPtr >::iterator it = aCellLife.begin();
              it != aCellLife.end(); ++it )
        {
          inside = inside && !this->GetEmbryo()->CellAtBorder ( *it );
        }
        if ( inside )
        {
          cellsLifeTime.insert ( aCellLife.size() );
        }
      }
    }
  }

  BOOST_FOREACH ( long cellLifeTime, cellsLifeTime )
  {
    os << cellLifeTime << std::endl;
  }
}


template <class EPtr>
    void LineageStatisticType< EPtr >::PlotDivisionIntensityStat ( std::string fileName )
{
  std::ofstream out ( fileName.c_str() );

  long canCells = 100;
  std::set< long >  times = this->GetEmbryo()->GetTimes();
  BOOST_FOREACH ( long time, times )
  {
    std::set< EPtr > dividing = this->GetEmbryo()->GetDividingCells ( time );

    
    BOOST_FOREACH ( EPtr cell, dividing )
    {
      std::list<EPtr> succ ;
      this->GetEmbryo()->GetSuccessor ( cell, succ );

      std::list< EPtr > cellLife = this->GetEmbryo()->GetCellLife ( cell );
      std::list< EPtr > cellLifeChild1 = this->GetEmbryo()->GetCellLife ( * ( succ.begin() ) );
      std::list< EPtr > cellLifeChild2 = this->GetEmbryo()->GetCellLife ( * ( succ.rbegin() ) );

      long c1 = cellLife.size();
      cell->GetIntensity ( 0 );
      
      long c2 = cellLifeChild1.size();
      long c3 = cellLifeChild2.size();
      
      if ( c1 > 10 && c2 > 10 && c3  > 10 )
      {
        std::list<double> infoVecP;
        {
          BOOST_FOREACH ( EPtr cell1, cellLife )
          {
            EmbryoStateType< EPtr > *embryoState;
            embryoState = this->GetEmbryo()->GetEmbryoState ( cell1->GetT() );
            embryoState->CalcIntensity();

            infoVecP.push_back ( cell1->GetIntensity ( 3 ) );
          }

          infoVecP.reverse();
          infoVecP.resize ( canCells, 0 );
          infoVecP.reverse();
        }

        //cellChild1
        std::list<double> infoVecC1;
        {
          BOOST_FOREACH ( EPtr cell1, cellLifeChild1 )
          {
            EmbryoStateType< EPtr > *embryoState;
            embryoState = this->GetEmbryo()->GetEmbryoState ( cell1->GetT() );
            embryoState->CalcIntensity();

            infoVecC1.push_back ( cell1->GetIntensity ( 3 ) );
          }
          infoVecC1.resize ( canCells, 0 );
        }
        //cellChild2
        std::list<double> infoVecC2;
        {
          BOOST_FOREACH ( EPtr cell1, cellLifeChild2 )
          {
            EmbryoStateType< EPtr > *embryoState;
            embryoState = this->GetEmbryo()->GetEmbryoState ( cell1->GetT() );
            embryoState->CalcIntensity();

            infoVecC2.push_back ( cell1->GetIntensity ( 3 ) );
          }
          infoVecC2.resize ( canCells, 0 );
        }

        std::copy ( infoVecP.begin(), infoVecP.end(), std::ostream_iterator<double> ( out, " " ) );
        std::copy ( infoVecC1.begin(), infoVecC1.end(), std::ostream_iterator<double> ( out, " " ) );
        out << std::endl;

        std::copy ( infoVecP.begin(), infoVecP.end(), std::ostream_iterator<double> ( out, " " ) );
        std::copy ( infoVecC2.begin(), infoVecC2.end(), std::ostream_iterator<double> ( out, " " ) );
        out << std::endl;

      }
    }
  }
}


template <class EPtr>
    void LineageStatisticType< EPtr >::PlotDivisionSpeedStat ( std::string fileName )
{
  std::ofstream out ( fileName.c_str() );

  long canCells = 100;
  std::set< long >  times = this->GetEmbryo()->GetTimes();
  BOOST_FOREACH ( long time, times )
  {
    std::set< EPtr > dividing = this->GetEmbryo()->GetDividingCells ( time );

    EmbryoStateType< EPtr > *embryoState;
    embryoState = this->GetEmbryo()->GetEmbryoState ( time );
    embryoState->CalcIntensity();
    
    
    BOOST_FOREACH ( EPtr cell, dividing )
    {
      std::list<EPtr> succ ;
      this->GetEmbryo()->GetSuccessor ( cell, succ );

      std::list< EPtr > cellLife = this->GetEmbryo()->GetCellLife ( cell );
      std::list< EPtr > cellLifeChild1 = this->GetEmbryo()->GetCellLife ( * ( succ.begin() ) );
      std::list< EPtr > cellLifeChild2 = this->GetEmbryo()->GetCellLife ( * ( succ.rbegin() ) );

      long c1 = cellLife.size();
      long c2 = cellLifeChild1.size();
      long c3 = cellLifeChild2.size();
      
      if ( c1 > 10 && c2 > 10 && c3  > 10 )
      {
        std::list<double> infoVecP;
        {
          BOOST_FOREACH ( EPtr cell1, cellLife )
          {
            infoVecP.push_back ( this->GetEmbryo()->GetSpeed ( cell1 ) );
          }

          infoVecP.reverse();
          infoVecP.resize ( canCells, 0 );
          infoVecP.reverse();
        }

        //cellChild1
        std::list<double> infoVecC1;
        {
          BOOST_FOREACH ( EPtr cell1, cellLifeChild1 )
          {
            infoVecC1.push_back ( this->GetEmbryo()->GetSpeed ( cell1 ) );
          }
          infoVecC1.resize ( canCells, 0 );
        }
        //cellChild2
        std::list<double> infoVecC2;
        {
          BOOST_FOREACH ( EPtr cell1, cellLifeChild2 )
          {
            infoVecC2.push_back ( this->GetEmbryo()->GetSpeed ( cell1 ) );
          }
          infoVecC2.resize ( canCells, 0 );
        }

        std::copy ( infoVecP.begin(), infoVecP.end(), std::ostream_iterator<double> ( out, " " ) );
        std::copy ( infoVecC1.begin(), infoVecC1.end(), std::ostream_iterator<double> ( out, " " ) );
        out << std::endl;

        std::copy ( infoVecP.begin(), infoVecP.end(), std::ostream_iterator<double> ( out, " " ) );
        std::copy ( infoVecC2.begin(), infoVecC2.end(), std::ostream_iterator<double> ( out, " " ) );
        out << std::endl;

      }
    }
  }
}




template <class EPtr>
    void LineageStatisticType< EPtr >::CellAgeByTimeStep ( std::string fileName )
{
  std::ofstream out ( fileName.c_str() );

  for (typename EmbryoType < EPtr >::iterator cellIt = this->GetEmbryo()->begin(); cellIt != this->GetEmbryo()->end(); ++cellIt)
  {
    EPtr cell;
    cell = *cellIt;
    
    std::list< EPtr > cellLife = this->GetEmbryo()->GetCellLife(cell);

    out << cell->GetT() << " " << cellLife.size() << std::endl;
  }

}




template <class EPtr>
    void LineageStatisticType< EPtr >::CellsInByTimeStep ( std::string fileName )
{
  std::vector<double> minValue(3);
  std::vector<double> maxValue(3);
  
  minValue[0] = 5;
  minValue[1] = 5;
  minValue[2] = 5;
  
  maxValue[0] = 695;
  maxValue[1] = 695;
  maxValue[2] = 137;
  
  
  long munRegions[3] = {1,1,1};
  double bufferSize[3] = {5, 5, 5};
  BoxType boxSelection;

  boxSelection.SetMin ( minValue );
  boxSelection.SetMax ( maxValue );
  boxSelection.SetNumRegions ( munRegions );
  boxSelection.SetBuffer ( bufferSize );
    
  
  
  std::ofstream out ( fileName.c_str() );
  std::set< long >  times = this->GetEmbryo()->GetTimes();
  for (std::set< long >::iterator timesIt = times.begin();timesIt != times.end();++timesIt )
  {
    long time = *timesIt;
    EmbryoStateType< EPtr > *embryoState;
    embryoState = this->GetEmbryo()->GetEmbryoState ( time );
    for ( typename EmbryoStateType< EPtr >::iterator stateIt = embryoState->begin();stateIt != embryoState->end();++stateIt )
    {
      EPtr aCell = *stateIt;
      if ( this->GetEmbryo()->GetTimeStepsLiving ( aCell ) == 0 )
      {
        
        if (!boxSelection.IsInside(aCell->GetX(),aCell->GetY(),aCell->GetZ(),0))
          if (boxSelection.IsInsideBuffer(aCell->GetX(),aCell->GetY(),aCell->GetZ(),0))
        {
          out << aCell->GetX()<< " " << aCell->GetY() << " " << aCell->GetZ() << " "<< aCell->GetT() << std::endl;
        }
      }
    }
  }
}

template <class EPtr>
    void LineageStatisticType< EPtr >::CellsOutByTimeStep ( std::string fileName )
{
  std::vector<double> minValue(3);
  std::vector<double> maxValue(3);
  
  minValue[0] = 5;
  minValue[1] = 5;
  minValue[2] = 5;
  
  maxValue[0] = 695;
  maxValue[1] = 695;
  maxValue[2] = 137;
  
  
  long munRegions[3] = {1,1,1};
  double bufferSize[3] = {5, 5, 5};
  
  BoxType boxSelection;

  boxSelection.SetMin ( minValue );
  boxSelection.SetMax ( maxValue );
  boxSelection.SetNumRegions ( munRegions );
  boxSelection.SetBuffer ( bufferSize );
    
  
  
  std::ofstream out ( fileName.c_str() );
  std::set< long >  times = this->GetEmbryo()->GetTimes();
  for (std::set< long >::iterator timesIt = times.begin();timesIt != times.end();++timesIt )
  {
    long time = *timesIt;
    EmbryoStateType< EPtr > *embryoState;
    embryoState = this->GetEmbryo()->GetEmbryoState ( time );
    for ( typename EmbryoStateType< EPtr >::iterator stateIt = embryoState->begin();stateIt != embryoState->end();++stateIt )
    {
      EPtr aCell = *stateIt;
      if ( this->GetEmbryo()->GetNumberSuccessor( aCell ) == 0 )
      {
        
        if (!boxSelection.IsInside(aCell->GetX(),aCell->GetY(),aCell->GetZ(),0))
          if (boxSelection.IsInsideBuffer(aCell->GetX(),aCell->GetY(),aCell->GetZ(),0))
        {
          out << aCell->GetX()<< " " << aCell->GetY() << " " << aCell->GetZ() << " " << aCell->GetT() << std::endl;
        }
      }
    }
  }
}
  
  
  
  



#endif
