/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef io_h
#define io_h

#include <fstream>
#include <iostream>
#include <boost/tokenizer.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/format.hpp>
#include <string>
#include "EmbryoType.h"


template <typename EPtr>
    long IOAppendEmbryoExecute( EmbryoType<EPtr> &embryo, long time, ofstream &fout );



long IOSaveHeader ( ofstream &fout )
{
  try
  {
    fout << "#id_center; " <<  "id_center_origin; " <<  "time; " <<  "id_center_mother; " <<  "x; " <<  "y; " <<  "z; " <<  "vx; " <<  "vy; " <<  "vz; " <<  "track_probability; " <<  "center_probability; " <<  std::endl;
    std::cout << "Header Saved" << std::endl;
  }
  catch ( ... )
  {
    std::cout << "Fail to save Header " << std::endl;
    return 1;
  }
  return 0;
}



//, 

template < typename G, class EPtr >
    long IOSaveLineageTree(G &embryo, long timeBegin, long timeEnd, std::ofstream &fout)
{
  try
  {
    std::set< long > times = embryo.GetTimes();
    for ( std::set< long >::iterator  timeIt = times.begin();timeIt != times.end();timeIt++ )
    {
      if ( timeBegin <= ( *timeIt ) && ( *timeIt ) <= timeEnd )
      {
        if (IOAppendEmbryoExecute< EPtr > ( embryo, *timeIt,fout ))
        {
          std::cout << "Fail to save Lineage Tree: " << std::endl;
          return 1;
        }
      }
    }
  }
  catch ( ... )
  {
    std::cout << "Fail to save Lineage Tree: " << std::endl;
    return 1;
  }
  return 0;
}


template <typename EPtr>
    long IOAppendEmbryoExecute( EmbryoType<EPtr> &embryo, long time, ofstream &fout )
{
  EmbryoStateType< EPtr > *embryoState;
  embryoState = embryo.GetEmbryoState ( time );

  std::vector < long > vec_id_center;
  std::vector < long > vec_id_center_origin;
  std::vector < long > vec_time;
  std::vector < long > vec_id_center_mother;
  std::vector < double > vec_x;
  std::vector < double > vec_y;
  std::vector < double > vec_z;
  std::vector < double > vec_vx;
  std::vector < double > vec_vy;
  std::vector < double > vec_vz;
  std::vector < double > vec_track_probability;
  std::vector < double > vec_center_probability;

  
  long i = 0;
  for ( typename EmbryoStateType<EPtr>::iterator it = embryoState->begin();it != embryoState->end();++it )
  {
    EPtr cell = *it;
    
    vec_id_center.push_back ( cell->GetIdDbNew() );
    vec_id_center_origin.push_back ( cell->GetIdDb() );
    vec_time.push_back ( cell->GetT() );

    std::list< EPtr > p_mother;
    embryo.GetPredecessor ( cell, p_mother );
    if ( p_mother.size() == 0 )
    {
      vec_id_center_mother.push_back ( -1 );
      vec_track_probability.push_back ( -1 );
    }
    else
    {
      EPtr cellMother = * ( p_mother.begin() );
      vec_id_center_mother.push_back ( cellMother->GetIdDbNew() );
      EdgePropertyType edgeProb = embryo.GetEdgeProperty(cellMother,cell);
      vec_track_probability.push_back ( edgeProb.GetProbability() );
    }

    vec_x.push_back ( cell->GetX() );
    vec_y.push_back ( cell->GetY() );
    vec_z.push_back ( cell->GetZ() );
    vec_vx.push_back ( cell->GetDX() );
    vec_vy.push_back ( cell->GetDY() );
    vec_vz.push_back ( cell->GetDZ() );
    vec_center_probability.push_back ( cell->GetProb() );
    i++;
  }

  try
  {
    for (long j = 0;j<i;j++)
    {
      fout << vec_id_center[i] << "; ";
      fout << vec_id_center_origin[i]<< "; ";
      fout << vec_time[i]<< "; ";
      fout << vec_id_center_mother[i]<< "; ";
      fout << vec_x[i]<< "; ";
      fout << vec_y[i]<< "; ";
      fout << vec_z[i]<< "; ";
      fout << vec_vx[i]<< "; ";
      fout << vec_vy[i]<< "; ";
      fout << vec_vz[i]<< "; ";
      fout << vec_track_probability[i]<< "; ";
      fout << vec_center_probability[i]<< "; ";
      fout << std::endl;
    }
  }
  catch ( ... )
  {
    std::cout << "Fail to save Lineage Tree " << std::endl;
    return 1;
  }
  std::cout << "Tree Uploaded, Time :"<< time << " Number:"<< i  <<  std::endl;
  return 0;
}


template <typename E, typename EPtr >
    void readSisterInfo ( long timeBegin, long timeEnd, EmbryoType< EPtr > *myEmbryo, std::string pattern )
{
  for ( int i = timeBegin;i <= timeEnd;i++ )
  {
    boost::format fmter ( pattern );
    fmter % i;
    string s  = fmter.str();
    std::cout << s  <<  std::endl;
    try
    {
      ifstream ifs ( s.c_str() );
      if ( ifs )
      {
        while ( !ifs.eof() )
        {
          int cell_NumberP, cell_NumberCh1, cell_NumberCh2;
          ifs >> cell_NumberP >> cell_NumberCh1 >> cell_NumberCh2;
          if ( cell_NumberP > 0 && cell_NumberCh1 > 0 && cell_NumberCh2 > 0 )
          {
            if ( ifs.eof() )
            {

              break;
            }
            EmbryoStateType< EPtr > *embryoState;
            embryoState = myEmbryo->GetEmbryoState ( i + 1 );
            if ( embryoState )
            {
              EPtr cell1 = embryoState->GetCellByNumber ( cell_NumberCh1 );
              std::cout << cell1  <<  std::endl;
              if ( cell1 )
              {
                cell1->SetIsNewBourne ( true );
              }
              EPtr cell2 = embryoState->GetCellByNumber ( cell_NumberCh2 );
              std::cout << cell2  <<  std::endl;
              if ( cell2 )
              {
                cell2->SetIsNewBourne ( true );
              }
            }
          }
        }
      }
    }

    catch ( ... )
    {
      std::cout << "Can not read file:"  << s.c_str() << std::endl;
    }

  }
}


template < typename G, typename E, typename EPtr >
    void IOReadCenters ( long timeBegin, long timeEnd, G *myEmbryo, std::string pattern, double spacingX, double spacingY, double spacingZ, std::set<EPtr> &setCells )
{
  long IDDB = 0;
  
  for ( int i = timeBegin;i <= timeEnd;i++ )
  {
    boost::format fmter ( pattern );
    fmter % i;
    string s  = fmter.str();
    try
    {
      ifstream ifs ( s.c_str() );
      if ( ifs )
      {
        int cell_Number = 1;
        while ( !ifs.eof() )
        {
          int  point[3];
          ifs >> point[0] >> point[1] >> point[2];

          if ( ifs.eof() )
          {
            break;
          }
          EPtr  cell ( new E ( point[0]*spacingX, point[1]*spacingY, point[2]*spacingZ, i, cell_Number++, IDDB++ ) );
          setCells.insert ( cell );
        }
      }
    }
    catch ( ... )
    {
      std::cout << "Can not read file:"  << s.c_str() << std::endl;
    }

  }
}





template < class G, class E, class EPtr >
    int IOLoadLineageTree(G &myEmbryo, std::string fileNamePattern, long timeBegin, long timeEnd )
{
  std::list< long > fromCell;
  std::list< long > toCell;
  
  std::cout << "Loeading Data from File....." << std::endl;
  
  for (long i = 0; i<1000;++i)
  {
    boost::format fmter ( fileNamePattern );
    fmter % i;
    std::string fileName = fmter.str();
    
    std::ifstream fin (fileName.c_str());
    if (fin)
    {
      std::string line;
      while(getline(fin,line)) {
      
        int found = line.find_first_of ( "#" );
        std::string data;
        if (found)
        {
          data = line.substr ( 0, found );
        }
        else
        {
          data = line;
        }
        
        typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
        boost::char_separator<char> sep("; ");
        tokenizer tokens(data, sep);
        long position = 0;
        for (tokenizer::iterator tok_iter = tokens.begin();tok_iter != tokens.end(); ++tok_iter)
        {
          std::string datum = *tok_iter;
          
          long id_center;
          long id_center_origin;
          long time;
          long id_center_mother;
          float x;
          float y;
          float z;
          float vx;
          float vy;
          float vz;
          
          try
          {
            switch (position)
            {
              case 0: id_center = boost::lexical_cast<long>( datum ); break;
              case 1: id_center_origin = boost::lexical_cast<long>( datum ) ;break;
              case 2: time = boost::lexical_cast<long>( datum );break;
              case 3: id_center_mother = boost::lexical_cast<long>( datum );break;
              case 4: x = boost::lexical_cast<float>( datum );break;
              case 5: y = boost::lexical_cast<float>( datum );break;
              case 6: z = boost::lexical_cast<float>( datum );break;
              case 7: vx = boost::lexical_cast<float>( datum );break;
              case 8: vy = boost::lexical_cast<float>( datum );break;
              case 9: vz = boost::lexical_cast<float>( datum );break;
              default:  ;
            }
          }
          catch( boost::bad_lexical_cast& lex )
          {
            std::cout << lex.what();
          }
          
          if (timeBegin <= time && time <= timeEnd)
          {
            EPtr cell ( new E ( x, y, z, time, id_center_origin, id_center ) );
                
            myEmbryo.AddCell ( cell );
          }
          
          
          if (timeBegin < time && time <= timeEnd)
          {
            fromCell.push_back(id_center_mother);
            toCell.push_back(id_center);
          }
          
          data = data.substr ( found + 1, data.size() - 1 );
          ++position;
        }
      }
    }
  }
  
  std::list< long >::iterator itFrom = fromCell.begin();
  std::list< long >::iterator itTo = toCell.begin();
  for ( ;itFrom != fromCell.end();++itFrom, ++itTo )
  {
    myEmbryo.AddRelationIdDb ( *itFrom, *itTo );
  }
  
  std::cout << "Loading Terminated" << std::endl;
}




#endif
