/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */
#ifndef ParallelGraphCellType_h
#define ParallelGraphCellType_h


#include <boost/property_map/parallel/distributed_property_map.hpp>
#include <boost/graph/distributed/connected_components_parallel_search.hpp>
#include <boost/graph/distributed/adjacency_list.hpp>
#include <boost/graph/distributed/adjacency_list.hpp>
#include <boost/graph/distributed/mpi_process_group.hpp>
#include <boost/graph/distributed/graphviz.hpp>
#include <boost/graph/distributed/mpi_process_group.hpp>
#include <boost/graph/detail/adjacency_list.hpp>
#include <boost/config.hpp>
#include <boost/throw_exception.hpp>
#include <boost/mpl/print.hpp>

#include <Recorder.h>

#include <functions.h>


#include <boost/graph/iteration_macros.hpp>
#include <boost/graph/connected_components.hpp>

#include <string>
#include <iostream>


#include "EdgePropertyType.h"
#include "SingularCallBack.h"
#include "BoxType4d.h"

#include <boost/graph/iteration_macros.hpp>

#include <boost/mem_fn.hpp>
#include <boost/bind.hpp>


#include <boost/config.hpp>
#include <boost/serialization/map.hpp>
#include <boost/serialization/is_bitwise_serializable.hpp>
#include <boost/dynamic_bitset.hpp>
#include <boost/pending/indirect_cmp.hpp>
//#include <boost/range/irange.hpp>
#include <boost/pool/pool_alloc.hpp>
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include "boost/lambda/lambda.hpp"
#include "boost/lambda/bind.hpp"
#include "boost/lambda/construct.hpp"
#include <boost/foreach.hpp>
#include <boost/graph/filtered_graph.hpp>
#include <boost/optional.hpp>


#include <strstream>
#include <sstream>
#include <fstream>


/**
	@author Camilo Melani
	\brief Implement a set of Cells with relations between cells of differents times steps.
	It could be seen as branching process, the cells have relations with cells in next times steps, they can have 1 relation or two.
 */



// namespace boost
// {
//   namespace graph
//   {
//     template< >
//     struct internal_vertex_name< CelluleTypePtr >
//     {
//       typedef pointer_extractor type;
//     };
//
//   }
// }



namespace boost
{
  namespace graph
  {
    template<>
    struct internal_vertex_name<CelluleType>
    {
      typedef multi_index::member<CelluleType, long, &CelluleType::IdDb> type;
    };

  }
}



template < typename EPtr, typename E>
class ParallelGraphCellType /*: public EmbryoType < EPtr >*/
{
  public:
    typedef ParallelGraphCellType < EPtr, E >  Self;
//     typedef EmbryoType < EPtr > Superclass;
 
  public:
    typedef boost::adjacency_list< boost::vecS, boost::distributedS< boost::graph::distributed::mpi_process_group, boost::vecS >, boost::undirectedS, E, EdgePropertyType > DistGraphType;


    typedef typename boost::graph_traits< DistGraphType >::edge_descriptor Edge;
    typedef typename boost::graph_traits< DistGraphType >::edge_iterator edge_iterator;
    typedef typename boost::graph_traits< DistGraphType >::vertex_descriptor Vertex;
    typedef typename boost::graph_traits< DistGraphType >::vertex_iterator vertex_iterator;
    typedef typename boost::graph_traits< DistGraphType >::adjacency_iterator adjacency_iterator;
    typedef typename boost::graph_traits< DistGraphType >::out_edge_iterator out_edge_iterator;

    /*    typedef typename boost::property_map< DistGraphType, boost::vertex_index_t>::type ComponentMap_t;
        typedef typename boost::iterator_property_map<std::vector< long >::iterator, ComponentMap_t> ComponentMap;*/

    typedef typename boost::property_map<DistGraphType, long E::*>::type PropertyNode;




//    typedef CallBack< Self, double, EPtr, EPtr > EnergyInterTimeNodeType;
    typedef CallBack< Self, double, EPtr > EnergyNodeType;

  private:
    friend class boost::serialization::access;
    template<class Archive> void serialize ( Archive & ar, const unsigned int version )
    {
      ar & graph &time_m & time_b;
    }

  public:
    long RenameCells ( boost::mpi::communicator &world );




    PropertyNode propertyComponent;
    DistGraphType graph;
  private:

    PropertyNode propertyTime;
    PropertyNode propertyIdDb;
    PropertyNode propertyNumberSuccessors;
    PropertyNode propertyNumberPredecessors;
    PropertyNode propertyName;
    PropertyNode propertyNewbourne;

    double time_m;
    double time_b;


//    void AdjacentVertexSameTime ( Vertex s, std::list< Vertex > &list );
    void GetSuccessor ( E* p_Cell, std::list< E* > &p_daughters ) ;
    void GetSuccessor ( Vertex s, std::list< Vertex > &list ) ;
    void GetPredecessor ( E *p_Cell, std::list< E* > &p_mother );
    void GetPredecessor ( Vertex s, std::list< Vertex > &list );

    long GetNumberSuccessor ( Vertex s ) ;
    long GetNumberPredecessor ( Vertex s );


    void UpdateProperties ();

    void GetCellLife ( Vertex s, std::list< Vertex > &cellLife );

  public:
    void Synchronize ( );
    long NumVertices();

    double GetAge ( Vertex s );
    double GetSpeed ( Vertex s );
    double GetSpeed ( Vertex s, long t );
    void SetCellName();

    void SetRealTime ( long timeStepA, double minutesA, long timeStepB, double minutesB );
    double GetTimeStep ( );
    double GetRealTime ( long timeStep );
    double GetRealTimeLiving ( EPtr a_Cell );


    long CalculateConnectedComponent ( PropertyNode &p_component );
    std::set< long > GetComponents ( BoxType4d &box, long regionNumber );
    std::set< EPtr > GetCells ( BoxType4d &box, long regionNumber );
    long  Save ( std::ostream &fout, BoxType4d &box );
    long  SaveAnnotations ( boost::mpi::communicator &world, std::ostream &fout );



//    virtual const std::set< long > &GetTimes();
  public:
    void AddCell ( E *p_Cell );
//    void RemoveCell ( EPtr p_Cell );
//    void RemoveCells ( std::set< EPtr > &cellsToDelete );
    bool AddRelationVertex ( Vertex s, Vertex v );
    bool AddRelationCell ( E *s, E *v );
    bool AddRelationIdDb ( long s, long v );
    bool IsLocal( long id_s);


    bool RemoveRelationCell ( E *s, E *v );
    bool ExistCell ( long Name );
    bool ExistEdge ( E *s, E *n );


//    std::set<EPtr>  GetCellsToEnumerate();
//    void AddCellToEnumerate ( EPtr p_Cell );
    bool GetMorphoFields ( boost::mpi::communicator &world, BoxType4d &box, long regionNumber, GraphCellType< EPtr > &newEmbryo, long dest );

    void MergeLinegeTree ( boost::mpi::communicator &world, GraphCellType< EPtr > &p_lineage, BoxType4d &boxSelection, long regionNumber );

    int DegreeIn ( E *p_Cell );
    int DegreeOut ( E *p_Cell );
    int DegreeIn ( Vertex s );
    int DegreeOut ( Vertex s );

//    EPtr GetCellName ( long name );

//    void AsignEdgeProperty ( );
    EdgePropertyType GetEdgeProperty ( E *from_Cell, E *to_Cell );
    void SetEdgeProperty ( E *from_Cell, E *to_Cell, EdgePropertyType edgeProperty );

//    EmbryoStateType < EPtr >* GetEmbryoState ( long time );

//    void RegisterNodeEnergy ( std::string energyName, double energyValue );
//    void RegisterNodeEnergy ( std::vector <std::string> &energy_name, std::vector <double>  &energy_value );
//    bool DeleteEnergy(std::string name);


  public:
    void write_graph ( const std::string &output_fileNameLineageTree );

//    void GetEnergy ( EPtr  cell, std::ostream &os );

//    double GetEnergy ( const std::string &name, Vertex s );
//    double GetEnergy ( Vertex s );

//    double GetEnergy ( const std::set< Vertex > &vertexSet );
//    double GetEnergy ( const std::set< EPtr > &cells );
//    double GetEnergy ( EPtr );

//    double GetEmbryoEnergy ( long time );
//    double GetEmbryoEnergy ( long timeBegin, long timeEnd );


//    double ElasticEnergy ( EPtr p_Cell );
//    double SymetricDaugthersEnergy ( EPtr s_Cell );
//    double NewBourneEnergy ( EPtr s_Cell );
//    double DistanceEnergy ( EPtr s_Cell );
//    double SucesorEnergy ( EPtr p_Cell );
//    double PredecessorEnergy ( EPtr p_Cell );
//    double AgeEnergy ( EPtr p_Cell );
//    double MaxSpeed ( EPtr p_Cell );
//    double AccelerationEnergy ( EPtr p_Cell );
//    double BorderEnergy ( EPtr p_Cell );
//    double PredictionEnergy ( EPtr p_Cell );

//    void WriteNearestMesh ( std::ostream& ofs, long time );
//    void SaveSQLFileAllEmbryo ( std::ostream& ofs, long id_process_tracking );
//    void SaveSQLFileEmbryoState ( std::ostream& ofs, long id_process_tracking, long time );

  public:
    ParallelGraphCellType();
    ~ParallelGraphCellType();





    template <typename CompMap>
    struct withMapVertex
    {
      withMapVertex() { }
      withMapVertex ( CompMap *comp_map, std::set< long > *comp_set ) : m_component ( comp_map ), m_comp_set ( comp_set ) { }
      template <typename Vertex>
      bool operator() ( const Vertex& v ) const
      {
        return m_comp_set->count ( get ( *m_component, v ) ) > 0;
      }
      CompMap *m_component;
      std::set< long > *m_comp_set;
    };

    template <typename CompMap>
    struct withMapEdge
    {
      withMapEdge() { }
      withMapEdge ( DistGraphType *graph, CompMap *comp_map, std::set< long > *comp_set ) : m_graph ( graph ), m_component ( comp_map ), m_comp_set ( comp_set ) { }

      template <typename Edge>
      bool operator() ( const Edge &e ) const
      {
        return ( m_comp_set->count ( get ( *m_component, source ( e, *m_graph ) ) ) > 0 && m_comp_set->count ( get ( *m_component, target ( e, *m_graph ) ) ) > 0 );
      }
      CompMap *m_component;
      std::set< long > *m_comp_set;
      DistGraphType *m_graph;
    };



};

template < typename EPtr, typename E >
ParallelGraphCellType< EPtr, E >::ParallelGraphCellType() /*: Superclass()*/
{
  propertyTime = get ( &E::T, graph );
  propertyNumberSuccessors = get ( &E::NumberSuccessors, graph );
  propertyNumberPredecessors = get ( &E::NumberPredecessors, graph );
  propertyNewbourne = get ( &E::Newbourne, graph );
  propertyIdDb = get ( &E::IdDb, graph );
  propertyName = get ( &E::Name, graph );
  propertyComponent = get ( &E::ComponentNumber, graph );
}


template < typename EPtr, typename E >
ParallelGraphCellType< EPtr, E >::~ParallelGraphCellType()
{}

template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::AddCell ( E *p_Cell )
{
  Vertex s;
  add_vertex ( *p_Cell, this->graph );
//  p_Cell->SetimageProvider ( this->GetImageProvider() );
}



// template < typename EPtr, typename E >
//     void ParallelGraphCellType< EPtr, E >::RemoveCell ( EPtr p_Cell )
// {
//     Vertex s;
//     s = *find_vertex(p_Cell->GetIdDb(), graph);
//     clear_vertex ( s,  this->graph );
//     remove_vertex ( s, this->graph );
// }

// template < typename EPtr, typename E >
//     void ParallelGraphCellType< EPtr, E >::RemoveCells ( std::set<EPtr> &cellsToDelete )
// {
//   for_each(cellsToDelete.begin(),
//            cellsToDelete.end(),
//            boost::bind( &ParallelGraphCellType< EPtr, E >::RemoveCell,this,_1));
// }

/*template < typename EPtr, typename E >
        void ParallelGraphCellType< EPtr, E >::RemoveCell ( EPtr p_Cell )
        {
        std::set<EPtr> cellsToDelete;
        cellsToDelete.insert ( p_Cell );
        RemoveCells ( cellsToDelete );
}



        template < typename EPtr, typename E >
        void ParallelGraphCellType< EPtr, E >::RemoveCells ( std::set<EPtr> &cellsToDelete )
        {
        std::set<long> nodesToReindex;
        for ( typename std::set<EPtr>::iterator it = cellsToDelete.begin();it != cellsToDelete.end();++it )
        {
        EPtr p_Cell = *it;
        typename MapCellNodeType::iterator toIt = this->mapCellNode.find ( p_Cell );
        if ( toIt != this->mapCellNode.end() )
        {
        this->multiSet.RemoveCell ( p_Cell );
        mapCellNode.erase ( p_Cell );
        cellsToEnumerate.erase ( p_Cell );

        Vertex s;
        s = toIt->second;;
        nodesToReindex.insert ( toIt->second );

        clear_vertex ( s,  this->graph );
        remove_vertex ( s, this->graph );
}
}
//   reindex the vertex with vertex greater than s
        for ( typename MapCellNodeType::iterator it = mapCellNode.begin(); it != mapCellNode.end();++it )
        {
        Vertex node = it->second;
        long index = std::distance ( nodesToReindex.begin(), nodesToReindex.lower_bound ( node ) ) ;
        it->second -= index;
}
}
*/



/**
         * @param from_Cell
         * @param to_Cell
         * @return true if the edge is added oo, false if the edge is not added.
 */
template < typename EPtr, typename E >
bool ParallelGraphCellType< EPtr, E >::AddRelationCell ( E *from_Cell, E *to_Cell )
{
  bool result = true;
  return AddRelationIdDb ( from_Cell->GetIdDb(), to_Cell->GetIdDb() );
}



template < typename EPtr, typename E >
bool ParallelGraphCellType< EPtr, E >::AddRelationVertex ( Vertex s, Vertex v )
{
  bool result = true;

  std::pair<Edge, bool> exist1 =  edge ( s, v, graph );
  std::pair<Edge, bool> exist2 =  edge ( v, s, graph );

  if ( !exist1.second && !exist2.second )
  {
//      if ( DegreeOut ( *s ) < 2 && DegreeIn ( *v ) < 1 )
    {
      long timeS ;
      long timeV;
      timeS = get ( propertyTime, s );
      timeV = get ( propertyTime, v );

      if ( abs ( timeS - timeV ) == 1 )
      {
        add_edge ( s, v, this->graph );
        result = true;
      }
      else
      {
        result = false;
        std::cout << "Error Adding link time" << timeS << " to " << timeV << " ";
      }
    }
  }
  return result;
}


template < typename EPtr, typename E >
    bool ParallelGraphCellType< EPtr, E >::IsLocal( long id_s)
{
  bool result = false;
  
  boost::optional<Vertex > s = find_vertex ( id_s, graph );
  if (s)
  {
    return s->owner ==  process_id ( graph.process_group() );
  }
  else
  {
    return false;
  }
}



/**
             * @param from_Cell
             * @param to_Cell
             * @return true if the edge is added oo, false if the edge is not added.
 */
template < typename EPtr, typename E >
bool ParallelGraphCellType< EPtr, E >::AddRelationIdDb ( long id_s, long id_v )
{
  bool result = false;

  boost::optional<Vertex > s = find_vertex ( id_s, graph );
  boost::optional<Vertex > v = find_vertex ( id_v, graph );

  if ( s && v )
  {
    result = AddRelationVertex ( *s, *v );
  }
  else
  {
    std::cout << "Error AddRelationVertex cell not found " << std::endl ;
  }

  return result;
}


template < typename EPtr, typename E >
bool ParallelGraphCellType< EPtr, E >::RemoveRelationCell ( E *from_Cell, E *to_Cell )
{
  Vertex s, v;
  bool result;
  s = *find_vertex ( from_Cell->GetIdDb(), graph );
  v = *find_vertex ( to_Cell->GetIdDb(), graph );

  remove_edge ( s, v, this->graph );

  result = true;

  return result;
}

template < typename EPtr, typename E >
long ParallelGraphCellType< EPtr, E >::GetNumberSuccessor ( Vertex s )
{
  std::list<Vertex> list;
  GetSuccessor ( s, list );
  return list.size();
}

template < typename EPtr, typename E >
long ParallelGraphCellType< EPtr, E >::GetNumberPredecessor ( Vertex s )
{
  std::list<Vertex> list;
  GetPredecessor ( s, list );
  return list.size();
}


template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::GetPredecessor ( Vertex s, std::list<Vertex> &list )
{
  long timeS = get ( propertyTime, s );

  adjacency_iterator vi, vi_end;
  for ( tie ( vi, vi_end ) = adjacent_vertices ( s, this->graph ); vi != vi_end; ++vi )
  {
    Vertex v = *vi;
    long timeV = get ( propertyTime, v );

    if ( timeV < timeS )
    {
      list.push_back ( *vi );
    }
  }
}


template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::GetSuccessor ( Vertex s, std::list<Vertex> &list )
{
  adjacency_iterator vi, vi_end;


  for ( boost::tie ( vi, vi_end ) = adjacent_vertices ( s, this->graph ); vi != vi_end; ++vi )
  {
    long timeS = get ( propertyTime, s );
    long timeV = get ( propertyTime, *vi );

    if ( timeV > timeS )
    {
      Vertex v = *vi;
      list.push_back ( v );
    }
  }
}

template < typename EPtr, typename E >
bool ParallelGraphCellType< EPtr, E >::ExistEdge ( E *from_Cell, E *to_Cell )
{
  bool result = false;

  Vertex s = *find_vertex ( from_Cell->GetIdDb(), graph );
  Vertex v = *find_vertex ( to_Cell->GetIdDb(), graph );

  adjacency_iterator vi, vi_end;

  for ( boost::tie ( vi, vi_end ) = adjacent_vertices ( s, this->graph ); vi != vi_end; ++vi )
  {
    if ( v == *vi )
    {
      result = true;
      return result;
    }
  }
  return result;
}


template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::GetPredecessor ( E *p_Cell, std::list< E* > &p_mother )
{
  try
  {
    std::list< Vertex > predList;
    Vertex s = *find_vertex ( p_Cell->GetIdDb(), graph );
    this->GetPredecessor ( s, predList );

    for ( typename std::list< Vertex >::iterator itPred = predList.begin(); itPred != predList.end();++itPred )
    {
      p_mother.push_back ( & ( this->graph[*itPred] ) );
    }
  }
  catch ( ... )
  {
    std::cout << "Error: GetPredecessor " << std::endl;
  }

}

template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::GetSuccessor ( E* p_Cell, std::list< E* > &p_daughters )
{
  try
  {
    std::list< Vertex > sucesors;

    Vertex s = *find_vertex ( p_Cell->GetIdDb(), graph );

    std::list< Vertex > listSucessorS;
    this->GetSuccessor ( s, listSucessorS );

    for ( typename std::list< Vertex >::iterator itSucS = listSucessorS.begin(); itSucS != listSucessorS.end();++itSucS )
    {
      E *a_Cell = & ( this->graph[*itSucS] );
      p_daughters.push_back ( a_Cell );
    }
  }
  catch ( ... )
  {
    std::cout << "Error: GetSuccessor" << std::endl;
  }
}



template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::SetEdgeProperty ( E *from_Cell, E *to_Cell, EdgePropertyType edgeProperty )
{
  Vertex s = *find_vertex ( from_Cell->GetIdDb(), graph );
  Vertex v = *find_vertex ( to_Cell->GetIdDb(), graph );

  std::pair<Edge, bool> resu;
  resu = edge ( s, v, this->graph );

  this->graph[resu.first] = edgeProperty;
}



// template < typename EPtr, typename E >
// void  ParallelGraphCellType< EPtr, E >::AsignEdgeProperty ( )
// {
//   std::set<EPtr> allCells = this->GetAllCells();
//   double maxEnergy = -100000;
//   for ( typename std::set<EPtr>::iterator it = allCells.begin();it != allCells.end();++it )
//   {
//     double actualEnergy = this->GetEnergy ( *it );
//     maxEnergy = ( actualEnergy > maxEnergy ) ? actualEnergy : maxEnergy;
//   }
//
//   for ( typename std::set<EPtr>::iterator it = allCells.begin();it != allCells.end();++it )
//   {
//     EPtr cell = *it;
//     std::list< EPtr > predList;
//     this->GetPredecessor ( cell, predList );
//     if ( predList.size() != 0 )
//     {
//       EPtr pred = * ( predList.begin() );
//
//       EdgePropertyType property;
//       property.SetProbability ( ( double ) this->GetEnergy ( pred ) / maxEnergy );
//       this->SetEdgeProperty ( pred, cell, property );
//     }
//   }
// }

template < typename EPtr, typename E >
EdgePropertyType ParallelGraphCellType< EPtr, E >::GetEdgeProperty ( E *from_Cell, E *to_Cell )
{
  Vertex s = *find_vertex ( from_Cell->GetIdDb(), graph );
  Vertex v = *find_vertex ( to_Cell->GetIdDb(), graph );

  std::pair<Edge, bool> resu;
  resu = edge ( s, v, this->graph );

  return this->graph[resu.first] ;
}






template < typename EPtr, typename E >
int  ParallelGraphCellType< EPtr, E >::DegreeOut ( E *p_Cell )
{
  std::list< E*> p_daughters;
  GetSuccessor ( p_Cell, p_daughters ) ;
  return p_daughters.size();
}

template < typename EPtr, typename E >
int  ParallelGraphCellType< EPtr, E >::DegreeOut ( Vertex s )
{
  std::list< Vertex > p_daughters;
  GetSuccessor ( s, p_daughters ) ;
  return p_daughters.size();
}


template < typename EPtr, typename E >
int  ParallelGraphCellType< EPtr, E >::DegreeIn ( E *p_Cell )
{
  std::list< E* > p_mother;
  GetPredecessor ( p_Cell, p_mother ) ;
  return p_mother.size();
}


template < typename EPtr, typename E >
int  ParallelGraphCellType< EPtr, E >::DegreeIn ( Vertex s )
{
  std::list< Vertex > p_mother;
  GetPredecessor ( s, p_mother ) ;
  return p_mother.size();
}


template < typename EPtr, typename E >
long ParallelGraphCellType< EPtr, E >::CalculateConnectedComponent ( PropertyNode &p_component )
{

  long num_components = 0;
//  std::vector< long > local_components_vec ( num_vertices ( graph ) );
//  component = ComponentMap ( local_components_vec.begin(), get ( boost::vertex_index, graph ) );
  double start = MPI_Wtime();
  num_components = boost::connected_components ( graph, p_component );
  double end = MPI_Wtime();
  return num_components;
}

template < typename EPtr, typename E >
std::set< long > ParallelGraphCellType< EPtr, E >::GetComponents ( BoxType4d &box, long regionNumber )
{
  std::set< long > result_componets;

  long in = 0;
  long out = 0;
  BGL_FORALL_VERTICES_T ( s, this->graph, DistGraphType )
  {
    E &cell = this->graph[s];
    if ( box.IsInside ( cell.GetX(), cell.GetY(), cell.GetZ(), cell.GetT(), regionNumber ) )
    {
      long c = cell.ComponentNumber;
      result_componets.insert ( c );
      in++;
    }
    else
    {
      out++;
    }
  }

  std::cout << regionNumber << "In " << in << " Out " << out << std::endl;

  return result_componets;
}





template < typename EPtr, typename E >
bool ParallelGraphCellType< EPtr, E >::GetMorphoFields ( boost::mpi::communicator &world, BoxType4d &box, long regionNumber, GraphCellType< EPtr > &newEmbryo, long dest )
{
  long rank = process_id ( graph.process_group() );

  BGL_FORALL_VERTICES_T ( u, graph, DistGraphType )
  {
    request ( propertyTime, u );
    request ( propertyIdDb , u );
    request ( propertyComponent, u );
  }
  synchronize ( propertyTime );
  synchronize ( propertyIdDb );
  synchronize ( propertyComponent );

  //Get The componets inside the box;
  std::set< long > componentSetTEMP = GetComponents ( box, regionNumber );
  std::vector < std::set< long > > vectorComponentSet;

  all_gather ( world, componentSetTEMP, vectorComponentSet );
  std::set< long > componentSet;
  for ( int i = 0;i < world.size();++i )
  {
    componentSet.insert ( vectorComponentSet[i].begin(), vectorComponentSet[i].end() );
  }


  //Filter the graph to show only vertex with component inside the box
  //Add all the filtered vertex and edges.

  this->Synchronize();

  std::map<Vertex, E> cellsMap;
  std::set< std::pair< Vertex, Vertex> > links;

  std::map< Vertex, long >  vectorCellsFlat;

  BGL_FORALL_VERTICES_T ( vertex, this->graph, DistGraphType )
  {
    long c = this->graph[vertex].ComponentNumber;

    if ( componentSet.count ( c ) > 0 )
    {
      cellsMap[vertex] = this->graph[vertex];

      std::pair<adjacency_iterator, adjacency_iterator> bounds = adjacent_vertices ( vertex, graph );
      adjacency_iterator it;
      for ( it = bounds.first; it != bounds.second; ++it )
      {
        Vertex v = *it;
        long timeS = get ( propertyTime, vertex );
        long timeV = get ( propertyTime, v );

        if ( timeS > timeV )
          links.insert ( std::make_pair ( v, vertex ) ) ;
        else
          links.insert ( std::make_pair ( vertex, v ) ) ;
      }
    }
  }

  this->Synchronize();

  //Send links and vertices to dest
  std::vector < std::map< Vertex, E > > vectorCellsMap;
  std::vector < std::set< std::pair< Vertex, Vertex> > >vectorLinks;

  gather ( world, cellsMap, vectorCellsMap, dest );
  this->Synchronize();
  this->Synchronize();
  this->Synchronize();

  gather ( world, links, vectorLinks, dest );

  this->Synchronize();
  this->Synchronize();
  this->Synchronize();

  bool salir = false;
  std::set<Vertex> vNotFound;

  if ( world.rank() == dest )
  {
    for ( int i = 0;i < world.size();++i )
    {
      //Add the cells
      for ( typename std::map< Vertex, E >::iterator it = vectorCellsMap[i].begin();it != vectorCellsMap[i].end(); ++it )
      {
        vectorCellsFlat[it->first] = it->second.GetIdDb();

        E *cell = new E;
        EPtr cell1 ( cell );
        *cell1 = it->second;
        newEmbryo.AddCell ( cell1 );
      }
    }

    for ( int i = 0;i < world.size();++i )
    {
      for ( typename std::set< std::pair< Vertex, Vertex> >::iterator it = vectorLinks[i].begin();it != vectorLinks[i].end(); ++it )
      {
        if ( vectorCellsFlat.find ( it->first ) != vectorCellsFlat.end() ||  vectorCellsFlat.find ( it->second ) != vectorCellsFlat.end() )
        {
          newEmbryo.AddRelationIdDb ( vectorCellsFlat[it->first], vectorCellsFlat[it->second] );
        }
        else
        {
          vNotFound.insert ( it->second );
          std::cout << "Error2: " << dest << " " << local ( it->second ) << " " << owner ( it->second ) << std::endl;
          salir = true;
        }
      }
    }
  }
  this->Synchronize();
  this->Synchronize();
  this->Synchronize();


  if ( salir )
  {
    return false;
  }
  return true;
}



template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::MergeLinegeTree ( boost::mpi::communicator &world, GraphCellType< EPtr > &p_lineage, BoxType4d &boxSelection, long regionNumber )
{
  long rank = process_id ( graph.process_group() );
  BGL_FORALL_VERTICES_T ( u, graph, DistGraphType )
  {
    request ( propertyTime, u );
  }
  synchronize ( propertyTime );



//Send all the lineages tree to all the processors
//  std::vector < GraphCellType< EPtr >  > vectorRecivedLineage;
//  all_gather ( world, p_lineage, vectorRecivedLineage );

//Lineage tree to vectors
  std::map<long, EPtr> cellsMapToSend;
  std::set< std::pair< long, long> > linksToSend;

  std::set< long > times = p_lineage.GetTimes();
  typename std::set< long >::iterator timesIt;
  for ( timesIt = times.begin();timesIt != times.end();++timesIt )
  {
    long i = *timesIt;

//    std::cout << rank << " " << GetActualTime() << " Time:" << i << " NewbourneCells " << p_lineage.GetNumberNewbourneCells ( i )
//    << " NumberDivCells " << p_lineage.GetNumberMotherCells ( i )
//    << std::endl;

    typename EmbryoStateType< EPtr >::iterator it1;
    for ( it1 = p_lineage[i].begin(); it1 != p_lineage[i].end(); ++it1 )
    {
      EPtr cell = *it1;
      if ( boxSelection.IsInside ( cell->GetX(), cell->GetY(), cell->GetZ(), cell->GetT(), regionNumber ) )
      {
        cellsMapToSend[cell->GetIdDb() ] = cell;
        std::list< EPtr > p_daughters;
        p_lineage.GetSuccessor ( cell, p_daughters );
        BOOST_FOREACH ( EPtr cell2, p_daughters )
        {
          linksToSend.insert ( std::make_pair ( cell->GetIdDb(), cell2->GetIdDb() ) );
          if ( cell->GetT() != cell2->GetT() - 1 )
          {
            std::cout << "ERROR:" << cell->GetT() << " " << cell2->GetT() << std::endl;
          }
        }
      }
    }
  }

  //Send links and vertices to dest

  std::vector < std::map< long, EPtr > > vectorCellsMap;
  std::vector < std::set< std::pair< long, long> > >vectorLinks;

  this->Synchronize();
  all_gather ( world, cellsMapToSend, vectorCellsMap );
  all_gather ( world, linksToSend, vectorLinks );

  long cant = 0;

  std::map< long, std::set< long > > links;
  typedef typename std::pair< long, EPtr > pairLongCellType;
  typedef typename std::pair< long, long > pairLongLongType;

  for ( int i = 0; i < world.size();++i )
  {
    BOOST_FOREACH ( pairLongCellType longCell, vectorCellsMap[i] )
    {
      links[longCell.first] = std::set<long>();

      boost::optional<Vertex > o = find_vertex ( longCell.first, graph );
      if ( o )
      {
        Vertex s = *o;
        //IF the cell is local
        if ( get ( get ( boost::vertex_owner, graph ), s ) == process_id ( graph.process_group() ) )
        {

          EPtr cell1 = longCell.second;
          E *cell2 = & ( graph[s] );

          //Is teh same cell
          if ( cell1->GetIdDb() - cell2->GetIdDb() + cell1->GetX() - cell2->GetX() + cell1->GetY() - cell2->GetY() + cell1->GetZ() - cell2->GetZ() + cell1->GetT() - cell2->GetT() > 5 )
          {
            std::cout << "Error: " << cell1->GetIdDb() << " " << cell2->GetIdDb() << " " << cell1->GetX() << " " <<  cell2->GetX() << " " << cell1->GetY() << " " << cell2->GetY() << " " << cell1->GetZ() << " " << cell2->GetZ() << " " << cell1->GetT() << " " << cell2->GetT() << std::endl;
          }
          else
          {
            //Update information in the distributed cells
            cell2->SetDeformation ( cell1->GetDeformation() );
          }
        }
      }
    }
  }

  for ( int i = 0; i < world.size();++i )
  {
    BOOST_FOREACH ( pairLongLongType link, vectorLinks[i] )
    {
      links[link.first].insert ( link.second );
    }
  }


  //Remove all of the recived cells
  if ( rank == 0 )
  {
    std::cout << rank << " " << GetActualTime()  << " Remove Links In region" << std::endl;
  }

  typedef typename std::pair< long, std::set< long > > pairIdLinksType;
  BOOST_FOREACH ( pairIdLinksType idCell, links )
  {
    boost::optional<Vertex > o = find_vertex ( idCell.first, graph );
    if ( o )
    {
      Vertex s = *o;
      //IF the cell is local
      if ( get ( get ( boost::vertex_owner, graph ), s ) == process_id ( graph.process_group() ) )
      {
        E *cell = & ( graph[s] );
        long timeS = get ( propertyTime, s );
        //Remove the edges from the distributed embryo
        bool end = false;
        while ( ! end )
        {
          end = true;
          std::pair<adjacency_iterator, adjacency_iterator> bounds = adjacent_vertices ( s, graph );
          adjacency_iterator it;
          for ( it = bounds.first; it != bounds.second; ++it )
          {
            Vertex v = *it;
            long timeV = get ( propertyTime, v );
            if ( timeS == timeV - 1 )
            {
              remove_edge ( s, v, graph );
              end = false;
              break;
            }
          }
        }
      }
    }
  }
  Synchronize();
  if ( rank == 0 )
  {
    std::cout << rank << " " << GetActualTime()  << " After Remove Links In region" << std::endl;
  }

  //Add the new links
  for ( int i = 0; i < world.size();i++ )
  {
    UpdateProperties();

    BGL_FORALL_VERTICES_T ( u, graph, DistGraphType )
    {
      request ( propertyNumberSuccessors, u );
      request ( propertyNumberPredecessors, u );
    }
    synchronize ( propertyNumberSuccessors );
    synchronize ( propertyNumberPredecessors );


    if ( i != world.rank() )
    {


    }
    else
    {
      typedef typename std::pair< long, std::set< long > > pairIdLinksType;
      BOOST_FOREACH ( pairIdLinksType idCell, links )
      {
        boost::optional<Vertex > o = find_vertex ( idCell.first, graph );
        if ( o )
        {
          Vertex s = *o;
          //IF the cell is local
          if ( get ( get ( boost::vertex_owner, graph ), s ) == process_id ( graph.process_group() ) )
          {
            E *cell1 = & ( graph[s] );
            {
              //Add the link

              BOOST_FOREACH ( long succ, idCell.second )
              {
                boost::optional<Vertex > v = find_vertex ( succ, graph );
                if ( v )
                {
                  long timeV = get ( propertyTime, *v );

                  if ( get ( propertyNumberSuccessors, s ) < 2  && get ( propertyNumberPredecessors, *v ) == 0 )
                  {
                    if ( this->AddRelationIdDb ( idCell.first, succ ) )
                    {
                      put ( propertyNumberSuccessors, s, get ( propertyNumberSuccessors, s ) + 1 );
                      put ( propertyNumberPredecessors, *v, get ( propertyNumberPredecessors, *v ) + 1 );
                    }
                    else
                    {
                      std::cout << "AddRelationIdDb Processor: " << process_id ( graph.process_group() ) << std::endl;
                    }
                  }
                }
              }
            }
          }
        }
        else
        {
          std::cout << rank << " " << GetActualTime()  << " Vertice no encontrado: " << idCell.first << std::endl;
        }
      }
    }
    synchronize ( graph );
  }
}



template < typename EPtr, typename E >
std::set< EPtr > ParallelGraphCellType< EPtr, E >::GetCells ( BoxType4d &box, long regionNumber )
{
  std::set< EPtr > cells;
  BGL_FORALL_VERTICES_T ( vertex, this->graph, DistGraphType )
  {
    E *cell = & ( this->graph[vertex] );
    if ( box.IsInside ( cell->GetX(), cell->GetY(), cell->GetZ(), cell->GetT(), regionNumber ) )
    {
      cells.insert ( cell );
    }
  }
  return cells;
}

template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::Synchronize()
{
  synchronize ( this->graph.process_group() );
}

template < typename EPtr, typename E >
long  ParallelGraphCellType< EPtr, E >::NumVertices()
{
  return num_vertices ( this->graph );
}




template < typename EPtr, typename E >
long  ParallelGraphCellType< EPtr, E >::Save ( std::ostream &fout, BoxType4d &box )
{
  long rank = process_id ( graph.process_group() );

  BGL_FORALL_VERTICES_T ( u, graph, DistGraphType )
  {
    request ( propertyIdDb, u );
    request ( propertyTime, u );
  }
  synchronize ( propertyIdDb );
  synchronize ( propertyTime );

  Synchronize();

  std::cout << rank << " " << GetActualTime() << " Number Vertex: " << this->NumVertices() << std::endl;

  std::ostringstream local_graph_out;
  BGL_FORALL_VERTICES_T ( s, this->graph, DistGraphType )
  {
    E *cell = & ( this->graph[s] );
    {
      long id_Mother = -1;
      std::list< Vertex > p_mother;

      GetPredecessor ( s, p_mother );

      if ( p_mother.size() == 1 )
      {
        id_Mother = get ( propertyIdDb, *p_mother.begin() );
      }
      else if ( p_mother.size() > 1 )
      {
        id_Mother = get ( propertyIdDb, *p_mother.begin() );
        std::cout << rank << " " << GetActualTime()  << " Error cell: " << cell->GetIdDb() << " with " << p_mother.size() << " Predecesors" << std::endl;
      }

      local_graph_out << cell->GetIdDb() << ",";
      local_graph_out << id_Mother << ",";
      local_graph_out << cell->GetX() << ",";
      local_graph_out << cell->GetY() << ",";
      local_graph_out << cell->GetZ() << ",";
      local_graph_out << cell->GetT() << ",";
      local_graph_out << cell->GetDX() << ",";
      local_graph_out << cell->GetDY() << ",";
      local_graph_out << cell->GetDZ() << ",";
      local_graph_out << cell->GetVirtual() << ",";
      local_graph_out << cell->GetName() << ",";
      local_graph_out << cell->GetCenterNumber() << ",";
      local_graph_out << cell->GetDeformation() << ",";
      local_graph_out << box.Distance ( cell->GetX(), cell->GetY(), cell->GetZ() ) ;
      local_graph_out << std::endl;
    }
  }

  if ( process_id ( graph.process_group() ) == 0 )
  {
    fout << local_graph_out.str();
    synchronize ( graph.process_group() );
    for ( int i = 1; i < num_processes ( graph.process_group() ); ++i )
    {
      long len;
      receive ( graph.process_group(), i, 0, len );
      char* data = new char [len+1];
      data[len] = 0;
      receive ( graph.process_group(), i, 1, data, len );
      fout << data;
      delete [] data;
    }
  }
  else
  {
    std::string result_str = local_graph_out.str();
    const char* data = result_str.c_str();

    long len = result_str.length();
    send ( graph.process_group(), 0, 0, len );
    send ( graph.process_group(), 0, 1, data, len );
    synchronize ( graph.process_group() );
  }
  synchronize ( graph.process_group() );
  synchronize ( graph.process_group() );
  synchronize ( graph.process_group() );
}



struct CellLifeInfo
{
  CellLifeInfo() : Cant ( 0 ), Divides ( 0 ), Orphan ( 0 ) {};
  long Cant;
  bool Orphan;
  bool Divides;

  friend class boost::serialization::access;
  template<class Archive> void serialize ( Archive & ar, const unsigned int version )
  {
    ar & Cant & Divides & Orphan;
  }
};

template < typename EPtr, typename E >
long  ParallelGraphCellType< EPtr, E >::SaveAnnotations ( boost::mpi::communicator &world, std::ostream &fout )
{

  std::ostringstream local_graph_out;

  std::map<long, CellLifeInfo> mapCellInfo;

  BGL_FORALL_VERTICES_T ( s, this->graph, DistGraphType )
  {
    E *cell = & ( this->graph[s] );

    if ( mapCellInfo.find ( cell->GetName() ) == mapCellInfo.end() )
    {
      mapCellInfo[cell->GetName() ] = CellLifeInfo();
    }

    mapCellInfo[cell->GetName() ].Cant++;
    mapCellInfo[cell->GetName() ].Orphan = mapCellInfo[cell->GetName() ].Orphan || ( get ( propertyNumberPredecessors, s ) == 0 );
    mapCellInfo[cell->GetName() ].Divides = mapCellInfo[cell->GetName() ].Divides || ( get ( propertyNumberSuccessors, s ) == 2 );
  }


  std::vector< std::map<long, CellLifeInfo> > vectorMapCellInfo;
  gather ( world, mapCellInfo, vectorMapCellInfo, 0 );

  if ( world.rank()  == 0 )
  {
    std::map<long, CellLifeInfo > mapCellInfoAccum;
    for ( int i = 0; i < world.size();++i )
    {
      for ( typename std::map<long, CellLifeInfo >::iterator it = vectorMapCellInfo[i].begin();it != vectorMapCellInfo[i].end();++it )
      {
        if ( mapCellInfoAccum.find ( it->first ) == mapCellInfoAccum.end() )
        {
          mapCellInfoAccum[it->first] = CellLifeInfo();
        }
        mapCellInfoAccum[it->first].Cant += it->second.Cant;
        mapCellInfoAccum[it->first].Orphan = ( mapCellInfoAccum[it->first].Orphan || it->second.Orphan ) ;
        mapCellInfoAccum[it->first].Divides = ( mapCellInfoAccum[it->first].Divides || it->second.Divides ) ;
      }
    }

    for ( typename std::map<long, CellLifeInfo >::iterator it = mapCellInfoAccum.begin();it != mapCellInfoAccum.end();++it )
    {
      local_graph_out << it->first << ",";
      local_graph_out << it->second.Cant << ",";
      local_graph_out << it->second.Orphan << ",";
      local_graph_out << it->second.Divides ;
      local_graph_out << std::endl;
    }

    fout << local_graph_out.str();
  }
}



template<typename Graph>
struct node_writer
{

  explicit node_writer ( Graph &graph ) : graph ( graph )
  {
//    CalculateConnectedComponent();
  }

  template<typename Descriptor>
  void operator() ( std::ostream& out, Descriptor k )
  {
    static const char* color_names[] =
    {
      "blue",
      "brown",
      "cyan",
      "darkgreen",
      "darkorchid",
      "darksalmon",
      "darkviolet",
      "deeppink",
      "gold3",
      "green",
      "magenta",
      "navy",
      "red",
      "yellow",
      "palegreen",
      "gray65",
      "gray21",
      "bisque2",
      "greenyellow",
      "indianred4",
      "lightblue2",
      "mediumspringgreen",
      "orangered",
      "orange"
    };
    const int colors = sizeof ( color_names ) / sizeof ( color_names[0] );

    try
    {
      int number = graph[k].Name;

      int c = number % colors;

      if ( 0 <= c && c < colors )
      {
        out << " [ " <<
        "style=\"filled\"," << "fillcolor=\"" << color_names[c] << "\" " <<
        ", label=\"" <<
        "IdDb: "  << graph[k].GetIdDb()  <<
        "\\n "  << graph[k].GetT() << " " << graph[k].GetX() << " " << graph[k].GetY() << " " << graph[k].GetZ() <<
        "\\nNumb: " << number << " C:" << c << "Colors:" << colors <<
        "\\nProc:" << get ( get ( boost::vertex_owner, graph ), k ) <<
        "\"" <<
        " ]";
      }
      else
      {
        out << " [ label=\"(" << graph[k].GetIdDb() << " number " << number << " " << c << "Colors:" << colors << ")\" ]";
      }

    }
    catch ( ... )
    {
      std::cout << " Error writing node" << std::endl;
    }
  }

private:
  typedef typename boost::property_map< Graph , boost::vertex_index_t>::type ComponentMap_t;
  typedef typename boost::iterator_property_map<std::vector< int >::iterator, ComponentMap_t> ComponentMap;

  Graph &graph;
  /*  ComponentMap component;

    long CalculateConnectedComponent( )
    {
      std::vector< int > local_components_vec ( num_vertices ( graph ) );
      component = ComponentMap ( local_components_vec.begin(), get ( boost::vertex_index, graph ) );
      int num_components = boost::connected_components ( graph, component );
      synchronize ( component );
      return num_components;
    }*/
};






template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::write_graph ( const std::string &output_fileNameLineageTree )
{
  write_graphviz ( output_fileNameLineageTree , graph, node_writer<DistGraphType> ( graph ) );
}



template < typename EPtr, typename E >
double ParallelGraphCellType< EPtr, E >::GetAge ( Vertex s )
{
  std::list< Vertex > cellLife;
  this->GetCellLife ( s, cellLife );
  long ageInMinutes = abs ( graph[s].GetT() - ( graph[*cellLife.begin() ].GetT() ) ) * this->GetTimeStep();
  return ageInMinutes;
}




template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::GetCellLife ( Vertex s, std::list< Vertex >  &cellLife )
{
  Vertex cell = s;
  do
  {
    cellLife.push_front ( cell );

    std::list< Vertex > p_mother;
    this->GetPredecessor ( cell, p_mother );

    if ( p_mother.size() == 0 )
    {
      break;
    }
    else
    {
      cell = * ( p_mother.begin() );
      // the cell is a mother
      std::list< Vertex > p_daughters;
      GetSuccessor ( cell, p_daughters );

      if ( p_daughters.size() == 2 )
      {
        break;
      }
    }
  }
  while ( 1 );

  // the cell is a mother
  std::list< Vertex > p_daughters;
  cell = s;
  GetSuccessor ( cell, p_daughters );
  if ( p_daughters.size() == 1 )
  {
    do
    {
      cell = * ( p_daughters.begin() );
      cellLife.push_back ( cell );

      p_daughters.clear();
      GetSuccessor ( cell, p_daughters );
    }
    while ( p_daughters.size() == 1 );
  }
}




template < typename EPtr, typename E >
double ParallelGraphCellType< EPtr, E >::GetSpeed ( Vertex s )
{
  std::list< Vertex > p_mother ;
  GetPredecessor ( s, p_mother );
  if ( p_mother.size() == 0 )
  {
    return 0;
  }
  else
  {
    Vertex v = * ( p_mother.begin() );
    double speed = sqrt ( pow ( graph[s].GetX() - graph[v].GetX(), 2 ) + pow ( graph[s].GetY() - graph[v].GetY(), 2 ) + pow ( graph[s].GetZ() - graph[v].GetZ(), 2 ) );
    return speed;
  }
}

template < typename EPtr, typename E >
double ParallelGraphCellType< EPtr, E >::GetSpeed ( Vertex s, long t )
{
  std::list< Vertex > cellLife;
  GetCellLife ( s, cellLife );
  typename std::list< Vertex >::iterator it = cellLife.begin();

  std::vector< double > instantSpeed ( cellLife.size() );

  for ( int i = 0;i < instantSpeed.size();++i )
  {
    instantSpeed[i] = GetSpeed ( *it );
  }

  std::vector< double > averageSpeed ( cellLife.size() );
  for ( int i = 0;i < instantSpeed.size();++i )
  {
    double sum;
    long begin = ( i - t ) < 0 ? 0 : i - t;
    long end = ( i + t ) < cellLife.size() ? ( i + t ) : cellLife.size();
    long cant = end - begin;
    sum = std::accumulate ( &averageSpeed[begin], &averageSpeed[end], 0 );

    averageSpeed[i] = sum / cant;
  }

  typename std::list< Vertex >::iterator position = std::find ( cellLife.begin(), cellLife.end(), s );
  return ( averageSpeed[distance ( cellLife.begin(),position ) ] );
}


template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::SetRealTime ( long timeStepA, double minutesA, long timeStepB, double minutesB )
{
  time_m = double ( minutesB - minutesA ) / double ( timeStepB - timeStepA );
  time_b = minutesB - time_m * timeStepB;
}

template < typename EPtr, typename E >
double ParallelGraphCellType< EPtr, E >::GetTimeStep ( )
{
  return time_m;
}

template < typename EPtr, typename E >
double ParallelGraphCellType< EPtr, E >::GetRealTime ( long timeStep )
{
  return ( timeStep*time_m + time_b );
}


template < typename EPtr, typename E >
double ParallelGraphCellType< EPtr, E >::GetRealTimeLiving ( EPtr a_Cell )
{
  long timesteps = this->GetTimeStepsLiving ( a_Cell );
  return timesteps * time_m;
}







template < typename EPtr, typename E >
long ParallelGraphCellType< EPtr, E >::RenameCells ( boost::mpi::communicator &world )
{
  synchronize ( graph );

  typedef std::pair<Vertex, Vertex> VertexPair;

  std::set< VertexPair > edgeContainer;

  UpdateProperties ();

  //Collect the links of the mother cells
  BGL_FORALL_VERTICES_T ( u, graph, DistGraphType )
  {
    std::list< Vertex > list;
    GetSuccessor ( u, list ) ;
    if ( list.size() == 2 )
    {
      for ( typename std::list< Vertex >::iterator it = list.begin(); it != list.end();++it )
      {
        edgeContainer.insert ( std::make_pair ( u, *it ) );
      }
    }
  }



  //Delete links
  for ( typename std::set< VertexPair >::iterator it = edgeContainer.begin(); it != edgeContainer.end();++it )
  {
    remove_edge ( it->first, it->second, this->graph );
  }
  synchronize ( graph );


  long cellNumber = CalculateConnectedComponent ( propertyName );


  for ( typename std::set< VertexPair >::iterator it = edgeContainer.begin(); it != edgeContainer.end();++it )
  {
    AddRelationVertex ( it->first, it->second );
  }
  synchronize ( graph );
  synchronize ( propertyName );
  synchronize ( graph.process_group() );

  return cellNumber;
}




template < typename EPtr, typename E >
void ParallelGraphCellType< EPtr, E >::UpdateProperties ()
{
  BGL_FORALL_VERTICES_T ( u, graph, DistGraphType )
  {
    request ( propertyTime, u );
  }
  synchronize ( propertyTime );



  BGL_FORALL_VERTICES_T ( vertex, this->graph, DistGraphType )
  {
    put ( propertyNumberSuccessors, vertex, GetNumberSuccessor ( vertex ) );
    put ( propertyNumberPredecessors, vertex, GetNumberPredecessor ( vertex ) );
  }

  synchronize ( propertyNumberSuccessors );
  synchronize ( propertyNumberPredecessors );


  BGL_FORALL_VERTICES_T ( vertex, this->graph, DistGraphType )
  {

    std::list<Vertex> list;
    GetPredecessor ( vertex, list );

    if ( list.size() == 1 )
    {
      put ( propertyNewbourne, vertex, get ( propertyNumberSuccessors, * ( list.begin() ) ) == 2 );
    }
    else
    {
      put ( propertyNewbourne, vertex, 1 );
    }
  }
  synchronize ( propertyNewbourne );
}



#endif
