/* SimAnn - Simulated Annealing Method for cell tracking in 3D+time 
 * Written in 2015 by BioEmergences CNRS USR bioemergences@inaf.cnrs-gif.fr
 * Paul Bourgine paul.bourgine@polytechnique.edu
 * Alessandro Sarti alessandro.sarti@ehess.fr
 * Camilo Melani camilomelani@gmail.com
 * Rene Doursat rene.doursat@inaf.cnrs-gif.fr
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright and related and neighboring rights to this software to the public domain worldwide. This software is distributed without any warranty.
 * You should have received a copy of the CC BY-NC-SA 4.0 Dedication along with this software. If not, see <https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>.
 */

#include <boost/graph/use_mpi.hpp>
#include <boost/graph/distributed/adjacency_list.hpp>
#include <boost/graph/distributed/mpi_process_group.hpp>
#include <boost/graph/iteration_macros.hpp>
#include <boost/graph/distributed/graphviz.hpp>
#include <boost/graph/distributed/distributed_graph_utility.hpp>
#include <boost/archive/text_iarchive.hpp>
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/binary_iarchive.hpp>
#include <boost/archive/binary_oarchive.hpp>
#include <boost/config.hpp>
#include <boost/program_options/options_description.hpp>
#include <boost/program_options/parsers.hpp>
#include <boost/program_options/variables_map.hpp>
#include <boost/mpl/and.hpp>
#include <boost/serialization/shared_ptr.hpp>
//#include <boost/exception.hpp>
#include <boost/format.hpp>


#include <sstream>
#include <time.h>
#include <string>
#include <fstream>
#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <list>
#include <vector>
#include <algorithm>
#include <iterator>
 



#include "SetPointsWithLocator.h"
#include "MultiTimeSetPoint.h"
#include "CelluleType.h" //done
#include "EmbryoType.h" //done, but dependencies todo
#include "GraphCellType.h" //read through and document unclear implementation 
#include "ParallelGraphCellType.h" //read through and document unclear implementation
#include "definitions.h"
#include "BoxType4d.h"
#include "SimulatedAnnealing.h" //do next
#include "IOfunctions.h"
#include "LineageConstructor.h"
#include "Disk.h"
#include "ParallelDisk.h"
//#include "ImageProvider.h"



/*! 
 * @file mainParallelTracking.cxx
 * @brief contains the main function for the algorithm
 */




namespace po = boost::program_options;


/*  *
 *
 * @param argc
 * @param argv[]
 * @return
 */
int main ( int argc, char* argv[] )
{
  boost::mpi::environment env ( argc, argv );
  boost::mpi::communicator world;
  int rank = world.rank();
  long Numerator;
  long NumeratorIncrement;

  std::string input_type ( "" );
  std::vector < std::string > input_stage;

  long input_time_start_number = 0 ;
  long input_time_end_number = 0 ;

  std::string input_filenameImagePatern ( "" );
  std::string input_centersPatern ( "" );
  std::string input_fileNameLineageTree ( "" );
  std::string input_vectorFieldPattern ( "" );

  std::string output_fileNameLineageTree ( "" );

  double experiment_deltaT = 1;
  double experiment_minutesBegin = 0;
  double input_ImageSpacingX = 1;
  double input_ImageSpacingY = 1;
  double input_ImageSpacingZ = 1;
  int input_ImageSizeX = 1;
  int input_ImageSizeY = 1;
  int input_ImageSizeZ = 1;


  double tracking_simAnnIterations = 1;
  long tracking_repeat = 1;
  long tracking_NumberNearestCells = 5;
  long tracking_timeWindow = 1;
  long tracking_IterationByCell = 10;
  double tracking_ProbabilityStart = .5;
  double tracking_ProbabilityEnd = .1;

  long parallel_NumberBlockX = 1;
  long parallel_NumberBlockY = 1;
  long parallel_NumberBlockZ = 1;
  double parallel_BufferSize = 10;

  std::string tracking_siblingDirectory ( "" );

  std::vector <std::string> energy_name ( 100, "" );
  std::vector <double> energy_value ( 100, 0 );
  std::string configFile ( "" );



  ///////////////////
  ///////////////////
  ///////////////////
  ///////////////////
  ///////////////////Command line Parsing
  ///////////////////
  ///////////////////
  ///////////////////
  ///////////////////
  //BEGIN Parcing Parameters

  try
  {
// Declare a group of options that will be
// allowed only on command line
    po::options_description generic ( "Generic options" );
    generic.add_options()
    ( "version,v", "print version string" )
    ( "help", "produce help message" )
    ( "configFile", po::value< string > ( &configFile ), "path of the config file with more options" )
    ;

    // Declare a group of options that will be
    // allowed both on command line and in
    // config file
    po::options_description config ( "Configuration" );
    config.add_options()
    ( "input.type", po::value<std::string> ( &input_type ), "centers_file (VTK file).\nlineageTree: Read LineageTree" )
    ( "input.ProcessType", po::value< vector< std::string > > (), "process stage: Initialization, Minimization, AddForward, AddBackward, FixShortTimeCells" )
    ( "input.time_start_number", po::value< long > ( &input_time_start_number ), "Initial Time Step" )
    ( "input.time_end_number", po::value< long > ( &input_time_end_number ), "Final Time Step" )
    ( "input.file.filenameImagePatern", po::value< string > ( &input_filenameImagePatern )->default_value ( "" ), "filenameImagePatern" ) 
    ( "input.file.ImageSpacingX", po::value< double > ( &input_ImageSpacingX )->default_value ( 1 ), "ImageSpacingX" )
    ( "input.file.ImageSpacingY", po::value< double > ( &input_ImageSpacingY )->default_value ( 1 ), "ImageSpacingY" )
    ( "input.file.ImageSpacingZ", po::value< double > ( &input_ImageSpacingZ )->default_value ( 1 ), "ImageSpacingZ" )
    ( "input.file.ImageSizeX", po::value< int > ( &input_ImageSizeX ), "ImageSizeX pixels" )
    ( "input.file.ImageSizeY", po::value< int > ( &input_ImageSizeY ), "ImageSizeY pixels" )
    ( "input.file.ImageSizeZ", po::value< int > ( &input_ImageSizeZ ), "ImageSizeZ pixels" )
    ( "input.file.input_centersPatern", po::value< string > ( &input_centersPatern )->default_value ( "" ), "input_centersPatern" )
    ( "output.file.fileNameLineageTree", po::value< string > ( &output_fileNameLineageTree )->default_value ( "" ), "Filename + .LineageTree, Filename + .CellLife" )
    ( "input.file.fileNameLineageTree", po::value< string > ( &input_fileNameLineageTree )->default_value ( "" ), "input_fileNameLineageTree" )
    ( "input.file.fileNameVectorFieldPattern", po::value< string > ( &input_vectorFieldPattern )->default_value ( "" ), "input_fileNameVectorFieldPattern" )
    ( "parallel.NumberBlockX", po::value< long > ( &parallel_NumberBlockX )->default_value ( 1 ), "parallel.NumberBlockX" )
    ( "parallel.NumberBlockY", po::value< long > ( &parallel_NumberBlockY )->default_value ( 1 ), "parallel.NumberBlockY" )
    ( "parallel.NumberBlockZ", po::value< long > ( &parallel_NumberBlockZ )->default_value ( 1 ), "parallel.NumberBlockZ" )
    ( "parallel.BufferSize", po::value< double > ( &parallel_BufferSize )->default_value ( 10 ), "parallel.BufferSize" )
    ( "experiment.deltaT", po::value< double > ( &experiment_deltaT ), "Delta Time Step in minutes" )
    ( "experiment.minutesBegin", po::value< double > ( &experiment_minutesBegin ), "The instant in minutes after the fertilization of the start of the movie" )
    ( "tracking.simAnnIterations", po::value< double> ( &tracking_simAnnIterations )->default_value ( 0 ), "a value from 0 to 1, 0 means no simulated annealing, 1 means full minimization (slow)" )
    ( "tracking.repeat", po::value< long > ( &tracking_repeat )->default_value ( 1 ), "number of times the algorithm is executed" )
    ( "tracking.NumberNearestCells", po::value< long > ( &tracking_NumberNearestCells )->default_value ( 5 ), "Number of cell near the selected by the algorithm that are selected to minimize" )
    ( "tracking.timeWindow", po::value< long > ( &tracking_timeWindow )->default_value ( 1 ), "When minimize the algorithm select cells in this time window" )
    ( "tracking.IterationByCell", po::value< long > ( &tracking_IterationByCell )->default_value ( 10 ), "Number of iterations. This number is multiplied by NumberNearestCells" )
    ( "tracking.ProbabilityStart", po::value< double > ( &tracking_ProbabilityStart )->default_value ( 0.5 ), "Initial probacility of a change" )
    ( "tracking.ProbabilityEnd", po::value< double > ( &tracking_ProbabilityEnd )->default_value ( 0.1 ), "Final probacility of a change" )
//    ( "tracking.siblingDirectory", po::value< string > ( &tracking_siblingDirectory )->default_value ( "" ), "Directory patern with the siblins info" )
    ;

    po::options_description energyOptions ( "Energy" );
    for ( int i = 0;i < energy_name.size();++i )
    {
      std::stringstream name;
      name << "energy.Name" << i;

      std::stringstream value;
      value << "energy.Value" << i;

      energyOptions.add_options()
      ( name.str().c_str(), po::value< std::string > ( &energy_name[i] )->default_value ( "" ), name.str().c_str() )
      ( value.str().c_str(), po::value< double > ( &energy_value[i] )->default_value ( 0 ), value.str().c_str() );
    }

    po::variables_map vm;
    po::options_description cmdline_options;
    cmdline_options.add ( generic ).add ( config ).add ( energyOptions );
    po::options_description config_file_options;
    config_file_options.add ( config ).add ( energyOptions );
    store ( parse_command_line ( argc, argv, cmdline_options ), vm );
    po::notify ( vm );

    if ( vm.count ( "help" ) )
    {
      cout << cmdline_options << "\n";
      env.abort ( 1 );
      return 1;
    }

    if (vm.count("input.ProcessType"))
    {
      input_stage = vm["input.ProcessType"].as< vector<string> >();
    }


    if ( configFile != "" )
    {
      ifstream ifs ( configFile.c_str() );
      if ( !ifs )
      {
        cout << "Could no open " << configFile << "\n";
        env.abort ( 1 );
        return 1;
      }
      store ( parse_config_file ( ifs, config_file_options, false ), vm );
      po::notify ( vm );
    }
  }
  catch ( const std::exception& e )
  {
    cout << e.what() << "\n";
    env.abort ( 1 );
    return 1;
  }




  //END Parcing Parameters
  ParallelGraphCellType< CelluleTypePtr, CelluleType > myEmbryoDist;
  std::cout << rank << " " << GetActualTime() << std::endl;

 
  long MaxIdReaded;
  //BEGIN INPUT
  if ( input_type == "centers_file" )
  {
    if ( rank == 0 )
    {
      std::cout << rank << " " << GetActualTime() << " centers_file" << std::endl;
    }
//      long cant_center = ParallelIOReadCenters< CelluleTypePtr, CelluleType > ( input_time_start_number, input_time_end_number, &myEmbryoDist, input_centersPatern, input_ImageSpacingX, input_ImageSpacingY, input_ImageSpacingZ );

    long MaxIdReaded = ParallelIOReadCentersVTK< CelluleTypePtr, CelluleType > ( world, input_time_start_number,
                       input_time_end_number, &myEmbryoDist, input_centersPatern, rank );

    if ( MaxIdReaded == 0 )
    {
      std::cout << rank << " " << GetActualTime() << " Error Loading Centers" << std::endl;
      env.abort ( 1 );
      exit ( 1 );
    }


    if ( rank == 0 )
    {
      std::cout << rank << " " << GetActualTime() <<  " End reading center file: " << MaxIdReaded << " centers." << std::endl;
    }
  }
  else if ( input_type == "lineageTree" )
  {
    if ( rank == 0 )
    {
      std::cout << rank << " " << GetActualTime() << " Loading Lineage Tree" << std::endl;
    }

    MaxIdReaded = ParallelIOLoadLineageTree< CelluleType, CelluleTypePtr > ( world, myEmbryoDist, input_fileNameLineageTree, rank );
    if ( MaxIdReaded == 0 )
    {
      std::cout << rank << " " << GetActualTime() << " Error Loading Lineage Tree" << std::endl;
      env.abort ( 1 );
      exit ( 1 );
    }

    if ( rank == 0 )
    {
      std::cout << rank << " " << GetActualTime() << " Loading Terminated MaxId: " << MaxIdReaded << std::endl;
    }
  }
  else
  {
    if ( rank == 0 )
    {
      std::cout << rank << " " << GetActualTime() << " No input" << std::endl;
    }
    env.abort ( 1 );
    exit ( 1 );
  }

  Numerator = MaxIdReaded + world.rank() + 1;
  NumeratorIncrement = world.size();

  /*    if ( tracking_siblingDirectory != "" )
      {
        readSisterInfo<CelluleType, CelluleTypePtr> ( input_time_start_number , input_time_end_number , &myEmbryoDist, tracking_siblingDirectory );
      }*/

  myEmbryoDist.Synchronize();

  if ( input_vectorFieldPattern != "" )
  {
    ParallelIOReadVectorField<CelluleType, CelluleTypePtr> ( world, input_time_start_number , input_time_end_number , myEmbryoDist, input_vectorFieldPattern );
    myEmbryoDist.Synchronize();
  }


  myEmbryoDist.Synchronize();
  std::cout << rank << " " << GetActualTime() << " Num Vertices:" << myEmbryoDist.NumVertices() << std::endl;

  // END GETTING THE EMBRYO FROM FILE
  if ( rank == 0 )
  {
    std::cout << rank << " " << GetActualTime()  << " Selecting regions" << std::endl;
  }

  //END INPUT
  std::set<CelluleTypePtr> cellsToAdd;

  BoxType4d boxSelection;
//  for ( long m = input_time_start_number;m < input_time_end_number;m += input_time_feed )
  {
    //BEGIN SET 4d Region
    vector < long > munRegions ( 4 );
    munRegions[0] = parallel_NumberBlockX;
    munRegions[1] = parallel_NumberBlockY;
    munRegions[2] = parallel_NumberBlockZ;
    munRegions[3] = 1;

    vector <double> minValue ( 4, 0 );
    minValue[0] = 0;
    minValue[1] = 0;
    minValue[2] = 0;
    minValue[3] = input_time_start_number;


    vector <double> maxValue ( 4 );
    maxValue[0] = input_ImageSizeX * input_ImageSpacingX;
    maxValue[1] = input_ImageSizeY * input_ImageSpacingY;
    maxValue[2] = input_ImageSizeZ * input_ImageSpacingZ;
    maxValue[3] = input_time_end_number;

    boxSelection.SetMin ( minValue );
    boxSelection.SetMax ( maxValue );
    boxSelection.SetNumRegions ( munRegions );


    //////Box with buffer
    vector <double> buffer ( 4 );
    buffer [0] = parallel_BufferSize;
    buffer [1] = parallel_BufferSize;
    buffer [2] = parallel_BufferSize;
    buffer [3] = parallel_BufferSize;

    BoxType4d boxSelectionBuffer;
    boxSelectionBuffer.SetMin ( minValue );
    boxSelectionBuffer.SetMax ( maxValue );
    boxSelectionBuffer.SetNumRegions ( munRegions );
    boxSelectionBuffer.SetBuffer ( buffer );
    //END SET 4d Region


    //BEGIN Extract Sub Region and Morphofield
    GraphCellType< CelluleTypePtr > embryoLocal;
    if ( rank == 0 )
    {
      std::cout << rank << " " << GetActualTime() << " Tracking from " << boxSelectionBuffer.GetMin() [3] << " to " << boxSelectionBuffer.GetMax() [3] << std::endl;
      std::cout << rank << " " << GetActualTime() << " Connected componets" << std::endl;
    }
    long numComponents = myEmbryoDist.CalculateConnectedComponent ( myEmbryoDist.propertyComponent );

    if ( rank == 0 )
    {
      std::cout << rank << " " << GetActualTime()  << " Number Connected componets:" << numComponents << std::endl;
    }

    bool good = true;
    for ( long regionNumber = 0;regionNumber < world.size();++regionNumber )
    {
      myEmbryoDist.Synchronize();
      good &=  myEmbryoDist.GetMorphoFields ( world, boxSelectionBuffer, regionNumber, embryoLocal, regionNumber );
      myEmbryoDist.Synchronize();
    }

    myEmbryoDist.Synchronize();

    if ( !good )
    {
      std::cout << rank << " " << GetActualTime()  << " Embryo Size:" << embryoLocal.size() << std::endl;
      env.abort ( 1 );
      exit ( 1 );
    }
    //END Extract Sub Region and Morphofield

    embryoLocal.SetRealTime ( 0, experiment_minutesBegin, 1, experiment_minutesBegin + experiment_deltaT );
    myEmbryoDist.Synchronize();

    //BEGIN Initialization
    /****************************
    ***** Initialization
    ****************************/

    if ( find ( input_stage.begin(), input_stage.end(), "Initialization" ) != input_stage.end() )
    {
      std::cout << rank << " " << GetActualTime() << " Starting Initialization " << std::endl;

      LineageConstructor < CelluleTypePtr > myLineageConstructor;
      myLineageConstructor.SetEmbryo ( &embryoLocal );
      myLineageConstructor.SetDoubleLinks ( ( long ) boxSelectionBuffer.GetMin() [3], ( long ) boxSelectionBuffer.GetMax() [3] - 1 );
      myLineageConstructor.SetForwardLinks ( ( long ) boxSelectionBuffer.GetMin() [3], ( long ) boxSelectionBuffer.GetMax() [3] - 1 );
      myLineageConstructor.SetBackwardLinks ( ( long ) boxSelectionBuffer.GetMin() [3] + 1, ( long ) boxSelectionBuffer.GetMax() [3] );
      myLineageConstructor.SetMitoticLinks ( ( long ) boxSelectionBuffer.GetMin() [3], ( long ) boxSelectionBuffer.GetMax() [3] - 1 );

      myEmbryoDist.Synchronize();
      if ( rank == 0 )
      {
        std::cout << rank << " " << GetActualTime()  << " End Initialization " << std::endl;
      }
    }
    //END Initialization

    SimulatedAnnealing< CelluleType, CelluleTypePtr > myMinimizator;
    myMinimizator.SetEmbryo ( &embryoLocal );
    embryoLocal.RegisterNodeEnergy ( energy_name, energy_value );
    myMinimizator.SetNumberNearestCells ( tracking_NumberNearestCells );
    myMinimizator.SetIterationByCell ( tracking_IterationByCell );
    myMinimizator.SetProbabilityStart ( tracking_ProbabilityStart );
    myMinimizator.SetProbabilityEnd ( tracking_ProbabilityEnd );
    myMinimizator.SetNumerator ( &Numerator );
    myMinimizator.SetNumeratorIncrement ( NumeratorIncrement );

    //BEGIN SIMULATED ANNEALING
    if ( find ( input_stage.begin(), input_stage.end(), "Minimization" ) != input_stage.end() )
    {
      if ( tracking_simAnnIterations > 0.00000001 )
      {
        std::cout << rank << " " << GetActualTime()  << " Starting Simulated Annealing " << std::endl;
        myMinimizator.MinimizeStart ( ( long ) boxSelectionBuffer.GetMin() [3], ( long ) boxSelectionBuffer.GetMax() [3] , 3, tracking_repeat, tracking_simAnnIterations, std::cout );
        std::cout << rank << " " << GetActualTime()  << " End Simulated Annealing " << std::endl;
      }
    }

    //END SIMULATED ANNEALING
 
  
    //BEGIN ADDING CELLS
    if ( find ( input_stage.begin(), input_stage.end(), "AddForward" ) != input_stage.end() )
    {
      bool addCell = true;
      std::cout << rank << " " << GetActualTime()  << " Start Adding Cells " << std::endl;

      myMinimizator.MinimizeAddingCells ( ( long ) boxSelectionBuffer.GetMin() [3], ( long ) boxSelectionBuffer.GetMax() [3], tracking_timeWindow, addCell, 0, 1, std::cout );

      std::cout << rank << " " << GetActualTime()  << " End Adding Cells " << std::endl;
    }

    if ( find ( input_stage.begin(), input_stage.end(), "AddBackward" ) != input_stage.end() )
    {
      bool addCell = true;
      std::cout << rank << " " << GetActualTime()  << " Start Adding Cells " << std::endl;

      myMinimizator.MinimizeAddingCells ( ( long ) boxSelectionBuffer.GetMin() [3], ( long ) boxSelectionBuffer.GetMax() [3], tracking_timeWindow, addCell, 0, -1, std::cout );

      std::cout << rank << " " << GetActualTime()  << " End Adding Cells "  << std::endl;
    }

    if ( find ( input_stage.begin(), input_stage.end(), "FixShortTimeCells" ) != input_stage.end() )
    {
      bool addCell = true;
      std::cout << rank << " " << GetActualTime()  << " Start Adding Cells " << std::endl;

      myMinimizator.FixShortTimeCells ( tracking_timeWindow,  std::cout );

      std::cout << rank << " " << GetActualTime()  << " End Adding Cells " << std::endl;
    }

    cellsToAdd = myMinimizator.virtualCellsAdded;
    //END ADDING CELLS


    myEmbryoDist.Synchronize();
    //Add the virtual cells to the distributed Embryo.
    long cellsNum = 0;
    for ( std::set<CelluleTypePtr>::iterator it = cellsToAdd.begin();it != cellsToAdd.end();++it )
    {
      CelluleTypePtr cell = *it;
      if ( boxSelection.IsInside ( cell->GetX(), cell->GetY(), cell->GetZ(), cell->GetT(), rank ) )
      {
        myEmbryoDist.AddCell ( cell.get() );
        cellsNum++;
      }
    }
    myEmbryoDist.Synchronize();


    //BEGIN MERGING LINEAGE TREE
    embryoLocal.UpdateProperties();
    myEmbryoDist.MergeLinegeTree ( world, embryoLocal, boxSelection, rank );

    myEmbryoDist.Synchronize();
    if ( rank == 0 )
    {
      std::cout << rank << " " << GetActualTime() << " End Merging: " << std::endl;
    }
    //END MERGING LINEAGE TREE
  }
 

  //BEGIN SAVING 

  myEmbryoDist.Synchronize();
  if ( rank == 0 )
  {
    std::cout << rank << " " << GetActualTime() << " End LineageTree " << std::endl;
  }


  if ( output_fileNameLineageTree != "" )
  {
    //Renaming cells life
    myEmbryoDist.RenameCells ( world );

    //    myEmbryoDist.CalculateConnectedComponent ( myEmbryoDist.propertyComponent );
    //    myEmbryoDist.write_graph ( output_fileNameLineageTree + std::string ( ".dot" ) );
    myEmbryoDist.Synchronize();

    if ( rank == 0 )
    {
      std::ofstream fout ( ( output_fileNameLineageTree ).c_str() );
      myEmbryoDist.Save ( fout, boxSelection );

      std::string output_fileNameLineageTreeName = output_fileNameLineageTree;
      std::string::size_type idx;
      idx = output_fileNameLineageTreeName.rfind('.');
      if(idx != std::string::npos)
      {
        output_fileNameLineageTreeName = output_fileNameLineageTreeName.substr(0,idx);
      }

      output_fileNameLineageTreeName = output_fileNameLineageTreeName + ".CellLife";
      
      std::ofstream foutAnnotations ( output_fileNameLineageTreeName.c_str() );
      myEmbryoDist.SaveAnnotations ( world, foutAnnotations );
    }
    else
    {
      myEmbryoDist.Save ( cout, boxSelection );
      myEmbryoDist.SaveAnnotations ( world, cout );
    }
  }
  //END SAVING

  myEmbryoDist.Synchronize();
  myEmbryoDist.Synchronize();

  return 0;
}
